/*
 * linux/drivers/mtd/maps/pnx0105-flash.c
 *
 * Flash memory access for PNX0105 board
 * 
 * Author: Vitaly Wool <vwool@ru.mvista.com>
 *
 * Copyright (c) 2004 MontaVista Software, Inc. 
 * This file is licensed under the terms of the GNU General Public 
 * License version 2. This program is licensed "as is" without any 
 * warranty of any kind, whether express or implied.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/init.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/system.h>

#ifdef CONFIG_ARCH_PNX0105

#define PNX0105_FLASH_BASE	FLASH0_BASE
#define PNX0105_FLASH_SIZE 	0x400000
/* 4MB; somehow FLASH0_LENGTH is defined as 2MB; REVISIT */
#define PNX0105_FLASHCTRL	(IO_ADDRESS(MPMC_BASE) + 0x200)

static unsigned int pnx0105_flash_size;

static struct mtd_partition *parsed_parts;
static int dynamic_parts = 0;

static const char *part_probes[] = { "cmdlinepart", "RedBoot", "afs", NULL };

static struct mtd_partition pnx0105_parts[] = {
	{
		.name		= "BootLoader",
		.size		= 0x10000, /* 64kb */
		.offset		= 0,
#if 0 /* enable to force read-only */
		.mask_flags	= MTD_WRITEABLE,
#endif
	}, 
	{
		.name		= "Stable kernel",
		.size		= 0xf0000, /* 960kb */
		.offset		= 0x10000,
	}, 
	{
		.name		= "Development kernel",
		.size		= 0x100000, /* 1Mb */
		.offset		= 0x100000,
	}, 
	{
		.name		= "Filesystem",
		.size		= 0x1f0000, /* almost 2Mb */
		.offset		= 0x200000,
	}, 
	{
		.name		= "Boot Parameters", /* 64kb */
		.size		= 0x10000,
		.offset		= 0x3f0000,
	} 
};



/*
 * Initialise the flash access systems:
 */
static void armflash_flash_init(void)
{
	void *mpmc_flashctl = ioremap_nocache(MPMC_BASE + 0x200, 0x10);

	if (mpmc_flashctl) {
		__raw_writel(0x81, mpmc_flashctl);
		iounmap(mpmc_flashctl);
	} else
		printk(KERN_ERR "Could not ioremap MPMC registers\n");
}

/*
 * Shutdown the flash access systems:
 *  - Disable VPP
 *  - Assert WP
 */
static void armflash_flash_exit(void)
{
}

static void armflash_flash_wp(int on)
{
}

static void armflash_set_vpp(struct map_info *map, int on)
{
}
#endif


static __u8 armflash_read8(struct map_info *map, unsigned long ofs)
{
	return __raw_readb(ofs + map->map_priv_2);
}

static __u16 armflash_read16(struct map_info *map, unsigned long ofs)
{
	return __raw_readw(ofs + map->map_priv_2);
}

static __u32 armflash_read32(struct map_info *map, unsigned long ofs)
{
	return __raw_readl(ofs + map->map_priv_2);
}

static void armflash_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
	memcpy(to, (void *) (from + map->map_priv_2), len);
}

static void armflash_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	__raw_writeb(d, adr + map->map_priv_2);
}

static void armflash_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	__raw_writew(d, adr + map->map_priv_2);
}

static void armflash_write32(struct map_info *map, __u32 d, unsigned long adr)
{
	__raw_writel(d, adr + map->map_priv_2);
}

static void armflash_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
	memcpy((void *) (to + map->map_priv_2), from, len);
}

static struct map_info armflash_map =
{
	.name =		"PNX0105",
	.read8 =	armflash_read8,
	.read16 =	armflash_read16,
	.read32 =	armflash_read32,
	.copy_from =	armflash_copy_from,
	.write8 =	armflash_write8,
	.write16 =	armflash_write16,
	.write32 =	armflash_write32,
	.copy_to =	armflash_copy_to,
	.set_vpp =	armflash_set_vpp,
};

static int __init
pnx0105_static_partitions(struct mtd_partition **parts)
{
	int nb_parts = 0;

	*parts       = pnx0105_parts;
	nb_parts     = ARRAY_SIZE(pnx0105_parts);

	return nb_parts;
}

static struct mtd_info *mtd;

static int __init armflash_cfi_init(void *base, u_int size)
{
	int nb_parts = 0;
	int ret = 0;

	armflash_flash_init();
	armflash_flash_wp(1);

	/*
	 * look for CFI based flash parts fitted to this board
	 */
	armflash_map.size       = size;
	armflash_map.buswidth   = 2;
	armflash_map.map_priv_2 = (unsigned long) base;
	armflash_map.virt = (unsigned long) base;

	/*
	 * Also, the CFI layer automatically works out what size
	 * of chips we have, and does the necessary identification
	 * for us automatically.
	 */
	mtd = do_map_probe("cfi_probe", &armflash_map);
	if (!mtd)
		return -ENXIO;

	mtd->owner = THIS_MODULE;

        do {
		nb_parts = parse_mtd_partitions(mtd, part_probes, &parsed_parts, 0);
		if (nb_parts > 0)
		{
			dynamic_parts = 1;
			break;
		}
		nb_parts = pnx0105_static_partitions(&parsed_parts);
	} while (0);

	if (nb_parts < 0) {
		printk(KERN_NOTICE "PNX0105 flash: no partition info available,"
				   "registering whole flash at once\n");
		add_mtd_device(mtd);
	} else {
		printk(KERN_NOTICE "Using cmdline partition definition\n");
		ret = add_mtd_partitions(mtd, parsed_parts, nb_parts);
		if (ret)
		    printk(KERN_ERR "mtd partition registration failed: %d\n", ret);
	}

	/*
	 * If we got an error, free all resources.
	 */
	if (ret < 0) {
		del_mtd_partitions(mtd);
		map_destroy(mtd);
	}

	return ret;
}

static void armflash_cfi_exit(void)
{
	if (mtd) {
		del_mtd_partitions(mtd);
		map_destroy(mtd);
	}
	if (parsed_parts && dynamic_parts)
		kfree(parsed_parts);
}

static int __init armflash_init(void)
{
	int err = -EBUSY;
	void *base;

	pnx0105_flash_size = PNX0105_FLASH_SIZE;

	if (request_mem_region(PNX0105_FLASH_BASE, pnx0105_flash_size, "flash") == NULL)
		goto out;

	base = ioremap(PNX0105_FLASH_BASE, pnx0105_flash_size);
	err = -ENOMEM;
	if (base == NULL)
		goto release;

	err = armflash_cfi_init(base, pnx0105_flash_size);
	if (err) {
		iounmap(base);
release:
		release_mem_region(PNX0105_FLASH_BASE, pnx0105_flash_size);
	}
out:
	return err;
}

static void __exit armflash_exit(void)
{
	armflash_cfi_exit();
	iounmap((void *)armflash_map.map_priv_2);
	release_mem_region(PNX0105_FLASH_BASE, pnx0105_flash_size);
	armflash_flash_exit();
}

module_init(armflash_init);
module_exit(armflash_exit);

MODULE_AUTHOR("Vitaly Wool <vwool@ru.mvista.com>");
MODULE_DESCRIPTION("PNX0105 CFI map driver");
MODULE_LICENSE("GPL");
