/*
 * $Id: ocelot.c,v 1.11 2003/05/20 20:59:30 dwmw2 Exp $
 *
 * Flash on Momenco Ocelot
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#ifdef CONFIG_MIPS64
#define ADDR64 0xffffffff00000000
#else
#define ADDR64 0
#endif
#define CHOOSESIZE(addr) (addr | ADDR64)

#define OCELOT_C_CS0_ADDR       CHOOSESIZE(0xfc000000)

#define OCELOT_C_REG_BOARDREV		0x0
#define OCELOT_C_REG_FPGA_REV		0x1
#define OCELOT_C_REG_FPGA_TYPE		0x2
#define OCELOT_C_REG_RESET_STATUS	0x3
#define OCELOT_C_REG_BOARD_STATUS	0x4
#define OCELOT_C_REG_CPCI_ID		0x5
#define OCELOT_C_REG_SET		0x6
#define OCELOT_C_REG_CLR		0x7
#define OCELOT_C_REG_EEPROM_MODE	0x9
#define OCELOT_C_REG_INTMASK		0xa
#define OCELOT_C_REG_INTSTAT		0xb
#define OCELOT_C_REG_UART_INTMASK	0xc
#define OCELOT_C_REG_UART_INTSTAT	0xd
#define OCELOT_C_REG_INTSET		0xe
#define OCELOT_C_REG_INTCLR		0xf

#define OCELOT_FPGA_WRITE(x, y) writeb(x, OCELOT_C_CS0_ADDR + OCELOT_C_REG_##y)
#define OCELOT_FPGA_READ(x) readb(OCELOT_C_CS0_ADDR + OCELOT_C_REG_##x)

#define FLASH_WINDOW_ADDR 0x2fc00000 
#define FLASH_WINDOW_SIZE 0x00080000
#define FLASH_BUSWIDTH    1
#define NVRAM_WINDOW_ADDR 0x2c800000 
#define NVRAM_WINDOW_SIZE 0x00007FF0
#define NVRAM_BUSWIDTH    1

static unsigned int cacheflush = 0;

static struct mtd_info *flash_mtd;
static struct mtd_info *nvram_mtd;

static void ocelot_ram_write(struct mtd_info *mtd, loff_t to, const void *from, size_t len, size_t *retlen, const u_char *buf)
{
        struct map_info *map = (struct map_info *)mtd->priv;
	size_t done = 0;

	/* If we use memcpy, it does word-wide writes. Even though we told the 
	   GT64120A that it's an 8-bit wide region, word-wide writes don't work.
	   We end up just writing the first byte of the four to all four bytes.
	   So we have this loop instead */
	*retlen = len;
	while(len) {
		__raw_writeb(*(unsigned char *) from, map->virt + to);
		from++;
		to++;
		len--;
	}
}

static struct mtd_partition *parsed_parts;

struct map_info ocelot_flash_map = {
	.name = "Ocelot C boot flash",
	.size = FLASH_WINDOW_SIZE,
	.buswidth = FLASH_BUSWIDTH,
	.phys = FLASH_WINDOW_ADDR,
};

struct map_info ocelot_nvram_map = {
	.name = "Ocelot C NVRAM",
	.size = NVRAM_WINDOW_SIZE,
	.buswidth = NVRAM_BUSWIDTH,
	.phys = NVRAM_WINDOW_ADDR,
};

static const char *probes[] = { "RedBoot", NULL };

static int __init init_ocelot_maps(void)
{
	int nr_parts;
	unsigned int brd_status;

       	printk(KERN_INFO "Momenco Ocelot C MTD mappings: Flash 0x%lx at 0x%lx, NVRAM 0x%lx at 0x%lx\n", 
	       FLASH_WINDOW_SIZE, FLASH_WINDOW_ADDR, NVRAM_WINDOW_SIZE, NVRAM_WINDOW_ADDR);
	brd_status = OCELOT_FPGA_READ(BOARD_STATUS);
	
	/* Now ioremap the NVRAM space */
	ocelot_nvram_map.virt = (unsigned long)ioremap_nocache(NVRAM_WINDOW_ADDR, NVRAM_WINDOW_SIZE);
	if (!ocelot_nvram_map.virt) {
		printk(KERN_NOTICE "Failed to ioremap Ocelot C NVRAM space\n");
		return -EIO;
	}

	simple_map_init(&ocelot_nvram_map);

	/* And do the RAM probe on it to get an MTD device */
	nvram_mtd = do_map_probe("map_ram", &ocelot_nvram_map);
	if (!nvram_mtd) {
		printk("NVRAM probe failed\n");
		goto fail_1;
	}
	nvram_mtd->owner = THIS_MODULE;
	nvram_mtd->erasesize = 16;
	/* Override the write() method */
	nvram_mtd->write = ocelot_ram_write;

	/* Now map the flash space */
	ocelot_flash_map.virt = (unsigned long)ioremap_nocache(FLASH_WINDOW_ADDR, FLASH_WINDOW_SIZE);
	if (!ocelot_flash_map.virt) {
		printk(KERN_NOTICE "Failed to ioremap Ocelot C flash space\n");
		goto fail_2;
	}
	/* Now the cached version */
	ocelot_flash_map.cached = (unsigned long)ioremap(FLASH_WINDOW_ADDR, FLASH_WINDOW_SIZE);

	simple_map_init(&ocelot_flash_map);

	/* Only probe for flash if the write jumper is present */
	if (brd_status & 0x40) {
		flash_mtd = do_map_probe("jedec", &ocelot_flash_map);
	} else {
		printk(KERN_NOTICE "Ocelot C flash write jumper not present. Treating as ROM\n");
	}
	/* If that failed or the jumper's absent, pretend it's ROM */
	if (!flash_mtd) {
		flash_mtd = do_map_probe("map_rom", &ocelot_flash_map);
		/* If we're treating it as ROM, set the erase size */
		if (flash_mtd)
			flash_mtd->erasesize = 0x10000;
	}
	if (!flash_mtd)
		goto fail3;

	add_mtd_device(nvram_mtd);

	flash_mtd->owner = THIS_MODULE;
	nr_parts = parse_mtd_partitions(flash_mtd, probes, &parsed_parts, 0);

	if (nr_parts > 0)
		add_mtd_partitions(flash_mtd, parsed_parts, nr_parts);
	else
		add_mtd_device(flash_mtd);

	return 0;
	
 fail3:	
	iounmap((void *)ocelot_flash_map.virt);
	if (ocelot_flash_map.cached)
			iounmap((void *)ocelot_flash_map.cached);
 fail_2:
	map_destroy(nvram_mtd);
 fail_1:
	iounmap((void *)ocelot_nvram_map.virt);

	return -ENXIO;
}

static void __exit cleanup_ocelot_maps(void)
{
	del_mtd_device(nvram_mtd);
	map_destroy(nvram_mtd);
	iounmap((void *)ocelot_nvram_map.virt);

	if (parsed_parts)
		del_mtd_partitions(flash_mtd);
	else
		del_mtd_device(flash_mtd);
	map_destroy(flash_mtd);
	iounmap((void *)ocelot_flash_map.virt);
	if (ocelot_flash_map.cached)
		iounmap((void *)ocelot_flash_map.cached);
}

module_init(init_ocelot_maps);
module_exit(cleanup_ocelot_maps);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Red Hat, Inc. - David Woodhouse <dwmw2@cambridge.redhat.com>");
MODULE_DESCRIPTION("MTD map driver for Momenco Ocelot C board");
