/*
 * drivers/mtd/maps/mpc85xx_cds.c
 * 
 * Mapping for Motorola 85xx_cds user flash
 *
 * Author: source@mvista.com
 *
 * 2004 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <asm/io.h>

#define FLASH_BASE_ADDR          0xff000000

static struct mtd_info *flash;

static struct map_info mpc85xx_cds_map = {
	.name =		"MPC85xx CDS flash",
	.size =		0x1000000,
	.buswidth =	2,
	.phys = 	FLASH_BASE_ADDR,
};

static struct mtd_partition mpc85xx_cds_partitions[] = {
	{
		.name =   "flash bank 2",
		.offset = 0,
		.size =   0x800000,
		.mask_flags = MTD_WRITEABLE,  /* force read-only */
	},
	{
		.name =   "flash boot bank - free",
		.offset = 0x800000,
		.size =   0x780000,
	},
	{
		.name =   "flash boot bank - firmware",
		.offset = 0xf80000,
		.size =   0x80000,
		.mask_flags = MTD_WRITEABLE,  /* force read-only */
	}
};

int __init init_mpc85xx_cds(void)
{
	mpc85xx_cds_map.virt =
		(unsigned long)ioremap(FLASH_BASE_ADDR, mpc85xx_cds_map.size);

	if (!mpc85xx_cds_map.virt) {
		printk(KERN_NOTICE "Failed to ioremap flash\n");
		return -EIO;
	}

	simple_map_init(&mpc85xx_cds_map);

	flash = do_map_probe("cfi_probe", &mpc85xx_cds_map);
	if (flash) {
		flash->owner = THIS_MODULE;
		add_mtd_partitions(flash, mpc85xx_cds_partitions,
					ARRAY_SIZE(mpc85xx_cds_partitions));
	} else {
		printk(KERN_NOTICE "map probe failed for flash\n");
		return -ENXIO;
	}

	return 0;
}

static void __exit cleanup_mpc85xx_cds(void)
{
	if (flash) {
		del_mtd_partitions(flash);
		map_destroy(flash);
	}

	if (mpc85xx_cds_map.virt) {
		iounmap((void *)mpc85xx_cds_map.virt);
		mpc85xx_cds_map.virt = 0;
	}
}

module_init(init_mpc85xx_cds);
module_exit(cleanup_mpc85xx_cds);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("<source@mvista.com>");
MODULE_DESCRIPTION("MTD map and partitions for the Motorola 85xx CDS");
