/*
 *
 * drivers/mtd/maps/ixp2000.c
 *
 * Mapping for the Intel XScale IXP2000 based systems
 *
 * Copyright (C) 2002 Intel Corp.
 *
 * Original Author: Naeem M Afzal <naeem.m.afzal@intel.com>
 *
 * Maintainer: Deepak Saxena <dsaxena@mvista.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * Changelog:
 *  28-Mar-2003: Andrzej Mialkowski <andrzej.mialkowski@intel.com>
 *     - Added optional flash banking code (IXDP2401/IXDP2801 for now)
 *     - Fixed IXP2400 errata 44 implementation, added CPU version detection
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <asm/mach-types.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <asm/errno.h>

#define WINDOW_ADDR 	0xc4000000
#define BUSWIDTH 	1

static inline unsigned long flash_bank_setup(struct map_info *map,
					     unsigned long ofs)
{
	if (map->map_priv_2 != 0) {
		/* Call banking routine */
		return ((unsigned long (*)(unsigned long)) 
			map->map_priv_2) (ofs);
	}
	/* There is no banking on this system */
	return ofs;
}

#ifdef __ARMEB__
/*
 * Rev A0 and A1 of IXP2400 silicon have a broken addressing unit which 
 * causes the lower address bits to be XORed with 0x11 on 8 bit accesses 
 * and XORed with 0x10 on 16 bit accesses. See the spec update, erratta 44.
 */
static int errata44_workaround = 0;

static inline unsigned long address_fix8_write(unsigned long addr)
{
	if (errata44_workaround) {
		return (addr ^ 3);
	}
	return addr;
}
#else

#define address_fix8_write(x)	(x)
#endif

static __u8 ixp2000_read8(struct map_info *map, unsigned long ofs)
{
	return *(__u8 *) (map->map_priv_1 + flash_bank_setup(map, ofs));
}

/*
 * We can't use the standard memcpy due to the broken SlowPort
 * address translation on rev A0 and A1 silicon. Once B0 silicon
 * is available and I can prove that the errata is fixed, I'll take
 * this out.
 */
static void ixp2000_copy_from(struct map_info *map, void *to,
			      unsigned long from, ssize_t len)
{
	from = flash_bank_setup(map, from);
	while(len--) 
		*(__u8 *) to++ = *(__u8 *)(map->map_priv_1 + from++);
}

static void ixp2000_write8(struct map_info *map, __u8 d, unsigned long ofs)
{
	*(__u8 *) (address_fix8_write(map->map_priv_1 +
				      flash_bank_setup(map, ofs))) = d;
}

static void ixp2000_copy_to(struct map_info *map, unsigned long to,
			    const void *from, ssize_t len)
{
	to = flash_bank_setup(map, to);
	while(len--) {
		unsigned long tmp = address_fix8_write(map->map_priv_1 + to++);
		*(__u8 *)(tmp) = *(__u8 *)(from++);
	}
}

static struct map_info ixp2000_map = {
	.name		= "IXP2000 flash",
	.buswidth	= BUSWIDTH,
	.read8		= ixp2000_read8,
	.copy_from	= ixp2000_copy_from,
	.write8		= ixp2000_write8,
	.copy_to	= ixp2000_copy_to
};

#ifdef CONFIG_ARCH_IXDP2400
static struct mtd_partition ixp2000_partitions[4] = {
	{
		.name		= "RedBoot",
		.size		= 0x00040000,
		.offset		= 0,
		.mask_flags	= MTD_WRITEABLE,  /* force read-only */
	},{
		.name		= "System Log",
		.size		= 0x00020000,
		.offset		= 0x00fa0000,
	},{
		.name		= "linux",
		.size		= 0x100000,
		.offset		= 0x00100000,
	},{
		.name		= "ramdisk",
		.size		= 0x400000,
		.offset		= 0x00200000,
	}
};
#elif defined(CONFIG_ARCH_IXDP2401) || defined(CONFIG_ARCH_IXDP2801)
/*
 * Both platforms are not using static partitions, protect only RedBoot area
 */
static struct mtd_partition ixp2000_partitions[2] = {
	{
		.name		= "RedBoot",
		.size		= 0x00200000,
		.offset		= 0,
		.mask_flags	= MTD_WRITEABLE /* force read-only */
	}, {
		.name		= "free",
		.offset		= 0x00200000,
	}
};
#elif defined(CONFIG_ARCH_IXDP2800)
static struct mtd_partition ixp2000_partitions[] = {
	{
		.name		= "vBOOT",
		.size		= 0x00100000,
		.offset		= 0,
		.mask_flags	= MTD_WRITEABLE  /* force read-only */
	},{
		.name		= "vWARE FFS",
		.size		= 0x00700000,
		.offset		= 0x00100000,
		.mask_flags	= MTD_WRITEABLE  /* force read-only */
	},{
		.name		= "vWARE free",
		.size		= 0x00400000,
		.offset		= 0x00800000,
		.mask_flags	= MTD_WRITEABLE  /* force read-only */
	},{
		.name		= "free",
		.size		= 0x00400000,
		.offset		= 0x00c00000,
	}
};
#elif defined (CONFIG_ARCH_IXMB995E)
static struct mtd_partition ixp2000_partitions[2] = {
	{
		.name		= "RedBoot",
		.size		= 0x00040000,
		.offset		= 0,
		.mask_flags	= MTD_WRITEABLE /* force read-only */
	}, {
		.name		= "free",
		.offset		= 0x00040000,
	}
};
#else 
#error No Architecture defined for MTD partition
#endif

#define NB_OF(x)  (sizeof(x)/sizeof(x[0]))

static struct mtd_info *mymtd;
static struct mtd_partition *parsed_parts;

static const char *probes[] = { "RedBoot", "cmdlinepart", NULL };

static int __init init_ixp2000(void)
{
	struct mtd_partition *parts;
	int nb_parts = 0;
	int parsed_nr_parts = 0;
	char *part_type = "Static";

	/*
	 * Setup read mode for FLASH
	 */
	*IXP2000_SLOWPORT_FRM = 1;

#if defined(__ARMEB__)
	/*
	 * Enable errata 44 workaround for NPUs with broken slowport
	 */

	errata44_workaround = npu_has_broken_slowport();
	printk(KERN_NOTICE "IXP2000 flash: Errata 44 workaround %s\n",
	       errata44_workaround ? "enabled" : "disabled");
#endif
	/*
	 * map_priv_2 stores banking routine address if banking is used
	 * map_priv_1 stores virtual adress of flash window
	 * size stores size of flash address space size (including banking)
	 */

#if defined (CONFIG_ARCH_IXDP2401)
	if (machine_is_ixdp2401()) {
		/* IXDP2401 uses set of set of 32MB banks */
		ixp2000_map.map_priv_1 =
		    (unsigned long) ioremap(WINDOW_ADDR,
					    IXDP2401_FLASH_WINDOW_SIZE);
		ixp2000_map.map_priv_2 =
		    (unsigned long) ixdp2401_set_flash_bank;
		ixp2000_map.size =
		    IXDP2401_FLASH_WINDOW_SIZE *
		    ixdp2401_get_flash_banks();
	}
#endif

#if defined (CONFIG_ARCH_IXDP2801)
	if (machine_is_ixdp2801()) {
		/* IXDP2801 uses set of set of 32MB banks */
		ixp2000_map.map_priv_1 =
		    (unsigned long) ioremap(WINDOW_ADDR,
					    IXDP2801_FLASH_WINDOW_SIZE);
		ixp2000_map.map_priv_2 =
		    (unsigned long) ixdp2801_set_flash_bank;
		ixp2000_map.size =
		    IXDP2801_FLASH_WINDOW_SIZE *
		    ixdp2801_get_flash_banks();
	}
#endif

#if defined (CONFIG_ARCH_IXDP2400) || defined (CONFIG_ARCH_IXDP2800)
	if (machine_is_ixdp2400() || machine_is_ixdp2800()) {
		/* IXDP2400 uses only 16MB from CS0 */
		ixp2000_map.map_priv_1 =
		    (unsigned long) ioremap(WINDOW_ADDR, 0x1000000);
		ixp2000_map.map_priv_2 = 0;
		ixp2000_map.size = 0x1000000;
	}
#endif

	if (!ixp2000_map.map_priv_1) {
		printk("Failed to ioremap\n");
		return -EIO;
	}

	mymtd = do_map_probe("cfi_probe", &ixp2000_map);
	if (!mymtd) {
		iounmap((void *)ixp2000_map.map_priv_1);
		return -ENXIO;
	}

	mymtd->owner = THIS_MODULE;
	mymtd->priv = &ixp2000_map;

	parsed_nr_parts = 
		parse_mtd_partitions(mymtd, probes, &parsed_parts, WINDOW_ADDR);

	if (parsed_nr_parts > 0) {
		parts = parsed_parts;
		nb_parts = parsed_nr_parts;
	} else {
		if(machine_is_ixdp2401() || machine_is_ixdp2801()) {
			/*
			 * Update free space size according to 
			 * detected chips (??)
			 */
			ixp2000_partitions[1].size =
				mymtd->size - ixp2000_partitions[1].offset;
		}
		parts = ixp2000_partitions;
		nb_parts = NB_OF(ixp2000_partitions);
	}
	printk(KERN_NOTICE "Using %s partition definition\n", part_type);
	add_mtd_partitions(mymtd, parts, nb_parts);
	return 0;
}

static void __exit cleanup_ixp2000(void)
{
	if (mymtd) {
		del_mtd_partitions(mymtd);
		map_destroy(mymtd);
		if (parsed_parts)
			kfree(parsed_parts);
	}
	if (ixp2000_map.map_priv_1) {
		iounmap((void *)ixp2000_map.map_priv_1);
	}
}

module_init(init_ixp2000);
module_exit(cleanup_ixp2000);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Deepak Saxena");
