/*
 * drivers/mtd/maps/intrinsity.c
 *
 * Intrinsity flash mapping.
 *
 * Author: MontaVista Software, Inc. <jsun@mvista.com> or <source@mvista.com>
 *
 * 2003-2004 (c) MontaVista, Software, Inc.  This file is licensed under the terms
 * of the GNU General Public License version 2.  This program is licensed
 * "as is" without any warranty of any kind, whether express or implied.
 */
 
/*
 * This is simple PhysMap mapping, but with workaround for
 * Intrinsity's RedBoot partitions.
 * Intrinsity's RedBoot partition table is stored in little endian mode
 * but the FastMIPS CPU operates in big endian mode, so we need to
 * do conversion between LE and BE when reading the partition table.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/config.h>
#include <linux/mtd/partitions.h>
#include <asm/intrinsity/fastmath.h>

#define FM_FLASH_ADDR PHYSADDR(FM_FLASH_BASE)
#define FM_FLASH_SIZE 0x800000
#define FM_BUSWIDTH 1

static struct mtd_info *mymtd;
struct map_info intrinsity_map = {
	.name = "intrinsity_flash",
	.phys = FM_FLASH_ADDR,
	.size = FM_FLASH_SIZE,
	.buswidth = FM_BUSWIDTH
};

#ifdef CONFIG_MTD_PARTITIONS
static struct mtd_partition intrinsity_mtd_parts[] __initdata = {
	{
		.name = "User FS",
		.offset = 0x0,
		.size = 0x400000
	},
	{
		.name = "RedBoot",
		.offset = MTDPART_OFS_APPEND,
		.size = MTDPART_SIZ_FULL,
		.mask_flags=MTD_WRITEABLE
	}
};
#define intrinsity_part_num  (sizeof(intrinsity_mtd_parts)/sizeof(struct mtd_partition))

static struct mtd_partition *mtd_parts;
static int                   mtd_parts_nb;

char *part_probes[] __initdata = {"RedBoot", "cmdlinepart", NULL};

#ifdef CONFIG_MTD_REDBOOT_PARTS

#include <asm/byteorder.h>

struct fis_image_desc {
    unsigned char name[16];      // Null terminated name
    unsigned long flash_base;    // Address within FLASH of image
    unsigned long mem_base;      // Address in memory where it executes
    unsigned long size;          // Length of image
    unsigned long entry_point;   // Execution entry point
    unsigned long data_length;   // Length of actual data
    unsigned char _pad[256-(16+7*sizeof(unsigned long))];
    unsigned long desc_cksum;    // Checksum over image descriptor
    unsigned long file_cksum;    // Checksum over image data
};

/*
 * *** HACK ***
 *
 * This is required to read the RedBoot table properly in BE mode.
 * We know that the first copy_from will come from the 
 * RedBoot parser, so we hack this in here.
 */
static void
intrinsity_copy_from(struct map_info *map, void *to,
		 			unsigned long from, ssize_t len)
{
	static int first = 1;
	memcpy_fromio(to, map->virt + from, len);
	
	if (unlikely(first)) {
		first = 0;
		struct fis_image_desc *desc = (struct fis_image_desc *)to;
		int i;
		for(i = 0; i < (len/sizeof(struct fis_image_desc)); i++) {
			desc[i].flash_base = __le32_to_cpu(desc[i].flash_base);
			desc[i].mem_base   = __le32_to_cpu(desc[i].mem_base);
			desc[i].size       = __le32_to_cpu(desc[i].size);
		}
	}
}
#endif /*CONFIG_MTD_REDBOOT_PARTS*/

#endif /* CONFIG_MTD_PARTITIONS */

int __init init_intrinsity_flash(void)
{
	printk(KERN_NOTICE "Intrinsity flash device: %lx at %lx\n", intrinsity_map.size, intrinsity_map.phys);
	intrinsity_map.virt = (unsigned long)ioremap(intrinsity_map.phys, intrinsity_map.size);

	if (!intrinsity_map.virt) {
		printk("Failed to ioremap\n");
		return -EIO;
	}

	simple_map_init(&intrinsity_map);
#ifdef CONFIG_MTD_REDBOOT_PARTS
	intrinsity_map.copy_from = intrinsity_copy_from;
#endif

	mymtd = do_map_probe("cfi_probe", &intrinsity_map);
	
	if (mymtd) {
		mymtd->owner = THIS_MODULE;

#ifdef CONFIG_MTD_PARTITIONS
		mtd_parts_nb = parse_mtd_partitions(mymtd, part_probes, 
						    &mtd_parts, 0);

		if (mtd_parts_nb > 0)
		{
#ifdef CONFIG_MTD_REDBOOT_PARTS
			/* Do not allow writing in RedBoot's partitions */
			int i;
			for (i = 0; i < mtd_parts_nb; i++) {
				if (!memcmp(mtd_parts[i].name, "RedBoot", 7) ||
					!memcmp(mtd_parts[i].name, "FIS directory", 13))
					mtd_parts[i].mask_flags |= MTD_WRITEABLE; 
			}
#endif /*CONFIG_MTD_REDBOOT_PARTS*/
			add_mtd_partitions (mymtd, mtd_parts, mtd_parts_nb);
			return 0;
		}

		if (intrinsity_part_num != 0) 
		{
			printk(KERN_NOTICE 
			       "Using board's partition definition\n");
			add_mtd_partitions (mymtd, intrinsity_mtd_parts, intrinsity_part_num);
			return 0;
		}

#endif /*CONFIG_MTD_PARTITIONS*/
		add_mtd_device(mymtd);

		return 0;
	}

	iounmap((void *)intrinsity_map.virt);
	return -ENXIO;
}

static void __exit cleanup_intrinsity_flash(void)
{
#ifdef CONFIG_MTD_PARTITIONS
	if (mtd_parts_nb) {
		del_mtd_partitions(mymtd);
		kfree(mtd_parts);
	} else if (intrinsity_part_num) {
		del_mtd_partitions(mymtd);
	} else {
		del_mtd_device(mymtd);
	}
#else
	del_mtd_device(mymtd);
#endif
	map_destroy(mymtd);

	iounmap((void *)intrinsity_map.virt);
	intrinsity_map.virt = 0;
}

module_init(init_intrinsity_flash);
module_exit(cleanup_intrinsity_flash);


MODULE_LICENSE("GPL");
MODULE_AUTHOR("David Woodhouse <dwmw2@infradead.org>");
MODULE_DESCRIPTION("Generic configurable MTD map driver");
