/*
 * drivers/mtd/maps/argan.c
 * 
 * Flash map driver for IBM Argan board
 *
 * Chose not to enable 8 bit flash as it contains the firware and board
 * info.  Thus only the 32bit flash is supported.
 * 
 * Author: <source@mvista.com>
 *
 * <2004> (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <platforms/argan.h>

static struct map_info argan32_map = {
   	.name 		= "IBM 750fx Eval 32-bit flash",
   	.size 		= 0x2000000,
   	.buswidth 	= 4,
	.phys 		= ARGAN_32BIT_BASE,
};

static struct mtd_partition argan32_partitions[] = {
	{
		.name 	= "32-bit flash Filesystem",
		.offset	= 0,
		.size	= 0x2000000,
	}
};

static struct mtd_info *flash32; 

int __init init_argan(void)
{
	/* 32-bit FLASH */

   	argan32_map.virt = 
		(unsigned long)ioremap(argan32_map.phys, argan32_map.size);

   	if (!argan32_map.virt) {
      		printk(KERN_NOTICE "Failed to ioremap 32-bit flash\n");
		return -EIO;
   	}

	simple_map_init(&argan32_map);

   	flash32 = do_map_probe("cfi_probe", &argan32_map);
   	if (flash32) {
   		flash32->owner = THIS_MODULE;
   		add_mtd_partitions(flash32, argan32_partitions, 
					ARRAY_SIZE(argan32_partitions));
   	} else {
      		printk(KERN_NOTICE "map probe failed for 32-bit flash\n");
		return -ENXIO;
	}

   	return 0;
}

static void __exit 
cleanup_argan(void)
{
   	if (flash32) {
      		del_mtd_partitions(flash32);
		map_destroy(flash32);
   	}

   	if (argan32_map.virt) {
      		iounmap((void *)argan32_map.virt);
	  	argan32_map.virt = 0;
   	}
}

module_init(init_argan);
module_exit(cleanup_argan);

MODULE_DESCRIPTION("MTD map and partitions for IBM 750fx Eval");
MODULE_AUTHOR("<mvista.com>");
MODULE_LICENSE("GPL");
