/*
 * BRIEF MODULE DESCRIPTION
 * Low-level MMC functions for the Motorola MX1 MMC/SD controller
 * SD support is not implemented
 * Based on: omap_mmc.c
 *
 * Author: MontaVista Software, Inc. <source@mvista.com>
 * 2003 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 */
#include <linux/module.h>
#include <linux/version.h>

#include <linux/init.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/delay.h>

#include <asm/irq.h>
#include <asm/unaligned.h>
#include <asm/io.h>
#include <asm/arch/hardware.h>
#include <linux/mmc/mmc_ll.h>
#include <asm/arch/platform.h>
#include <asm/arch/mx1ads-gpio.h>

#include <linux/pm.h>
#include "mx1_mmcsd_def.h"

void
Start_Stop_Clk(u32 Clk_en);
static void
mx1_mmc_fix_sd_detect(void);

enum mx1_request_type {
	RT_NO_RESPONSE,
	RT_RESPONSE_ONLY,
	RT_READ,
	RT_WRITE
};

struct mx1_mmc_data {
	u32 clock;		/* Current clock frequency */
	struct mmc_request *request;	/* Current request */
	enum mx1_request_type type;
};

static void
mx1_mmc_handle_int(struct mx1_mmc_data *sd);

static struct mx1_mmc_data g_mx1_mmc_data;
static u16 card_state;
static int rx_counter = 0;
static int tx_counter = 0;

#define MX1_MMC_MASTER_CLOCK 16000000

static __inline__ void
mmc_delay(void)
{
	udelay(1);
}

/**************************************************************************
 *   Clock routines
 **************************************************************************/

u32
_MMCSD_SoftReset(void)
{
	/* System Reset */
	Start_Stop_Clk(0x8);

	/* System Reset, MMC core enable & Stop CLk */

	Start_Stop_Clk(0xD);

	Start_Stop_Clk(0x5);
	Start_Stop_Clk(0x5);
	Start_Stop_Clk(0x5);
	Start_Stop_Clk(0x5);
	Start_Stop_Clk(0x5);
	Start_Stop_Clk(0x5);
	Start_Stop_Clk(0x5);
	Start_Stop_Clk(0x5);

	/* choose big endian FIFO mode */
	_reg_MMC_STR_STP_CLK |= 0x00000020;

	/* Set MMC Response Time-Out Register */
	_reg_MMC_RES_TO = 0xFF;

	/* Set Block length register */
	_reg_MMC_BLK_LEN = 512;

	/* Set MMC Number of Blocks Register */
	_reg_MMC_NOB = 1;

	return 0;

}

static int
mx1_mmc_stop_clock(void)
{
	/* Clear bit 1 of STR_STP_CLK to disable clock */

	do {
		_reg_MMC_STR_STP_CLK &= ~MMCSD_CLOCK_MASK;
		_reg_MMC_STR_STP_CLK |= 0x05;
	}
	while (_reg_MMC_STATUS & 0x100);

	return 0;

}

u32
mx1_clock_run(void)
{
	return (_reg_MMC_STATUS) & 0x100;
}

static int
mx1_mmc_start_clock(void)
{

	u8 count = 0;
	do {
		if (count++ == 0) {
			/* do this every 256 loops */
			_reg_MMC_STR_STP_CLK &= ~MMCSD_CLOCK_MASK;
			_reg_MMC_STR_STP_CLK |= 0x06;
		}
	}
	while (!mx1_clock_run());

	return 0;

}

void
Start_Stop_Clk(u32 Clk_en)
{
	/* when the clk is STARTED check whether the MMC is enable and reset is not set */
	if (((Clk_en & 0x2) != 0) && ((Clk_en & 0x8) == 0)
	    && ((Clk_en & 0x4) != 0)) {
		mx1_mmc_start_clock();
	}
	/* when the clk is STOPPED, check whether the MMC is enable and reset is not set */
	else if (((Clk_en & 0x1) != 0) && ((Clk_en & 0x8) == 0)
		 && ((Clk_en & 0x4) != 0)) {
		mx1_mmc_stop_clock();
	}
	/* reset or enable the system */
	else {
		_reg_MMC_STR_STP_CLK &= ~MMCSD_CLOCK_MASK;
		_reg_MMC_STR_STP_CLK |= Clk_en;
	}
}

static int
mx1_mmc_set_clock(u32 rate)
{
	u8 clk_rate, prescaler;

	DEBUG(3, ": Set rate: %d \n", rate);

	u32 master = MX1_MMC_MASTER_CLOCK;	/* Default master clock */
	u8 divisor;

	if (rate == 0) {
		/* Disable MMC_SD_CLK */
		mx1_mmc_stop_clock();
		return MMC_NO_ERROR;
	}

	divisor = master / rate;

	for (clk_rate = 0; clk_rate <= 6; clk_rate++) {
		if ((prescaler = divisor / (1 << clk_rate)) <= 5)
			break;	/* prescaler=divisor/2**clk_rate */
	}

	mx1_mmc_stop_clock();

	prescaler = prescaler & 0x07;
	clk_rate = clk_rate & 0x07;

	if (rate == MMC_CLOCK_SLOW)
		_reg_MMC_CLK_RATE = 0x3f;
	else
		_reg_MMC_CLK_RATE = (prescaler << 3) | clk_rate;

	mmc_delay();

	/* Time out values */

	mx1_mmc_start_clock();

	_reg_MMC_RES_TO = 0xFF;
	_reg_MMC_READ_TO = 0x2DB4;

	g_mx1_mmc_data.clock = rate;

	return MMC_NO_ERROR;
}

static void
mx1_mmc_set_transfer(u16 block_len, u16 nob)
{
	_reg_MMC_BLK_LEN = block_len;
	_reg_MMC_NOB = nob;
}

static void
mx1_mmc_rx(void)
{
	int i = 0;
	u16 buf;

	{
		u8 *tmp = g_mx1_mmc_data.request->buffer + rx_counter;
		u32 status = _reg_MMC_STATUS & 0xFFFF;
		while (1) {

			buf = (u16) _reg_MMC_BUFFER_ACCESS;
			*tmp++ = (u8) (buf >> 8);
			*tmp++ = (buf & 0xFF);
			if (_reg_MMC_STATUS & MX1STAT_CRC_READ_ERR) {
				DEBUG(3, ": CRC_ERR %04x \n", _reg_MMC_STATUS);
			}
			if (_reg_MMC_STATUS & MX1STAT_TIME_OUT_READ) {
				DEBUG(3, ": TIMEOUT %04x \n", _reg_MMC_STATUS);
			}

			status = _reg_MMC_STATUS & 0xFFFF;
			i++;
			if (i == 8)
				break;
		}
	}
	rx_counter += i * 2;

	if (rx_counter >= g_mx1_mmc_data.request->block_len) {

		DEBUG(3, ": RX Transfer finished\n");

		g_mx1_mmc_data.request->nob--;

	}

}

static void
mx1_mmc_tx(void)
{

	int i = 0;
	u16 buf;
	u8 c1, c2;
	u8 *tmp = g_mx1_mmc_data.request->buffer + tx_counter;

	if (tx_counter <
	    g_mx1_mmc_data.request->block_len * g_mx1_mmc_data.request->nob) {
		while (1) {
			c1 = *tmp++;
			c2 = *tmp++;
			buf = (c1 << 8) | c2;
			(u16) _reg_MMC_BUFFER_ACCESS = buf;
			i++;
			if (i == 8)
				break;
		}
		tx_counter += i * 2;
	}
	if (tx_counter >= g_mx1_mmc_data.request->block_len) {
		_reg_MMC_INT_MASK = MX1_WRITE_OP_DONE_MASK;
	}
}

static int
mx1_mmc_exec_command(struct mmc_request *request)
{
	u16 dat_control = 0;
	u16 int_mask = 0;

	_reg_MMC_INT_MASK = MX1_AUTO_CARD_DETECT_MASK & 0x7f;

	int_mask = 0x7b;

	mx1_mmc_stop_clock();	/* Clear status bits */
	switch (request->cmd) {
	case MMC_READ_SINGLE_BLOCK:
		mx1_mmc_set_transfer(request->block_len, request->nob);
		break;
	case MMC_WRITE_BLOCK:
		mx1_mmc_set_transfer(request->block_len, request->nob);
		break;
	}

	dat_control = 0;

	/* Set response type */

	switch (request->rtype) {
	case RESPONSE_NONE:
		break;
	case RESPONSE_R1:
	case RESPONSE_R1B:
		dat_control |= MMCSDB_R1;
		break;
	case RESPONSE_R2_CID:
	case RESPONSE_R2_CSD:
		dat_control |= MMCSDB_R2;
		break;
	case RESPONSE_R3:
		dat_control |= MMCSDB_R3;
		break;
	case RESPONSE_R4:
		dat_control |= MMCSDB_R4;
		break;
	case RESPONSE_R5:
		dat_control |= MMCSDB_R5;
		break;
	}

	/* Set command type */
	switch (request->cmd) {

	case MMC_GO_IDLE_STATE:
		dat_control |= MMCSDB_INIT;
		break;
	case MMC_READ_SINGLE_BLOCK:
	case MMC_READ_MULTIPLE_BLOCK:
		dat_control |= MMCSDB_DATEN;
		break;

	case MMC_WRITE_BLOCK:
	case MMC_WRITE_MULTIPLE_BLOCK:
		dat_control |= MMCSDB_DATEN | MMCSDB_WRRD;
		break;
	case MMC_SELECT_CARD:
		if (request->arg != 0)
			dat_control |= MMCSDB_BSY | MMCSDB_R1;
		else
			dat_control |= MMCSDB_BSY;
		break;
	case MMC_LOCK_UNLOCK:
		dat_control |= MMCSDB_DATEN | MMCSDB_WRRD;
		break;
	}

	/* Send command */

	DEBUG(3, ": Send cmd : %d  Arg: %x\n", request->cmd, request->arg);
	_reg_MMC_CMD = request->cmd;

	/* Set argument */
	_reg_MMC_ARGH = (request->arg) >> 16;
	_reg_MMC_ARGL = (request->arg) & 0xffff;
	_reg_MMC_CMD_DAT_CONT = dat_control;

	if (request->cmd == MMC_READ_SINGLE_BLOCK
	    || request->cmd == MMC_READ_MULTIPLE_BLOCK) {
		g_mx1_mmc_data.type = RT_READ;
		rx_counter = 0;
	} else if (request->cmd == MMC_WRITE_BLOCK
		   || request->cmd == MMC_WRITE_MULTIPLE_BLOCK) {
		g_mx1_mmc_data.type = RT_WRITE;
		tx_counter = 0;
	} else if (request->rtype == RESPONSE_NONE) {
		g_mx1_mmc_data.type = RT_NO_RESPONSE;
	} else {
		g_mx1_mmc_data.type = RT_RESPONSE_ONLY;
	}

	_reg_MMC_INT_MASK = int_mask;
	mx1_mmc_start_clock();

	return MMC_NO_ERROR;
}

static void
mx1_mmc_send_command(struct mmc_request *request)
{
	int retval;

	g_mx1_mmc_data.request = request;
	request->result = MMC_NO_RESPONSE;	/* Flag to indicate don't have a result yet */

	if (request->cmd == MMC_CIM_RESET) {

		/* Reset MX1 MMC hardware */
		_MMCSD_SoftReset();
		retval = mx1_mmc_set_clock(MMC_CLOCK_SLOW);
		if (retval) {
			/* If any error occured -> exit with error code */

			request->result = retval;
			mmc_cmd_complete(request);
			return;
		}
		request->result = MMC_NO_ERROR;
		request->rtype = RESPONSE_NONE;
		mmc_cmd_complete(request);
		return;
	}

	retval = mx1_mmc_exec_command(request);

	if (retval) {

		/* If any error occured -> exit with error code */

		request->result = retval;
		mmc_cmd_complete(request);
	}

}

/**************************************************************************/
/* TODO: Fix MMC core for correct response processing */

static void
mx1_mmc_get_response(struct mmc_request *request)
{
	int i;

	u16 rs;
	u32 temp;

	request->result = MMC_NO_ERROR;	/* Mark this as having a request result of some kind */
	DEBUG(3, ": Get response \n");
	switch (request->rtype) {
	case RESPONSE_R1:
	case RESPONSE_R1B:
		request->response[0] = request->cmd;

		temp = _reg_MMC_RES_FIFO;
		rs = (u16) temp;
		request->response[1] = (u8) (rs & 0x00FF);

		temp = _reg_MMC_RES_FIFO;
		rs = (u16) temp;
		request->response[2] = (u8) (rs >> 8);
		if (request->cmd == 3 && request->response[1] == 0
		    && request->response[2] == 0x40) {
			request->response[2] &=
			    request->response[2] & ~(1 << 6);
		}		/* mmc device does not clear this bit after APP_CMD command for some mmc cards */
		request->response[3] = (u8) (rs & 0x00FF);

		temp = _reg_MMC_RES_FIFO;
		rs = (u16) temp;
		request->response[4] = (u8) (rs >> 8);
		break;

	case RESPONSE_R3:
	case RESPONSE_R4:
	case RESPONSE_R5:

		request->response[0] = 0x3f;

		temp = _reg_MMC_RES_FIFO;
		rs = (u16) temp;
		request->response[1] = (u8) (rs & 0x00FF);

		temp = _reg_MMC_RES_FIFO;
		rs = (u16) temp;
		request->response[2] = (u8) (rs >> 8);
		request->response[3] = (u8) (rs & 0x00FF);

		temp = _reg_MMC_RES_FIFO;
		rs = (u16) temp;
		request->response[4] = (u8) (rs >> 8);

		break;

	case RESPONSE_R2_CID:
	case RESPONSE_R2_CSD:
		request->response[0] = 0x3f;	/* for mmc core */

		for (i = 0; i < 8; i++) {

			temp = _reg_MMC_RES_FIFO;
			rs = (u16) temp;
			request->response[2 * i + 1] = (u8) (rs >> 8);
			request->response[2 * i + 2] = (u8) (rs & 0x00FF);

		}

	case RESPONSE_NONE:
	default:
		break;
	}

}

static void
mx1_mmc_handle_int(struct mx1_mmc_data *sd)
{
	int retval = MMC_NO_ERROR;
	u16 status = _reg_MMC_STATUS;

	if (status == 0x8140)
		return;

	mx1_mmc_stop_clock();	/* Clear status bits */
	_reg_MMC_INT_MASK = MX1_AUTO_CARD_DETECT_MASK & 0x7f;	/* mask and clear all interrupts */

	/* Data or Command time-out? */

	if (status & (MX1STAT_TIME_OUT_RESP | MX1STAT_TIME_OUT_READ)) {
		DEBUG(1, " TIMEOUT: MMC/SD status: %x \n", status);
		retval = MMC_ERROR_TIMEOUT;
		sd->request->result = MMC_ERROR_TIMEOUT;
		mx1_mmc_get_response(sd->request);
		goto terminate_int;
	}

	/* CRC error? */

	if ((status & (MX1STAT_RESP_CRC_ERR /* | MX1STAT_CRC_READ_ERR */ ))) {
		DEBUG(1, " MMCSD_RESP_CRC_ERR: MMC/SD status: %x \n", status);
		retval = MMC_ERROR_CRC;
		goto terminate_int;
	}

	if (((status & MX1STAT_END_CMD_RESP)
	     || (status & MX1STAT_DATA_TRANS_DONE)
	     || (status & MX1STAT_WRITE_OP_DONE))
	    && (sd->request->result == MMC_NO_RESPONSE)) {

		mx1_mmc_get_response(sd->request);
	}

	switch (g_mx1_mmc_data.type) {
	case RT_NO_RESPONSE:
		break;

	case RT_WRITE:
		if (sd->request->nob) {
			/* Start another transfer */
		}
		break;

	case RT_READ:
		if (sd->request->nob) {
			/* Start another transfer */
		}
		break;

	case RT_RESPONSE_ONLY:
		if (sd->request->result < 0) {
			printk(KERN_INFO
			       "MMC/SD: Illegal interrupt - command hasn't finished\n");
			retval = MMC_ERROR_TIMEOUT;
		}
		break;

	}

      terminate_int:
	sd->request->result = retval;
	mmc_cmd_complete(sd->request);
}

/* Handle IRQ */

static void
mx1_mmc_int(int irq, void *dev_id, struct pt_regs *regs)
{
	struct mx1_mmc_data *sd = (struct mx1_mmc_data *) dev_id;
	u32 status = _reg_MMC_STATUS & 0xFFFF;

	DEBUG(4, " 0x%x\n", _reg_MMC_STATUS);
	if ((status & ~MX1STAT_WRITE_CRC_ERROR_CODE) == 0x40
	    || (status & ~MX1STAT_WRITE_CRC_ERROR_CODE) == 0x8040) {
		mmc_eject(0);
		_reg_MMC_INT_MASK |= ~MX1_AUTO_CARD_DETECT_MASK;
		mx1_mmc_fix_sd_detect();
		_reg_MMC_INT_MASK &= MX1_AUTO_CARD_DETECT_MASK;
	} else
	    if ((sd->request->cmd == MMC_READ_SINGLE_BLOCK
		 || sd->request->cmd == MMC_READ_MULTIPLE_BLOCK)) {
		if (status & MX1STAT_APPL_BUFF_FF) {
			mx1_mmc_rx();
		} else if (status & MX1STAT_DATA_TRANS_DONE) {
			mx1_mmc_handle_int(sd);
		}
	} else
	    if ((sd->request->cmd == MMC_WRITE_BLOCK
		 || sd->request->cmd == MMC_WRITE_MULTIPLE_BLOCK)) {
		if (status & MX1STAT_WRITE_OP_DONE) {
			mx1_mmc_handle_int(sd);
		} else if (status & MX1STAT_DATA_TRANS_DONE) {
		} else if (status & MX1STAT_APPL_BUFF_FE) {
			mx1_mmc_tx();
		}
	} else {
		mx1_mmc_handle_int(sd);
	}
}

/* Detect Card */
static void
mx1_mmc_fix_sd_detect(void)
{
	card_state = (_reg_MMC_STATUS & 0xFFFF) & MX1STAT_CARD_PRESENCE;
	if (card_state > 0) {
		mmc_insert(0);
	} else {
		mmc_eject(0);
	}
}

static int
mx1_mmc_slot_is_empty(int slot)
{
	card_state = _reg_MMC_STATUS & MX1STAT_CARD_PRESENCE;
	return card_state == 0;
}

/**************************************************************************
 *
 *                       Hardware initialization
 *
 *************************************************************************/
static int
mx1_mmc_slot_up(void)
{

	int retval;

	/* MX1 MMC slot hardware init */

	/* Perclock 2 */
	_reg_PCDR &= 0xFFFFFF0F;
	_reg_PCDR |= 0x50;

	/* GPIO */

	retval = mx1_register_gpios(PORT_B, (1 << 13) |	/* MMC: SD_CMD */
				    (1 << 12) |	/* MMC: SD_CLK */
				    (1 << 11) |	/* MMC: SD_DAT */
				    (1 << 10) |	/* MMC: SD_DAT */
				    (1 << 9) |	/* MMC: SD_DAT */
				    (1 << 8),	/* MMC: SD_DAT */
				    PRIMARY | NOINTERRUPT);

	if (retval < 0)
		goto error;

      error:
	mx1_unregister_gpios(PORT_B,
			     (1 << 8) | (1 << 9) | (1 << 10) | (1 << 11) | (1 <<
									    12)
			     | (1 << 13));
	return retval;
}

static void
mx1_mmc_slot_down(void)
{

	disable_irq(IRQ_SDHC);

	/* Power Off MMC through FPGA */

	/* Disable MMC/SD clock and enable HI-Z on the MMC.DAT2 pin */

}

/* Standard PM functions */

static int
mx1_mmc_suspend(void)
{
	printk("mx1_mmc_suspend \n");
	mx1_mmc_stop_clock();
	mx1_mmc_slot_down();
	return 0;
}

static void
mx1_mmc_resume(void)
{
	printk("mx1_mmc_resume \n");
	mx1_mmc_set_clock(g_mx1_mmc_data.clock);

	enable_irq(IRQ_SDHC);
}

#ifdef CONFIG_PM

static int
mx1_mmc_pm_callback(struct pm_dev *pm_dev, pm_request_t req, void *data)
{

	switch (req) {
	case PM_SUSPEND:
		mmc_eject(0);
		mx1_mmc_suspend();
		break;

	case PM_RESUME:
		mx1_mmc_resume();
		mx1_mmc_fix_sd_detect();
		break;
	}

	return 0;

}

#endif

static int
mx1_mmc_slot_init(void)
{
	int retval;
	long flags;

	/* Basic service interrupt */

	local_irq_save(flags);

	mx1_mmc_slot_up();

	_MMCSD_SoftReset();

	/* check hardware revision */
	if ((_reg_MMC_REV_NO & 0xFFFF) != 0x0390) {
		printk(KERN_ERR "MMC Hardware revision = 0x%x",
		       _reg_MMC_REV_NO);
	}
	_reg_MMC_INT_MASK = MX1_AUTO_CARD_DETECT_MASK & 0x7f;	/* Clear all interrupts */

	retval = request_irq(IRQ_SDHC, mx1_mmc_int,
			     SA_INTERRUPT, "mx1_mmc_int", &g_mx1_mmc_data);

	if (retval) {
		printk(KERN_CRIT "MMC/SD: unable to grab MMC IRQ\n");
		return retval;
	}
#ifdef CONFIG_PM
	pm_register(PM_UNKNOWN_DEV, PM_SYS_UNKNOWN, mx1_mmc_pm_callback);
#endif

	enable_irq(IRQ_SDHC);
	local_irq_restore(flags);

	return retval;
}

static void
mx1_mmc_slot_cleanup(void)
{
	long flags;

	local_irq_save(flags);

	mx1_mmc_slot_down();

	free_irq(IRQ_SDHC, &g_mx1_mmc_data);

	local_irq_restore(flags);
}

/***********************************************************/

static struct mmc_slot_driver dops = {
	.owner = THIS_MODULE,
	.name = "MX1 MMC",
	.ocr = 0x00ff8000,
	.flags = MMC_SDFLAG_MMC_MODE,

	.init = mx1_mmc_slot_init,
	.cleanup = mx1_mmc_slot_cleanup,
	.is_empty = mx1_mmc_slot_is_empty,
	.send_cmd = mx1_mmc_send_command,
	.set_clock = mx1_mmc_set_clock
};

int __init
mx1_mmc_init(void)
{
	int retval;

	retval = mmc_register_slot_driver(&dops, 1);
	if (retval < 0)
		printk(KERN_INFO "MMC/SD: unable to register slot\n");

	return retval;
}

void __exit
mx1_mmc_cleanup(void)
{
	mmc_unregister_slot_driver(&dops);
	mx1_unregister_gpios(PORT_B,
			     (1 << 8) | (1 << 9) | (1 << 10) | (1 << 11) | (1 <<
									    12)
			     | (1 << 13));
}

#ifdef MODULE
module_init(mx1_mmc_init);
module_exit(mx1_mmc_cleanup);
#endif

MODULE_DESCRIPTION("MX1 MMC/SD");
MODULE_LICENSE("GPL");
EXPORT_NO_SYMBOLS;
