/******************************************************************************

    mx2ads-cam.c
    driver for Motorola MX2ADS on-board CMOS Image Sensor

    Author: MontaVista Software, Inc. <source@mvista.com>
    Copyright (c) 2004 MontaVista Software, Inc.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

********************************************************************************/
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/mm.h>

#include <linux/delay.h>
#include <asm/io.h>

#include <asm/dma.h>
#include <asm/arch/dma.h>
#include <asm/arch/i2c.h>
#include <asm/arch/gpio.h>
#include <asm/arch/pll.h>
#include <asm/uaccess.h>
#include <asm/arch/hardware.h>
#include <asm/arch/irqs.h>

#ifdef CONFIG_PM
#include <linux/pm.h>
#endif

#include <linux/i2c.h>
#include <linux/ioport.h>

#include "mx2ads-cam.h"

#ifndef CONFIG_MX2TO1
#define MX2CAM_PERCLKDIV 0x2
#endif

static void capture_interrupt(void *v);

static unsigned long
current_time_ms(void)
{
	struct timeval now;

	do_gettimeofday(&now);
	return now.tv_sec * 1000 + now.tv_usec / 1000;
}

/**************************************************************************
 * Micron Mi-Soc 0343 Image Sensor specific definitions
 **************************************************************************/

/*
 * 1bit shift is introduced due to the i2c driver requirements
 * (it will shift it at one bit again)
 */
#define MISOC0343_I2C_ADDR (0xb8 >> 1)

#define MISOC0343_BYPP 2

#define MISOC0343_IFP_WRITE16(_reg, _val) \
    do {                                                              \
	    misoc0343_i2c_write16_check(MISOC0343_R1, MISOC0343_IFP); \
	    misoc0343_i2c_write16((_reg), (_val));                    \
    } while(0)

#define MISOC0343_IC_WRITE16(_reg, _val) \
    do {                                                              \
	    misoc0343_i2c_write16_check(MISOC0343_R1, MISOC0343_IC);  \
	    misoc0343_i2c_write16((_reg), (_val));                    \
    } while(0)

#define MISOC0343_IFP_READ16(_reg, _val) \
    do {                                                               \
	    misoc0343_i2c_write16_check(MISOC0343_R1, MISOC0343_IFP);  \
	    misoc0343_i2c_read16((_reg), &(_val));                     \
    } while(0)

#define MISOC0343_IC_READ16(_reg, _val) \
    do {                                                               \
	    misoc0343_i2c_write16_check(MISOC0343_R1, MISOC0343_IC);   \
	    misoc0343_i2c_read16((_reg), &(_val));                     \
    } while(0)

static void misoc0343_i2c_write16(uint8_t reg_addr, uint16_t data);
static void misoc0343_i2c_read16(uint8_t reg_addr, uint16_t * data);
static int misoc0343_i2c_write16_check(uint8_t reg_addr, uint16_t data);

#define CSI_IRQ_ALL   (CSICR1_SOF_INTEN |  \
		       CSICR1_STATFF_INTEN | CSICR1_RXFF_INTEN | \
		       CSICR1_RXFFOR_INTEN | CSICR1_STATFFOR_INTEN |  \
		       CSICR1_COF_INTEN | CSICR1_EOF_INTEN)

#define CSI_IRQ_SOF   (CSICR1_SOF_INTEN)

#define CSI_GPIO_MASK (0x003ffc00)

static struct capture_device mx2ads_capture_device;

/**************************************************************************
 * Micron Mi-Soc 0343 Image Sensor specific functions
 **************************************************************************/

/*
 * Reset camera
 */
static void
misoc0343_reset(void)
{
	/*
	 * Reset Image core
	 */
	MISOC0343_IC_WRITE16(MISOC0343_IC_RESET, 0x01);
	MISOC0343_IC_WRITE16(MISOC0343_IC_RESET, 0x00);

	/*
	 * Reset Image Flow
	 */
	MISOC0343_IFP_WRITE16(MISOC0343_IFP_RESET, 0x01);
	MISOC0343_IFP_WRITE16(MISOC0343_IFP_RESET, 0x00);

	return;
}

/*
 * turn camera into 565rgb mode
 */
static void
misoc0343_setformat_565rgb(void)
{
	uint16_t reg_val;

	MISOC0343_IFP_READ16(MISOC0343_IFP_FORMAT, reg_val);

	reg_val |= MISOC0343_FORMAT_565RGB;

	MISOC0343_IFP_WRITE16(MISOC0343_IFP_FORMAT, reg_val);

	return;
}

/*
 * I2C access routines: reg_addr - Micron register address, data - data to be
 * written (or read).
 */

static void
misoc0343_i2c_write16(uint8_t reg_addr, uint16_t data)
{
	struct i2c_msg msg;
	uint8_t buf[3];

	buf[0] = reg_addr;
	buf[1] = (uint8_t) ((data & 0xff00) >> 8);
	buf[2] = (uint8_t) (data & 0x00ff);

	msg.addr = MISOC0343_I2C_ADDR;
	msg.flags = I2C_M_WR;
	msg.len = 3;
	msg.buf = (char *) buf;

	if (mx2_i2c_xfer(&msg, 1) != 1) {
		pr_debug("mx2_i2c_xfer failed\n");
	}

	return;
}

static int
misoc0343_i2c_write16_check(uint8_t reg_addr, uint16_t data)
{
	int rc = 0;
	uint16_t tmp_data;

	misoc0343_i2c_write16(reg_addr, data);
	misoc0343_i2c_read16(reg_addr, &tmp_data);

	if (data != tmp_data) {
		pr_debug
		    ("%s: i2c data write check failure, addr %x, write %x read %x \n",
		     __FUNCTION__, reg_addr, data, tmp_data);
		rc = -1;
	}

	return rc;
}

static void
misoc0343_i2c_read16(uint8_t reg_addr, uint16_t * data)
{
	struct i2c_msg msg[2];	/* Msg for write and read cycles */
	uint8_t buf[4];

	buf[0] = reg_addr;
	buf[1] = buf[2] = buf[3] = 0;

	msg[0].addr = MISOC0343_I2C_ADDR;
	msg[0].flags = I2C_M_WR;
	msg[0].len = 1;
	msg[0].buf = (char *) buf;

	msg[1].addr = MISOC0343_I2C_ADDR;
	msg[1].flags = I2C_M_RD;
	msg[1].len = 2;
	msg[1].buf = (char *) (buf + 1);

	if (mx2_i2c_xfer(msg, 2) != 2) {
		pr_debug("mx2_i2c_xfer failed\n");
	}

	*data = (buf[1] << 8) | buf[2];

	return;
}

/*
 * Initialise board-specfic settings to init Micron camera
 */
static void
mx2ads_misoc0343_initialise(void)
{
	uint16_t reg_val;

	CSI_REG_READ(EXP_IO, reg_val);

	/*
	 * Make standby low (disable standby)
	 */
	reg_val &= ~EXP_IO_CSI_CTL0;

	CSI_REG_WRITE(EXP_IO, reg_val);

	udelay(20);

	/*
	 * Make reset low (reset sensor)
	 */
	reg_val &= ~EXP_IO_CSI_CTL1;

	CSI_REG_WRITE(EXP_IO, reg_val);

	udelay(200);

	/*
	 * Make reset high
	 */
	reg_val |= EXP_IO_CSI_CTL1;

	CSI_REG_WRITE(EXP_IO, reg_val);

	udelay(200);

	return;
}

/*
 * Check - whether Micron compatible camera present
 */
static int
misoc0343_detect(void)
{
	uint16_t reg_val;

	misoc0343_reset();

	MISOC0343_IC_READ16(MISOC0343_IC_VERSION, reg_val);

	if (reg_val != MISOC0343_CORE_VERSION)
		return -1;

	MISOC0343_IFP_READ16(MISOC0343_IFP_VERSION, reg_val);

	if (reg_val != MISOC0343_IMAGE_VERSION)
		return -1;

	return 0;
}

static void
misoc0343_set_format(struct v4l2_pix_format *fmt)
{
	uint16_t reg_val;

	if (fmt->width >= 640) {
		fmt->width = 640;
		fmt->height = 480;
		reg_val = MISOC0343_DECIMATION_VGA;
	} else if (fmt->width == 352) {
		fmt->height = 288;
		reg_val = MISOC0343_DECIMATION_CIF;
	} else if (fmt->width == 320) {
		fmt->height = 240;
		reg_val = MISOC0343_DECIMATION_QVGA;
	} else if (fmt->width == 176) {
		fmt->height = 144;
		reg_val = MISOC0343_DECIMATION_QCIF;
	} else {
		fmt->width = 160;
		fmt->height = 120;
		reg_val = MISOC0343_DECIMATION_QQVGA;
	}

	MISOC0343_IFP_WRITE16(MISOC0343_IFP_DECIMATION, reg_val);

	/*
	 * Invert picture
	 */
	MISOC0343_IC_WRITE16(MISOC0343_IC_RDMODE,
			     (MISOC0343_RDMODE_UPSDOWN |
			      MISOC0343_RDMODE_MIRRORED |
			      MISOC0343_RDMODE_BOOSTRST |
			      MISOC0343_RDMODE_ALLFRAMES));

	MISOC0343_IFP_WRITE16(MISOC0343_IFP_SHWIDTH, MISOC0343_SHWIDTH_DEFAULT);

	misoc0343_setformat_565rgb();

	return;
}

/**************************************************************************
 * MX21 CSI module specific functions
 **************************************************************************/

static void csi_dma_start(void);

/* Dma channel for data tranfer between CSI and memory buffer */
int csi_dma_chan;

/* Pointer to the memory area for image capturing via DMA and size of memory area */
uint8_t *csi_dma_buf = 0;
int csi_dma_buf_size = 0;

static void
csi_irq_disable(uint32_t mask)
{
	uint32_t reg_val;

	CSI_REG_READ(CSISR, reg_val);

	reg_val = (CSISR_COF_INT | CSISR_SOF_INT | CSISR_EOF_INT |
		   CSISR_RXFF_FULL | CSISR_STATFF_FULL | CSISR_RXFFOR_INT |
		   CSISR_STATFFOR_INT);

	CSI_REG_WRITE(CSISR, reg_val);

	CSI_REG_READ(CSICR1, reg_val);

	reg_val &= ~mask;

	CSI_REG_WRITE(CSICR1, reg_val);
}

static void
csi_irq_enable(uint32_t mask)
{
	uint32_t reg_val;

	CSI_REG_READ(CSICR1, reg_val);

	reg_val |= mask;

	CSI_REG_WRITE(CSICR1, reg_val);
}

static void
csi_misoc0343_565rgb_initialise(struct capture_device *dev)
{
	uint32_t reg_val = 0;

	CSI_REG_WRITE(CSICR1, 0);

	/*
	 * Configure CSI:
	 */

	/*
	 * Latch on rising edge, gated clock mode, hsync active mode;
	 * RGB565 mode
	 */
	reg_val = (CSICR1_REDGE | CSICR1_GCLK_MODE | CSICR1_HSYNC_POL_HIGH);

	/*
	 * Syncrhonous FIFO clear; MCLK divided by 6;
	 */
	reg_val &= ~0xf000;	/*clear old MCLK setting */
	reg_val |= (CSICR1_FCC_SCLR | CSICR1_MCLKDIV_6 | CSICR1_MCLKEN);

	/*
	 * External VSync, SOF polarity High Edge
	 */
	reg_val |= (CSICR1_EXT_VSYNC | CSICR1_SOF_POL_RISE);

	/*
	 * FIFO depth 16;
	 */
	reg_val |= CSICR1_RXFF_LEVEL_24;

	/*
	 * Swaping enable; big endian data representation
	 */
	reg_val |= (CSICR1_SWAP16_EN | CSICR1_PACK_DIR_MSBFIRST);

	CSI_REG_WRITE(CSICR1, reg_val);

	/*
	 * Setup RxFIFO limit
	 */
	CSI_REG_WRITE(CSIRXCNT, dev->capture_size);

	return;
}

/*
 * Initialise CSI for first use (set gpios, reserve memory)
 */
int
csi_generic_initialise(struct capture_device *dev)
{
	uint32_t reg_val;

	/*
	 * reserve virtual addresses for CSI registers physical address
	 */
	if (!request_mem_region(CSI_BASE, CSI_IO_SIZE, "MX21.CSI registers")) {
		printk(KERN_ERR
		       "MX2 Camera: CSI memory region is already in use\n");
		pr_debug("Address=0x%08x, size=0x%x\n", CSI_BASE, CSI_IO_SIZE);
		return -1;
	}
	dev->mem_region_reserved = 1;

	/*
	 * Enable CSI GPIOs
	 */
	if (mx2_register_gpios(PORT_B, CSI_GPIO_MASK, PRIMARY)) {
		printk(KERN_ERR
		       "MX2 Camera: CSI GPIO pins are already in use\n");
		return -1;
	}

	/*
	 * Before access to CSI registers, CSI clock control should be configured
	 */

#ifdef CONFIG_MX2TO1
	mx_module_clk_open(HCLK_MODULE_CSI);
#else
	/* adjust CSI PerClock 4 */
	CRM_PCDR1 = (CRM_PCDR1 & ~0x3f000000) | (MX2CAM_PERCLKDIV << 24);
	/* enable CSI PerClock 4 and HCLK */
	CRM_PCCR0 |= (1 << 31) | (1 << 22);
#endif

	/*
	 * Enable clock for censor
	 */
	CSI_REG_READ(CSICR1, reg_val);

	reg_val &= ~0xf000;	/*clear old MCLK setting */
	reg_val |= (CSICR1_MCLKDIV_6 | CSICR1_MCLKEN);

	CSI_REG_WRITE(CSICR1, reg_val);

	return 0;
}

/*
 * Cleanup CSI
 */
void
csi_generic_cleanup(struct capture_device *dev)
{
	if (dev->mem_region_reserved)
		release_mem_region(CSI_BASE, CSI_IO_SIZE);

	mx2_unregister_gpios(PORT_B, CSI_GPIO_MASK);

#ifdef CONFIG_MX2TO1
	mx_module_clk_close(HCLK_MODULE_CSI);
#else
	/* disable CSI PerClock 4 and HCLK */
	CRM_PCCR0 &= ~((1 << 31) | (1 << 22));
#endif

	return;
}

/*
 * Handler for CSI IRQ Processing
 */
static void
csi_irq_handler(int irq, void *dev_id, struct pt_regs *regs)
{
	uint32_t reg_val = 0;

	CSI_REG_READ(CSISR, reg_val);

	if (reg_val & CSISR_SOF_INT) {

		CSI_REG_WRITE(CSISR, CSISR_SOF_INT);

		csi_dma_start();

		csi_irq_disable(CSI_IRQ_SOF);

		return;
	}

	pr_debug("IMAGE SENSOR ERROR: unknown interrupt CSISR=0x%08x\n",
		 reg_val);

	return;
}

/*
 * Activate dma cycle
 */
static void
csi_dma_start(void)
{
	mx2_dma_disable(csi_dma_chan);
	mx2_set_dma_destination(csi_dma_chan,
				virt_to_phys((void *) csi_dma_buf),
				csi_dma_buf_size);
	mx2_dma_enable(csi_dma_chan);

	return;
}

/*
 * Handler for CSI-DMA IRQ Processing
 */
static void
csi_dma_handler(dmach_t channel, void *dev, int status)
{
	if (status & DMA_DONE) {
		capture_interrupt(&mx2ads_capture_device);
	} else {
		if (status & DMA_BURST_TIMEOUT)
			pr_debug("%s: DMA_BURST_TIMEOUT\n", __FUNCTION__);
		if (status & DMA_TRANSFER_ERROR)
			pr_debug("%s: DMA_TRANSFER_ERROR\n", __FUNCTION__);
		if (status & DMA_BUFFER_OVERFLOW)
			pr_debug("%s: DMA_BUFFER_OVERFLOW\n", __FUNCTION__);
		if (status & DMA_REQUEST_TIMEOUT)
			pr_debug("%s: DMA_REQUEST_TIMEOUT\n", __FUNCTION__);
	}

	return;
}

/*
 * Perpare CSI (in open call) for image reading/writing
 */
static int
csi_generic_open(void)
{
	csi_irq_disable(CSI_IRQ_ALL);

	if (request_irq(INT_CSI, csi_irq_handler, SA_INTERRUPT,
			"CSI Image Sensor IRQ", NULL)) {
		printk(KERN_ERR "MX2 Camera: failed to request irq\n");
		return -1;
	}

	for (csi_dma_chan = 0; csi_dma_chan < MAX_DMA_CHANNELS; csi_dma_chan++) {
		if (!request_dma(csi_dma_chan, "CSI Image Sensor DMA channel"))
			break;
	}

	if (csi_dma_chan == MAX_DMA_CHANNELS) {
		printk(KERN_ERR "MX2 Camera: failed to request DMA\n");
		csi_dma_chan = -1;
		return -1;
	}

	if (mx2_request_dma_irq(csi_dma_chan, csi_dma_handler,
				"CSI DMA Handler") < 0) {
		printk(KERN_ERR
		       "MX2 Camera: failed to request DMA IRQ for channel %d\n",
		       csi_dma_chan);
		return -1;
	}

	mx2_conf_dma_source(csi_dma_chan, DMA_TYPE_FIFO, DMA_MEM_SIZE_32,
			    _CSI_CSIRXR);
	mx2_conf_dma_destination(csi_dma_chan, DMA_TYPE_LINEAR,
				 DMA_MEM_SIZE_32, 0);
	mx2_conf_dma_memdir(csi_dma_chan, DMA_MEMDIR_INCR);
	mx2_conf_dma_request(csi_dma_chan, DMA_REQUEST_EN,
			     DMA_REQ_CSI_RX, DMA_REQ_TIMEOUT_DIS, 0, 0, 0);
	mx2_conf_dma_burst(csi_dma_chan, 64, 0);	/* burst length - 64 */

	return 0;
}

/*
 * Deallocate CSI-related resources
 */
static void
csi_generic_close(void)
{
	csi_irq_disable(CSI_IRQ_ALL);

	free_irq(INT_CSI, NULL);

	mx2_free_dma_irq(csi_dma_chan, "CSI DMA Handler");

	free_dma(csi_dma_chan);

	return;
}

static void
csi_capture_activate(uint8_t * buffer, int size)
{
	csi_irq_disable(CSI_IRQ_ALL);

	csi_irq_enable(CSI_IRQ_SOF);

	csi_dma_buf = buffer;
	csi_dma_buf_size = size;

	return;
}

static void
csi_capture_abort(void)
{
	csi_irq_disable(CSI_IRQ_ALL);

	mx2_dma_disable(csi_dma_chan);

	return;
}

static void
csi_suspend(void)
{
	csi_capture_abort();
#ifdef CONFIG_MX2TO1
	mx_module_clk_close(HCLK_MODULE_CSI);
#else
	/* disable CSI PerClock 4 and HCLK */
	CRM_PCCR0 &= ~((1 << 31) | (1 << 22));
#endif

	return;
}

static void
csi_resume(void)
{
#ifdef CONFIG_MX2TO1
	mx_module_clk_open(HCLK_MODULE_CSI);
#else
	/* enable CSI PerClock 4 and HCLK */
	CRM_PCCR0 |= (1 << 31) | (1 << 22);
#endif
	return;
}

/**************************************************************************
 * Capturing primitives
 **************************************************************************/

/*
 *  Supported capture formats
 */
static struct v4l2_fmtdesc capfmt[] = {
	{1, {"RGB-16 (5-6-5)"},
	 V4L2_PIX_FMT_RGB565, 0, 16, {0, 0},
	 }
};
#define NUM_CAPFMT (sizeof(capfmt)/sizeof(capfmt[0]))

static void capture_abort(struct capture_device *dev);
static int capture_convert_image(struct capture_device *dev,
				 uint8_t * capture_buffer,
				 uint8_t * output_buffer, int output_is_user);

static void
DeallocateBuffer(struct capture_device *dev)
{
	if (dev->capture_buffer != NULL) {
		free_pages((unsigned long) dev->capture_buffer,
			   get_order(dev->capture_buffer_size));
	}

	dev->capture_buffer = NULL;
	return;
}

static void
AllocateBuffer(struct capture_device *dev)
{
	DeallocateBuffer(dev);

	dev->capture_buffer_size = dev->capture_size;
	dev->capture_buffer = (uint8_t *) __get_dma_pages(GFP_KERNEL,
							  get_order(dev->
								    capture_buffer_size));
}

static int
capture_new_format(struct capture_device *dev)
{
	dev->ready_to_capture = 0;

	switch (dev->clientfmt.pixelformat) {
	case V4L2_PIX_FMT_GREY:
		dev->clientfmt.depth = 8;
		break;

	case V4L2_PIX_FMT_YUV420:
		dev->clientfmt.depth = 12;
		break;

	case V4L2_PIX_FMT_RGB555:
	case V4L2_PIX_FMT_RGB565:
		dev->clientfmt.flags = 0;
	case V4L2_PIX_FMT_YUYV:
	case V4L2_PIX_FMT_UYVY:
		dev->clientfmt.depth = 16;
		break;

	case V4L2_PIX_FMT_BGR24:
		dev->clientfmt.depth = 24;
		dev->clientfmt.flags = 0;
		break;

	case V4L2_PIX_FMT_BGR32:
		dev->clientfmt.depth = 32;
		dev->clientfmt.flags = 0;
		break;

	default:
		pr_debug("unknown format %4.4s\n",
			 (char *) &dev->clientfmt.pixelformat);
		dev->clientfmt.depth = 16;
		dev->clientfmt.pixelformat = V4L2_PIX_FMT_YUYV;
		dev->clientfmt.flags = 0;
		break;
	}

	dev->capture_bypp = MISOC0343_BYPP;

	if (dev->clientfmt.width < MIN_WIDTH)
		dev->clientfmt.width = MIN_WIDTH;
	if (dev->clientfmt.height < MIN_HEIGHT)
		dev->clientfmt.height = MIN_HEIGHT;

	if (dev->clientfmt.width > MAX_WIDTH)
		dev->clientfmt.width = MAX_WIDTH;
	if (dev->clientfmt.height > MAX_HEIGHT)
		dev->clientfmt.height = MAX_HEIGHT;

	dev->clientfmt.width &= ~3;
	dev->clientfmt.height &= ~3;

	misoc0343_set_format(&dev->clientfmt);

	dev->clientfmt.sizeimage =
	    (dev->clientfmt.width * dev->clientfmt.height *
	     dev->clientfmt.depth) / 8;

	dev->capture_size =
	    dev->clientfmt.width * dev->clientfmt.height * dev->capture_bypp;

	csi_misoc0343_565rgb_initialise(dev);

	return 0;
}

static void
capture_close(struct capture_device *dev)
{
	capture_abort(dev);

	dev->ready_to_capture = 0;

	DeallocateBuffer(dev);
	dev->capture_buffer = NULL;
	dev->capture_buffer_size = 0;

	return;
}

static int
capture_begin(struct capture_device *dev)
{
	capture_abort(dev);

	if (dev->ready_to_capture) {
		return dev->ready_to_capture;
	}

	if (dev->capture_buffer_size < dev->capture_size) {
		AllocateBuffer(dev);
		if (dev->capture_buffer == NULL) {
			dev->capture_buffer_size = 0;
			pr_debug("Can't allocate capture buffer"
				 " %d bytes\n", dev->capture_size);
			return dev->ready_to_capture;
		}
	}
	pr_debug("Ready to capture!\n");

	return (dev->ready_to_capture = 1);
}

/*
 * Start an image capture
 */
static void
capture_grab_frame(struct capture_device *dev)
{
	if (dev->ready_to_capture && dev->capture_started)
		return;

	capture_begin(dev);
	if (!dev->ready_to_capture)
		return;

	/*
	 * Start the camera h/w.
	 */
	csi_capture_activate(dev->capture_buffer, dev->capture_size);

	dev->capture_started = 1;
	dev->capture_completed = 0;

	return;
}

/*
 * Stop the music!
 */
static void
capture_abort(struct capture_device *dev)
{
	pr_debug("capture abort\n");

	/*  Turn off the capture hardware  */
	csi_capture_abort();
	dev->capture_started = 0;

	/*
	 * Wake up any processes that might be waiting for a frame
	 * and let them return an error
	 */
	wake_up_interruptible(&dev->new_video_frame);

	return;
}

/*  Read captured data into a user buffer.
 *  Return: negative = error
 *          0        = keep waiting
 *          positive = count of bytes read successfully
 */
static long
capture_read(struct capture_device *dev,
	     uint8_t * user_buffer, int user_buffer_size)
{
	int len = user_buffer_size;
	unsigned long now;

	if (!dev->capture_completed) {
		/* No interrupt has occurred yet, or DMA didn't finish.  */
		pr_debug("No data ready.\n");
		if (!dev->capture_started) {
			capture_grab_frame(dev);
		}
		return 0;	/* caller should keep waiting */
	}

	now = current_time_ms();
	if (now - dev->time_acquired > MAX_FRAME_AGE) {
		/* Frame in buffer is stale, get a new one */
		pr_debug("Stale frame, re-acquiring.\n");
		dev->capture_started = 0;
		capture_grab_frame(dev);
		return 0;	/* caller should keep waiting */
	}

	len = capture_convert_image(dev, dev->capture_buffer, user_buffer, 1);
	dev->capture_started = 0;
	capture_grab_frame(dev);
	return len;
}

/*
 * Convert the next frame.
 * Returns length of data or negative for error.
 */
static int
capture_convert_image(struct capture_device *dev,
		      uint8_t * capture_buffer,
		      uint8_t * output_buffer, int output_is_user)
{
	if (!dev->capture_started) {
		pr_debug("capture not started??\n");
		return 0;
	}

	if (output_is_user) {
		copy_to_user(output_buffer, capture_buffer,
			     dev->clientfmt.sizeimage);
	} else {
		memcpy(output_buffer, capture_buffer, dev->clientfmt.sizeimage);
	}

	dev->perf.frames++;
	dev->perf.bytesout += dev->clientfmt.sizeimage;

	return dev->clientfmt.sizeimage;
}

/*  The hardware has issued the interrupt signal, do any post-capture
 *  processing that may be necessary.
 *  [This function is called indirectly through the immediate task queue;
 *  it executes at elevated IRQL, but it is interruptible. (It's a b.h.)]
 */
static void
capture_interrupt(void *v)
{
	struct capture_device *dev = (struct capture_device *) v;

	pr_debug("Capture interrupt\n");

	/*  TODO: Check for an interrupt pending on the device, and  */
	/*        return if there is no interrupt pending  */
	/*  (In this hardware-less demo I'll just check the completed flag) */
	if (!dev->capture_started || dev->capture_completed)
		return;

	dev->capture_completed = 1;

	if (dev->capture_buffer == NULL) {
		pr_debug("Can't process the interrupt\n");
		return;
	}

	dev->time_acquired = current_time_ms();

	/* DMA might not have finished, but we'll check in read() */

	wake_up_interruptible(&dev->new_video_frame);
	return;
}

/**************************************************************************
 * MX21 Video4l2 entrypoints
 **************************************************************************/

static int
v4l2_open(struct v4l2_device *v, int flags, void **idptr)
{
	struct capture_device *dev = (struct capture_device *) v;
	int i, n;
	int cap;

	pr_debug("MX2ADS CMOS Image Sensor Driver Open\n");

	for (i = 0, n = -1, cap = 0; i < MAX_OPENS; i++) {
		if (!dev->open_data[i].isopen)
			n = i;	/* available open_data structure */
		else if (!dev->open_data[i].noncapturing)
			cap = 1;	/* another open is already capturing */
	}

	if (n == -1) {		/* No available open_data structures */
		pr_debug("No more opens on this device\n");
		return -EBUSY;
	}

	if (flags & O_NONCAP)	/*  Non-capturing open */
		dev->open_data[n].noncapturing = 1;
	else if (cap) {
		pr_debug("No more capturing opens on this device\n");
		return -EBUSY;
	} else {
		dev->open_data[n].noncapturing = 0;

		/*  Keep track of whether there is a capturing open  */
		dev->capturing_opens++;
	}

	/*
	 * As It comes from v4l2 examples - don't use MOD_INC macro
	 */

	dev->open_count++;
	dev->open_data[n].isopen = 1;
	dev->open_data[n].dev = dev;
	*idptr = &dev->open_data[n];

	if (dev->open_count == 1) {

		if (csi_generic_open()) {
			return -EFAULT;
		}

		dev->ready_to_capture = 0;	/* benchmark changes parameters! */
		dev->capture_completed = 0;
		dev->capture_started = 0;
	}
	pr_debug("Open succeeded\n");

	return 0;
}

static void
v4l2_close(void *id)
{
	struct device_open *o = (struct device_open *) id;
	struct capture_device *dev = o->dev;

	pr_debug("MX2ADS CMOS Image Sensor Driver Close\n");

	if (!o->noncapturing) {
		dev->capturing_opens--;
	}

	o->isopen = 0;
	dev->open_count--;

	if (dev->open_count == 0)
		csi_generic_close();

	return;
}

static long
v4l2_write(void *id, const char *buf, unsigned long count, int noblock)
{
	pr_debug("%s: operation not supported\n", __FUNCTION__);
	return -EINVAL;
}

static int
v4l2_ioctl(void *id, unsigned int cmd, void *arg)
{
	struct device_open *o = (struct device_open *) id;
	struct capture_device *dev = o->dev;

	pr_debug("MX2ADS CMOS Image Sensor Driver Ioctl\n");

	switch (cmd) {
	case VIDIOC_QUERYCAP:
		{
			struct v4l2_capability *b = arg;

			strcpy(b->name, dev->v.name);
			b->type = V4L2_TYPE_CAPTURE;
			b->flags = V4L2_FLAG_READ | V4L2_FLAG_SELECT;
			b->inputs = 0;
			b->outputs = 0;
			b->audios = 0;
			b->maxwidth = MAX_WIDTH;
			b->maxheight = MAX_HEIGHT;
			b->minwidth = MIN_WIDTH;
			b->minheight = MIN_HEIGHT;
			b->maxframerate = 30;
			return 0;
		}

	case VIDIOC_ENUM_PIXFMT:
		{
			struct v4l2_fmtdesc *f = arg;

			if (f->index < 0 || f->index >= NUM_CAPFMT)
				return -EINVAL;
			*f = capfmt[f->index];
			return 0;
		}

	case VIDIOC_G_FMT:
		{
			struct v4l2_format *fmt = arg;

			if (fmt->type != V4L2_BUF_TYPE_CAPTURE) {
				pr_debug("G_FMT wrong buffer type %d\n",
					 fmt->type);
				return -EINVAL;
			}

			fmt->fmt.pix = dev->clientfmt;
			return 0;
		}

	case VIDIOC_S_FMT:
		{
			struct v4l2_format *fmt = arg;

			if (o->noncapturing) {
				pr_debug
				    ("S_FMT illegal in non-capturing open\n");
				return -EPERM;
			}

			if (fmt->type != V4L2_BUF_TYPE_CAPTURE) {
				pr_debug("S_FMT wrong buffer type %d\n",
					 fmt->type);
				return -EINVAL;
			}

			dev->clientfmt = fmt->fmt.pix;

			capture_abort(dev);
			if (capture_new_format(dev))
				return -EINVAL;

			fmt->fmt.pix = dev->clientfmt;
			return 0;
		}

	case VIDIOC_G_COMP:
	case VIDIOC_S_COMP:
		return -EINVAL;

	case VIDIOC_REQBUFS:
	case VIDIOC_QUERYBUF:
	case VIDIOC_QBUF:
	case VIDIOC_DQBUF:
		return -EINVAL;

	case VIDIOC_STREAMON:
	case VIDIOC_STREAMOFF:
		return -EINVAL;

	case VIDIOC_ENUM_FBUFFMT:
	case VIDIOC_G_FBUF:
	case VIDIOC_S_FBUF:
		return -EINVAL;

	case VIDIOC_G_WIN:
	case VIDIOC_S_WIN:
		return -EINVAL;

	case VIDIOC_PREVIEW:
		return -EINVAL;

	case VIDIOC_G_PERF:
		{
			memcpy(arg, &dev->perf, sizeof (dev->perf));
			return 0;
		}

	case VIDIOC_G_INPUT:
		{
			int input = 0;
			memcpy(arg, &input, sizeof (input));
			return 0;
		}

	case VIDIOC_S_INPUT:
		return -EINVAL;

	case VIDIOC_G_PARM:
	case VIDIOC_S_PARM:
		return -EINVAL;

	case VIDIOC_G_STD:
	case VIDIOC_S_STD:
	case VIDIOC_ENUMSTD:
		return -EINVAL;

	case VIDIOC_ENUMINPUT:
		return -EINVAL;

	case VIDIOC_QUERYCTRL:
	case VIDIOC_QUERYMENU:
	case VIDIOC_G_CTRL:
	case VIDIOC_S_CTRL:
		return -EINVAL;

	case VIDIOC_G_TUNER:
	case VIDIOC_S_TUNER:
		return -EINVAL;

	case VIDIOC_G_FREQ:
	case VIDIOC_S_FREQ:
		return -EINVAL;

	case VIDIOC_G_AUDIO:
	case VIDIOC_S_AUDIO:
		return -EINVAL;

	default:
		return -ENOIOCTLCMD;
	}
	return 0;
}

static int
v4l2_poll(void *id, struct file *file, poll_table * table)
{
	struct device_open *o = (struct device_open *) id;
	struct capture_device *dev = o->dev;

	pr_debug("MX2ADS CMOS Image Sensor Driver Poll\n");

	if (o->noncapturing) {
		pr_debug("poll() illegal in non-capturing open\n");
		return POLLERR;
	}

	/*
	 * Note: streaming not supported
	 */

	/*  Capture is through read() call */

	if (dev->capture_completed)	/* data is ready now */
		return (POLLIN | POLLRDNORM);

	capture_grab_frame(dev);	/* does nothing if capture is in progress */
	if (!dev->ready_to_capture) {
		pr_debug("Can't grab frames!\n");
		return POLLERR;
	}

	poll_wait(file, &dev->new_video_frame, table);
	return 0;
}

static long
v4l2_read(void *id, char *buf, unsigned long count, int noblock)
{
	struct device_open *o = (struct device_open *) id;
	struct capture_device *dev = o->dev;
	long len = 0;
	long my_timeout;

	pr_debug("MX2ADS CMOS Image Sensor Driver Read\n");

	if (o->noncapturing) {
		pr_debug("read() illegal in non-capturing open\n");
		return -EPERM;
	}

	capture_grab_frame(dev);	/* does nothing if capture is in progress */

	if (!dev->ready_to_capture) {
		pr_debug("Can't grab frames!\n");
		return 0;
	}

	my_timeout = HZ;
	while (len == 0) {
		if (!dev->capture_completed) {
			if (noblock)
				return -EAGAIN;
			my_timeout =
			    interruptible_sleep_on_timeout(&dev->
							   new_video_frame,
							   my_timeout);
		}

		if (my_timeout == 0) {
			pr_debug("Timeout on read\n");
			break;
		}
		len = capture_read(dev, buf, count);
	}

	return len;
}

static int
v4l2_init_done(struct v4l2_device *v)
{
	struct capture_device *dev = (struct capture_device *) v;

	pr_debug("MX2ADS CMOS Image Init_Done\n");

	dev->capture.capability = V4L2_CAP_TIMEPERFRAME | V4L2_MODE_HIGHQUALITY;
	dev->capture.capturemode = 0;
	dev->capture.extendedmode = 0;
	dev->capture.timeperframe = 1;

	dev->clientfmt.width = DEFAULT_WIDTH;
	dev->clientfmt.height = DEFAULT_HEIGHT;

	dev->clientfmt.depth = DEFAULT_DEPTH;
	dev->clientfmt.pixelformat = V4L2_PIX_FMT_RGB565;
	dev->clientfmt.flags = 0;
	dev->clientfmt.bytesperline = 0;
	dev->clientfmt.sizeimage = 0;

	capture_new_format(dev);

	return 0;
}

static int
config_a_device(struct capture_device *dev)
{
	snprintf(dev->v.name, 32, "MX21 CMOS Camera V4L2 Driver");

	dev->v.type = V4L2_TYPE_CAPTURE;
	dev->v.minor = 0;

	dev->v.open = v4l2_open;
	dev->v.close = v4l2_close;
	dev->v.read = v4l2_read;
	dev->v.write = v4l2_write;
	dev->v.ioctl = v4l2_ioctl;
	dev->v.mmap = NULL;
	dev->v.poll = v4l2_poll;

	dev->v.initialize = v4l2_init_done;
	dev->v.priv = NULL;

	return 0;
}

#ifdef CONFIG_PM

static struct pm_dev *mx2ads_image_pmdev;

static int
mx2ads_image_pm_callback(struct pm_dev *pmdev, pm_request_t rqst, void *data)
{
	if (mx2ads_capture_device.capturing_opens != 0)
		return 0;

	switch (rqst) {
	case PM_SUSPEND:
		csi_suspend();
		break;
	case PM_RESUME:
		csi_resume();
		break;
	}
	return 0;
}
#endif

void __init mx2ads_image_exit(void);

/*initialize*/
int __init
mx2ads_image_init(void)
{
	struct capture_device *dev = &mx2ads_capture_device;

	memset(dev, 0, sizeof (struct capture_device));

	init_waitqueue_head(&dev->new_video_frame);

	pr_debug("MX2ADS CMOS Image Sensor Driver initialising\n");

#ifdef CONFIG_PM
	mx2ads_image_pmdev = pm_register(PM_UNKNOWN_DEV, PM_SYS_UNKNOWN,
					 mx2ads_image_pm_callback);
	if (!mx2ads_image_pmdev)
		printk(KERN_WARNING
		       "MX2 Camera: failed to initialize power management\n");
#endif

	/* Inititialise CSI resources: GPIO, Clocks, etc... */
	if (csi_generic_initialise(dev)) {
		printk(KERN_ERR
		       "MX2 Camera: Failed to initialise CSI module\n");
		mx2ads_image_exit();
		return -ENODEV;
	}

	/* Initialise board-specific resources */
	mx2ads_misoc0343_initialise();

	if (misoc0343_detect()) {
		printk(KERN_ERR
		       "MX2 Camera: Failed to detect CMOS Sensor Micron MI-SOC-0343 compatible\n");
		mx2ads_image_exit();
		return -ENODEV;
	}

	if (config_a_device(dev)) {
		mx2ads_image_exit();
		return -ENODEV;
	}

	if (v4l2_register_device((struct v4l2_device *) dev) != 0) {
		printk(KERN_ERR "MX2 Camera: Couldn't register v4l2 device.\n");
		mx2ads_image_exit();
		return -ENODEV;
	}

	dev->is_registered = 1;

	pr_debug("MX2ADS CMOS Image Sensor Driver initialised\n");

	return 0;
}

/*deinitialize*/
void __init
mx2ads_image_exit(void)
{
	struct capture_device *dev = &mx2ads_capture_device;

	pr_debug("MX2ADS CMOS Cleanup\n");

	capture_close(dev);

	/* Cleanup CSI resources: GPIO, Clocks, etc... */
	csi_generic_cleanup(dev);

	if (dev->is_registered) {
		v4l2_unregister_device((struct v4l2_device *) dev);
	}
#ifdef CONFIG_PM
	if (mx2ads_image_pmdev)
		pm_unregister(mx2ads_image_pmdev);
#endif

	memset(dev, 0, sizeof (struct capture_device));

	return;
}

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("MX2ADS Image Sensor Driver");
MODULE_LICENSE("GPL");

module_init(mx2ads_image_init);
module_exit(mx2ads_image_exit);
