/*
 *  Implementation of ATA/DMA/UDMA functions specific for PNX0105
 *  ATA controller. Based on ide-dma.c
 *
 *  Author:	Vitaly Wool
 *  Created:	Sep 6, 2004
 *  Copyright:	MontaVista Software Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */

#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/timer.h>
#include <linux/mm.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/ide.h>
#include <linux/pci.h>

#include <asm/io.h>
#include <asm/irq.h>

#define PRD_BYTES	8
#define PRD_ENTRIES	(PAGE_SIZE / (2 * PRD_BYTES))

/* Uncomment to have a lot of debugging messages */
/* #define PNX0105_ATA_DEBUG */

#ifdef PNX0105_ATA_DEBUG
#define DEBUG 1
#else 
#define DEBUG 0
#endif

static int real_sdma_mode = 0;

int pnx0105_real_sdma_mode(void)
{
	static int num_calls = 0;
	if (num_calls > 1)
		return 1;
	else
		return (num_calls++ > 1);
}

static byte pnx0105_eighty_ninty_three (ide_drive_t *drive)
{
        return ((byte) ((HWIF(drive)->udma_four) &&
                        (drive->id->hw_config & 0x6000)) ? 1 : 0);
}

static int ide_raw_build_sglist (ide_hwif_t *hwif, struct request *rq)
{
	struct scatterlist *sg = hwif->sg_table;
	int nents = 0;
	ide_task_t *args = rq->special;
	unsigned char *virt_addr = rq->buffer;
	int sector_count = rq->nr_sectors;

	pr_debug("%s: hwif %p, sg %p, args %p, virt_addr %p, sector_count %d\n",
                __FUNCTION__, hwif, sg, args, virt_addr, sector_count);
	if (args->command_type == IDE_DRIVE_TASK_RAW_WRITE)
		hwif->sg_dma_direction = PCI_DMA_TODEVICE;
	else
		hwif->sg_dma_direction = PCI_DMA_FROMDEVICE;
	
	if (sector_count > 128) {
		memset(&sg[nents], 0, sizeof(*sg));
		sg[nents].address = virt_addr;
		sg[nents].length = 128  * SECTOR_SIZE;
		nents++;
		virt_addr = virt_addr + (128 * SECTOR_SIZE);
		sector_count -= 128;
	}
	memset(&sg[nents], 0, sizeof(*sg));
	sg[nents].address = virt_addr;
	sg[nents].length =  sector_count  * SECTOR_SIZE;
	nents++;
   
	return pci_map_sg(0, sg, nents, hwif->sg_dma_direction);
}

static int ide_build_sglist (ide_hwif_t *hwif, struct request *rq)
{
	struct buffer_head *bh;
	struct scatterlist *sg = hwif->sg_table;
	unsigned long lastdataend = ~0UL;
	int nents = 0;

	if (hwif->sg_dma_active)
		BUG();

	if (rq->cmd == READ)
		hwif->sg_dma_direction = PCI_DMA_FROMDEVICE;
	else
		hwif->sg_dma_direction = PCI_DMA_TODEVICE;

	bh = rq->bh;
	do {
		struct scatterlist *sge;

		/*
		 * continue segment from before?
		 */
		if (bh_phys(bh) == lastdataend) {
			sg[nents - 1].length += bh->b_size;
			lastdataend += bh->b_size;
			continue;
		}

		/*
		 * start new segment
		 */
		if (nents >= PRD_ENTRIES)
			return 0;

		sge = &sg[nents];
		memset(sge, 0, sizeof(*sge));

		if (bh->b_page) {
			sge->page = bh->b_page;
			sge->offset = bh_offset(bh);
		} else {
			if (((unsigned long) bh->b_data) < PAGE_SIZE)
				BUG();

			sge->address = bh->b_data;
		}

		sge->length = bh->b_size;
		lastdataend = bh_phys(bh) + bh->b_size;
		nents++;
	} while ((bh = bh->b_reqnext) != NULL);

	return pci_map_sg(0, sg, nents, hwif->sg_dma_direction);

}

/*
 * ide_build_dmatable() prepares a dma request.
 * Returns 0 if all went okay, returns 1 otherwise.
 * May also be invoked from trm290.c
 */
int ide_build_dmatable (ide_drive_t *drive, ide_dma_action_t func)
{
	unsigned int *table = HWIF(drive)->dmatable_cpu;
	unsigned int count = 0;
	int i;
	struct scatterlist *sg;

	if (HWGROUP(drive)->rq->cmd == IDE_DRIVE_TASKFILE) {
		HWIF(drive)->sg_nents = i = ide_raw_build_sglist(HWIF(drive), HWGROUP(drive)->rq);
	}
	else {
		HWIF(drive)->sg_nents = i = ide_build_sglist(HWIF(drive), HWGROUP(drive)->rq);
	}

	if (!i)
		return 0;

	sg = HWIF(drive)->sg_table;
	while (i) {
		u32 cur_addr;
		u32 cur_len;
		cur_addr = sg_dma_address(sg);
		cur_len = sg_dma_len(sg);

		/*
		 * Fill in the dma table, without crossing any 64kB boundaries.
		 */

		while (cur_len) {
			u32 xcount, bcount = 0x1000 - (cur_addr & 0xfff);
			
			if (count++ >= PRD_ENTRIES)
				BUG();

			if (bcount > cur_len)
				bcount = cur_len;
			*table++ = cpu_to_le32(cur_addr);
			xcount = bcount & 0xffff;
			if (xcount == 0x0000) {
				/* 
				 * Most chipsets correctly interpret a length
				 * of 0x0000 as 64KB, but at least one
				 * (e.g. CS5530) misinterprets it as zero (!).
				 * So here we break the 64KB entry into two
				 * 32KB entries instead.
				 */
				if (count++ >= PRD_ENTRIES)
					goto use_pio_instead;

				*table++ = cpu_to_le32(0x8000);
				*table++ = cpu_to_le32(cur_addr + 0x8000);
				xcount = 0x8000;
			}
			xcount = (xcount >> 2) -1;
			*table++ = cpu_to_le32(xcount);
			cur_addr += bcount;
			cur_len -= bcount;
		}

		sg++;
		i--;
	}

	if (count) {
		*--table |= cpu_to_le32(0x80000000);
		table = HWIF(drive)->dmatable_cpu;
		return count;
	}
use_pio_instead:
	printk("%s: reverting drive %s to PIO\n", __FUNCTION__, drive->name);
	pci_unmap_sg(0,
		     HWIF(drive)->sg_table,
		     HWIF(drive)->sg_nents,
		     HWIF(drive)->sg_dma_direction);
	HWIF(drive)->sg_dma_active = 0;
	return 0; /* revert to PIO for this request */
}

static int config_drive_for_dma (ide_drive_t *drive)
{
	int config_allows_dma = 1;
	struct hd_driveid *id = drive->id;
	ide_hwif_t *hwif = HWIF(drive);
#if 0
	int speed;
#endif

#if DEBUG
	short *tmp = (short *)id;
	int i;
	if (tmp) {
		for (i = 0; i < 256; i++)
			printk("%02d: %04x\t", i, tmp[i]);
		printk("\n");
	}
#endif

#ifdef CONFIG_IDEDMA_ONLYDISK
	if (drive->media != ide_disk)
		config_allows_dma = 0;
#endif

	hwif->autodma = 1; 
	real_sdma_mode = 1;
	if (id)
		pr_debug("%s: field_valid %p:%x, dma_ultra %x, dma_mword %x, dma_1word %x\n", __FUNCTION__, (void *)&id->field_valid, id->field_valid, id->dma_ultra, id->dma_mword, id->dma_1word);
	if (id && (id->capability & 1) && hwif->autodma && config_allows_dma) {
		/* Enable DMA on any drive that has UltraDMA (mode 6/7/?) enabled */
		if ((id->field_valid & 4) && (pnx0105_eighty_ninty_three(drive)))
			if ((id->dma_ultra & (id->dma_ultra >> 14) & 2)) {
				printk("%s: UltraDMA (mode 6/7/?)\n", __FUNCTION__);
#if 0 /* not yet */
				ide_config_drive_speed(drive, XFER_UDMA_6);
#endif
				return hwif->dmaproc(ide_dma_on, drive);
		}
		/* Enable DMA on any drive that has UltraDMA (mode 3/4/5) enabled */
		if ((id->field_valid & 4) && (pnx0105_eighty_ninty_three(drive)))
			if ((id->dma_ultra & (id->dma_ultra >> 11) & 7)) {
				pr_debug("%s: UltraDMA (mode 3/4/5)\n", __FUNCTION__);
#if 0 /* not yet */
				ide_config_drive_speed(drive, XFER_UDMA_3);
#endif
				return hwif->dmaproc(ide_dma_on, drive);
			}
		/* Enable DMA on any drive that has UltraDMA (mode 0/1/2) enabled */
		if (id->field_valid & 4)	/* UltraDMA */
			if ((id->dma_ultra & (id->dma_ultra >> 8) & 7)) {
				pr_debug("%s: UltraDMA (mode 0/1/2)\n", __FUNCTION__);
#if 0 /* not yet */
				ide_config_drive_speed(drive, XFER_UDMA_0);
#endif
				return hwif->dmaproc(ide_dma_on, drive);
			}
		/* Enable DMA on any drive that has mode2 DMA (multi or single) enabled */
		if (id->field_valid & 2)	/* regular DMA */
			if ((id->dma_mword & 0x404) == 0x404 || (id->dma_1word & 0x404) == 0x404) {
				pr_debug("%s: MDMA\n", __FUNCTION__);
#if 0 /* not yet */
				if (id->dma_mword & 0x4)
					ide_config_drive_speed(drive, XFER_MW_DMA_2);
				else if (id->dma_mword & 0x3)
					ide_config_drive_speed(drive, XFER_MW_DMA_0);
				else
					ide_config_drive_speed(drive, XFER_SW_DMA_0);
#endif
				return hwif->dmaproc(ide_dma_on, drive);
			}
	}
	pr_debug("%s: failed to configure DMA mode; going to PIO mode\n", __FILE__);
	real_sdma_mode = 0;
#if 0 /* not yet */
	if (id-> field_valid & 1)
		speed = (id->eide_pio_modes & 0x03) + 2;
	else
		speed = XFER_PIO_0;
	ide_config_drive_speed(drive, speed);
#endif
	return hwif->dmaproc(ide_dma_on, drive);
}

/*
 * dma_intr() is the handler for disk read/write DMA interrupts
 */
ide_startstop_t ide_dma_intr (ide_drive_t *drive)
{
	int i;
	byte stat, dma_stat;

	dma_stat = HWIF(drive)->dmaproc(ide_dma_end, drive);
	stat = GET_STAT();			/* get drive status */
	if (OK_STAT(stat,DRIVE_READY,drive->bad_wstat|DRQ_STAT)) {
		if (!dma_stat) {
			struct request *rq = HWGROUP(drive)->rq;
			rq = HWGROUP(drive)->rq;
			for (i = rq->nr_sectors; i > 0;) {
				i -= rq->current_nr_sectors;
				ide_end_request(1, HWGROUP(drive));
			}
			return ide_stopped;
		}
		printk("%s: dma_intr: bad DMA status (dma_stat=%x)\n", 
		       drive->name, dma_stat);
	}
	return ide_error(drive, "dma_intr", stat);
}

#ifndef CONFIG_BLK_DEV_IDEDMA_TIMEOUT
/*
 * 1 dmaing, 2 error, 4 intr
 */
static int dma_timer_expiry (ide_drive_t *drive)
{
	byte dma_stat = inb(HWIF(drive)->dma_base + PNX0105_PCCARD_BM_STAT_REG_OFFSET);

	pr_debug("%s: dma_timer_expiry: dma status == 0x%02x\n", drive->name, dma_stat);

#if 0
	HWGROUP(drive)->expiry = NULL;	/* one free ride for now */
#endif

	if (dma_stat & 2) {	/* ERROR */
		byte stat = GET_STAT();
		return ide_error(drive, "dma_timer_expiry", stat);
	}
	if (dma_stat & 1)	/* DMAing */
		return WAIT_CMD;
	return 0;
}
#else /* CONFIG_BLK_DEV_IDEDMA_TIMEOUT */
static ide_startstop_t ide_dma_timeout_revovery (ide_drive_t *drive)
{
	ide_hwgroup_t *hwgroup	= HWGROUP(drive);
	ide_hwif_t *hwif	= HWIF(drive);
	int enable_dma		= drive->using_dma;
	unsigned long flags;
	ide_startstop_t startstop;

	spin_lock_irqsave(&io_request_lock, flags);
	hwgroup->handler = NULL;
	del_timer(&hwgroup->timer);
	spin_unlock_irqrestore(&io_request_lock, flags);

	drive->waiting_for_dma = 0;

	startstop = ide_do_reset(drive);

	if ((enable_dma) && !(drive->using_dma))
		(void) hwif->dmaproc(ide_dma_on, drive);

	return startstop;
}
#endif /* CONFIG_BLK_DEV_IDEDMA_TIMEOUT */

int report_drive_dmaing (ide_drive_t *drive)
{
	struct hd_driveid *id = drive->id;

	if ((id->field_valid & 4) && (pnx0105_eighty_ninty_three(drive)) &&
	    (id->dma_ultra & (id->dma_ultra >> 14) & 3)) {
		if ((id->dma_ultra >> 15) & 1) {
			printk(", UDMA(mode 7)");	/* UDMA BIOS-enabled! */
		} else {
			printk(", UDMA(133)");	/* UDMA BIOS-enabled! */
		}
	} else if ((id->field_valid & 4) && (pnx0105_eighty_ninty_three(drive)) &&
	  	  (id->dma_ultra & (id->dma_ultra >> 11) & 7)) {
		if ((id->dma_ultra >> 13) & 1) {
			printk(", UDMA(100)");	/* UDMA BIOS-enabled! */
		} else if ((id->dma_ultra >> 12) & 1) {
			printk(", UDMA(66)");	/* UDMA BIOS-enabled! */
		} else {
			printk(", UDMA(44)");	/* UDMA BIOS-enabled! */
		}
	} else if ((id->field_valid & 4) &&
		   (id->dma_ultra & (id->dma_ultra >> 8) & 7)) {
		if ((id->dma_ultra >> 10) & 1) {
			printk(", UDMA(33)");	/* UDMA BIOS-enabled! */
		} else if ((id->dma_ultra >> 9) & 1) {
			printk(", UDMA(25)");	/* UDMA BIOS-enabled! */
		} else {
			printk(", UDMA(16)");	/* UDMA BIOS-enabled! */
		}
	} else if (id->field_valid & 4) {
		printk(", (U)DMA");	/* Can be BIOS-enabled! */
	} else {
		printk(", DMA");
	}
	return 1;
}

static void pnx0105_destroy_dmatable (ide_drive_t *drive)
{
	struct scatterlist *sg = HWIF(drive)->sg_table;
	int nents = HWIF(drive)->sg_nents;

	pci_unmap_sg(0, sg, nents, HWIF(drive)->sg_dma_direction);
	HWIF(drive)->sg_dma_active = 0;
}

int pnx0105_dmaproc (ide_dma_action_t func, ide_drive_t *drive)
{
	ide_hwif_t *hwif		= HWIF(drive);
	unsigned long dma_base		= hwif->dma_base;
	byte unit			= (drive->select.b.unit & 0x01);
	unsigned int count, writing = 0, set_high = 1;
	byte dma_stat;

	pr_debug("%s: func %d, drive %p\n", __FUNCTION__, func, drive);

	switch (func) {
		case ide_dma_off:
			printk("%s: DMA disabled\n", drive->name);
		case ide_dma_off_quietly:
			/* 
			 * We are just faking that DMA is disabled.
			 */
			return 0;
		case ide_dma_on:
			drive->using_dma = (func == ide_dma_on);
			if (drive->using_dma) {
				if (!inl(PNX0105_PCCARD_IO_BASE + PNX0105_PCCARD_AUTOMATION_OFFSET))
					outl (0x01, PNX0105_PCCARD_IO_PORT + PNX0105_PCCARD_AUTOMATION_OFFSET); /* enable automation */
			}
			return 0;
		case ide_dma_check:
			return config_drive_for_dma (drive);
		case ide_dma_write:
			writing = 1 << 3;
		case ide_dma_read:
			SELECT_READ_WRITE(hwif,drive,func);
			if (!(count = ide_build_dmatable(drive, func)))
				return 1;	/* try PIO instead of DMA */
			outl(hwif->dmatable_dma, 
			     dma_base + PNX0105_PCCARD_BM_PRD_REG_OFFSET); /* PRD table */
			outb(writing, dma_base);			/* specify r/w */
			outb(inb(dma_base + PNX0105_PCCARD_BM_STAT_REG_OFFSET) | 6, 
			     dma_base + PNX0105_PCCARD_BM_STAT_REG_OFFSET);	/* clear INTR & ERROR flags */
			outb(inb(dma_base + PNX0105_PCCARD_BM_INT_REG_OFFSET) | 6, 
			     dma_base + PNX0105_PCCARD_BM_INT_REG_OFFSET);
			drive->waiting_for_dma = 1;
			if (drive->media != ide_disk)
				return 0;
#ifdef CONFIG_BLK_DEV_IDEDMA_TIMEOUT
			ide_set_handler(drive, &ide_dma_intr, 2*WAIT_CMD, NULL);	/* issue cmd to drive */
#else /* !CONFIG_BLK_DEV_IDEDMA_TIMEOUT */
			ide_set_handler(drive, &ide_dma_intr, WAIT_CMD, dma_timer_expiry);	/* issue cmd to drive */
#endif /* CONFIG_BLK_DEV_IDEDMA_TIMEOUT */
			if ((HWGROUP(drive)->rq->cmd == IDE_DRIVE_TASKFILE) &&
			    (drive->addressing == 1)) {
				ide_task_t *args = HWGROUP(drive)->rq->special;
				OUT_BYTE(args->tfRegister[IDE_COMMAND_OFFSET], IDE_COMMAND_REG);
			} else if (drive->addressing) {
				if (real_sdma_mode)
					outb(!writing ? WIN_READDMA_EXT : WIN_WRITEDMA_EXT, IDE_COMMAND_REG);
				else
					outb(!writing ? WIN_READ_EXT : WIN_WRITE_EXT, IDE_COMMAND_REG);
			} else {
				if (real_sdma_mode)
					OUT_BYTE(!writing ? WIN_READDMA : WIN_WRITEDMA, IDE_COMMAND_REG);
				else
					OUT_BYTE(!writing ? WIN_READ : WIN_WRITE, IDE_COMMAND_REG);
			}
			return HWIF(drive)->dmaproc(ide_dma_begin, drive);
		case ide_dma_begin:
		{
			/* Note that this is done *after* the cmd has
			 * been issued to the drive, as per the BM-IDE spec.
			 * The Promise Ultra33 doesn't work correctly when
			 * we do this part before issuing the drive cmd.
			 */
			outb(inb(dma_base) | 1, dma_base);		/* start DMA */
			return 0;
		}
		case ide_dma_end: /* returns 1 on error, 0 otherwise */
			drive->waiting_for_dma = 0;
			outb(inb(dma_base) & ~1, dma_base);	/* stop DMA */
			dma_stat = inb(dma_base + PNX0105_PCCARD_BM_STAT_REG_OFFSET);		/* get DMA status */
			outb(dma_stat | 6, dma_base + PNX0105_PCCARD_BM_STAT_REG_OFFSET);	/* clear the INTR & ERROR bits */
			dma_stat = inb(dma_base + PNX0105_PCCARD_BM_INT_REG_OFFSET);
			outb(dma_stat | 6, dma_base + PNX0105_PCCARD_BM_INT_REG_OFFSET);
			pnx0105_destroy_dmatable(drive);	/* purge DMA mappings */
			return (dma_stat & 7) != 4 ? (0x10 | dma_stat) : 0;	/* verify good DMA status */
		case ide_dma_test_irq: /* returns 1 if dma irq issued, 0 otherwise */
			dma_stat = inb(dma_base + PNX0105_PCCARD_BM_STAT_REG_OFFSET);
#if 0  /* do not set unless you know what you are doing */
			if (dma_stat & 4) {
				byte stat = GET_STAT();
				outb(dma_base + PNX0105_PCCARD_BM_STAT_REG_OFFSET, dma_stat & 0xE4);
			}
#endif
			return dma_stat & 4 ? 1 : 0;	/* return 1 if INTR asserted */
		case ide_dma_bad_drive:
		case ide_dma_good_drive:
			return 0; 
		case ide_dma_verbose:
			return report_drive_dmaing(drive);
		case ide_dma_timeout:
			// FIXME: Many IDE chipsets do not permit command file register access
			// FIXME: while the bus-master function is still active.
			// FIXME: To prevent deadlock with those chipsets, we must be extremely
			// FIXME: careful here (and in ide_intr() as well) to NOT access any
			// FIXME: registers from the 0x1Fx/0x17x sets before terminating the
			// FIXME: bus-master operation via the bus-master control reg.
			// FIXME: Otherwise, chipset deadlock will occur, and some systems will
			// FIXME: lock up completely!!
#ifdef CONFIG_BLK_DEV_IDEDMA_TIMEOUT
			/*
			 * Have to issue an abort and requeue the request
			 * DMA engine got turned off by a goofy ASIC, and
			 * we have to clean up the mess, and here is as good
			 * as any.  Do it globally for all chipsets.
			 */
			outb(0x00, dma_base);		/* stop DMA */
			dma_stat = inb(dma_base + PNX0105_PCCARD_BM_STAT_REG_OFFSET);	/* get DMA status */
			outb(dma_stat | 6, dma_base + PNX0105_PCCARD_BM_STAT_REG_OFFSET);	/* clear the INTR & ERROR bits */
			printk("%s: %s: Lets do it again!" \
				"stat = 0x%02x, dma_stat = 0x%02x\n",
				drive->name, ide_dmafunc_verbose(func),
				GET_STAT(), dma_stat);

			if (dma_stat & 0xF0)
				return ide_dma_timeout_revovery(drive);

			printk("%s: %s: (restart_request) Lets do it again!" \
				"stat = 0x%02x, dma_stat = 0x%02x\n",
				drive->name, ide_dmafunc_verbose(func),
				GET_STAT(), dma_stat);

			return restart_request(drive);  // BUG: return types do not match!!
//#else
//			return HWGROUP(drive)->handler(drive);
#endif /* CONFIG_BLK_DEV_IDEDMA_TIMEOUT */
		case ide_dma_retune:
		case ide_dma_lostirq:
			printk("ide_dmaproc: chipset supported func only: %d\n", func);
			return 1;
		default:
			printk("ide_dmaproc: unsupported func: %d\n", func);
			return 1;
	}

}

int pnx0105_ide_setup_dma (ide_hwif_t *hwif, unsigned long dma_base, unsigned int num_ports)
{
	printk("    %s: BM-DMA at 0x%04lx-0x%04lx", hwif->name, dma_base, dma_base + 4*(num_ports - 1));
	if (!request_region(dma_base, num_ports, hwif->name)) {
		printk(" -- ERROR, PORT ADDRESSES ALREADY IN USE\n");
		return -1;
	}
	hwif->dma_base = dma_base;
	hwif->dmatable_cpu = consistent_alloc(GFP_KERNEL, PRD_ENTRIES * PRD_BYTES, &hwif->dmatable_dma);

	hwif->sg_table = kmalloc(sizeof(struct scatterlist) * PRD_ENTRIES,
				 GFP_KERNEL);
	if (hwif->sg_table == NULL) {
		consistent_free(hwif->dmatable_cpu, PRD_ENTRIES * PRD_BYTES, hwif->dmatable_dma);
		goto dma_alloc_failure;
	}

	hwif->dmaproc = &pnx0105_dmaproc;

	printk("\n");
	return 0;

dma_alloc_failure:
	printk(" -- ERROR, UNABLE TO ALLOCATE DMA TABLES\n");
	return -1;
}

