/*
 * linux/drivers/ide/pci/cs5535.c       Version 0.1 May 1, 2004
 *
 * Copyright (c) 2003, 2004 Advanced Micro Devices, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * The full GNU General Public License is included in this distribution in the
 * file called COPYING
 * 
 * Documentation:
 *  CS5535 documentation available from Advanced Micro Devices 
 *  http://www.amd.com
 */

#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/mm.h>
#include <linux/ioport.h>
#include <linux/blkdev.h>
#include <linux/hdreg.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/ide.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/msr.h>

#include "ide-timing.h"
#include "ide_modes.h"
#include "cs5535.h"

/*#define DEBUG */

#ifdef DEBUG
#define DPRINTK(fmt,args...) printk("cs5535-ide: " fmt, ##args)
#else
#define DPRINTK(fmt,args...)
#endif

#if defined(DISPLAY_CS5535_TIMINGS) && defined(CONFIG_PROC_FS)
#include <linux/stat.h>
#include <linux/proc_fs.h>

byte cs5535_proc = 0;
extern int (*cs5535_display_info) (char *, char **, off_t, int);	/* ide-proc.c */
static struct pci_dev *bmide_dev;
static u8 w80 = 0;

/* FIXME:  rewrite this? */

static int
cs5535_get_info(char *buffer, char **addr, off_t offset, int count)
{
	char *p = buffer;
	unsigned long bibma = pci_resource_start(bmide_dev, 4);

	u8 c0 = 0;

	/*
	 * at that point bibma+0x2 et bibma+0xa are byte registers
	 * to investigate:
	 */

	c0 = inb_p((u16) bibma + 0x02);

	p += sprintf(p, "\n                                "
		     "AMD/NS 5535 Chipset.\n");

	p += sprintf(p, "--------------- drive0 --------- drive1 ------\n");
	p += sprintf(p, "DMA enabled:    %s              %s\n",
		     (c0 & 0x20) ? "yes" : "no ", (c0 & 0x40) ? "yes" : "no ");

	pci_read_config_byte(bmide_dev, CS5535_CABLE_DETECT, &c0);

	p += sprintf(p, "80 Wire Cable:  %s              %s\n",
		     (c0 & 1) ? "yes" : "no", (c0 & 2) ? "yes" : "no");

	return p - buffer;
}

#endif				/* DISPLAY_CS5535_TIMINGS && CONFIG_PROC_FS */

/* Format I PIO settings. We seperate out cmd and data for safer timings */

static unsigned int cs5535_pio_cmd_timings[5] =
    { 0xF7F4, 0x53F3, 0x13F1, 0x5131, 0x1131 };
static unsigned int cs5535_pio_dta_timings[5] =
    { 0xF7F4, 0xF173, 0x8141, 0x5131, 0x1131 };

static unsigned int cs5535_mwdma_timings[3] =
    { 0x7F0FFFF3, 0x7F035352, 0x7f024241 };

static unsigned int cs5535_udma_timings[5] =
    { 0x7F7436A1, 0x7F733481, 0x7F723261, 0x7F713161, 0x7F703061 };

/* Macros to check if the register is the reset value -  reset value is an
   invalid timing and indicates the register has not been set previously */

#define CS5535_BAD_PIO(timings) ( (timings&~0x80000000)==0x00009172 )
#define CS5535_BAD_DMA(timings) ( (timings & 0x000FFFFF) == 0x00077771 )

static void
cs5535_set_speed(ide_drive_t * drive, u8 speed)
{

	unsigned long reg = 0, dummy;
	int unit = drive->select.b.unit;

	DPRINTK("cs5535_set_speed speed %x\n", speed);

	/* Set the PIO timings */

	if ((speed & XFER_MODE) == XFER_PIO) {
		u8 pioa = speed - XFER_PIO_0;
		u8 piob =
		    ide_get_best_pio_mode(&(HWIF(drive)->drives[!unit]), 255, 4,
					  NULL);
		u8 cmd = pioa < piob ? pioa : piob;

		/* Write the speed of the current drive */

		reg =
		    (cs5535_pio_cmd_timings[cmd] << 16) |
		    cs5535_pio_dta_timings[pioa];
		wrmsr(unit ? ATAC_CH0D1_PIO : ATAC_CH0D0_PIO, reg, 0);

		/* And if nessesary - change the speed of the other drive */
		rdmsr(unit ? ATAC_CH0D0_PIO : ATAC_CH0D1_PIO, reg, dummy);

		if (((reg >> 16) & cs5535_pio_cmd_timings[cmd]) !=
		    cs5535_pio_cmd_timings[cmd]) {
			reg &= 0x0000FFFF;
			reg |= cs5535_pio_cmd_timings[cmd] << 16;
			wrmsr(unit ? ATAC_CH0D0_PIO : ATAC_CH0D1_PIO, reg, 0);
		}

		/* Set bit 31 of the DMA register for PIO format 1 timings */
		rdmsr(unit ? ATAC_CH0D1_DMA : ATAC_CH0D0_DMA, reg, dummy);
		wrmsr(unit ? ATAC_CH0D1_DMA : ATAC_CH0D0_DMA, reg | 0x80000000,
		      0);

	} else {

		rdmsr(unit ? ATAC_CH0D1_DMA : ATAC_CH0D0_DMA, reg, dummy);

		reg &= 0x80000000;	/* Preserve the PIO format bit */

		if (speed >= XFER_UDMA_0 && speed <= XFER_UDMA_7)
			reg |= cs5535_udma_timings[speed - XFER_UDMA_0];
		else if (speed >= XFER_MW_DMA_0 && speed <= XFER_MW_DMA_2)
			reg |= cs5535_mwdma_timings[speed - XFER_MW_DMA_0];
		else
			return;

		wrmsr(unit ? ATAC_CH0D1_DMA : ATAC_CH0D0_DMA, reg, 0);
	}
}

/* Configure the drive to the new speed */

static int
cs5535_set_drive(ide_drive_t * drive, u8 speed)
{

	DPRINTK("cs5535_set_drive speed %x\n", speed);

	if (speed != drive->current_speed)
		ide_config_drive_speed(drive, speed);

	cs5535_set_speed(drive, speed);

	drive->current_speed = speed;
	return 0;
}

static void
cs5535_tuneproc(ide_drive_t * drive, u8 pio)
{
	u8 speed = 0;

	DPRINTK("cs5535_tuneproc(%d) pio %x\n", drive->select.b.unit, pio);

	if (pio == 255)
		speed = ide_find_best_mode(drive, XFER_PIO | XFER_EPIO);
	else if (pio < 100)
		speed = XFER_PIO_0 + min_t(byte, pio, 5);
	else if (pio < 200)
		speed = XFER_MW_DMA_0 + min_t(byte, (pio - 100), 2);
	else
		speed = XFER_UDMA_0 + min_t(byte, (pio - 200), 7);

	cs5535_set_drive(drive, speed);
}

static int
cs5535_config_dma(ide_drive_t * drive)
{

	u8 speed = 0;
	unsigned long map = XFER_MWDMA | XFER_UDMA;
	if (w80)
		map |= XFER_UDMA_66;

	/* Get the best mode for this drive */
	speed = ide_find_best_mode(drive, map);

	if ((speed & XFER_MODE) != XFER_PIO) {
		cs5535_set_drive(drive, speed);

		if (HWIF(drive)->autodma)
			return HWIF(drive)->dmaproc(ide_dma_on, drive);
	}

	return HWIF(drive)->dmaproc(ide_dma_off_quietly, drive);
}

int
cs5535_dmaproc(ide_dma_action_t func, ide_drive_t * drive)
{
	switch (func) {
	case ide_dma_check:
		return cs5535_config_dma(drive);
	default:
		break;
	}
	/* Other cases are done by generic IDE-DMA code. */
	return ide_dmaproc(func, drive);
}

unsigned int __init
init_chipset_cs5535(struct pci_dev *dev, const char *name)
{
	u8 bit;

#if defined(DISPLAY_CS5535_TIMINGS) && defined(CONFIG_PROC_FS)
	if (!cs5535_proc) {
		cs5535_proc = 1;
		bmide_dev = dev;
		cs5535_display_info = &cs5535_get_info;
	}
#endif				/* DISPLAY_CS5535_TIMINGS && CONFIG_PROC_FS */

	pci_read_config_byte(bmide_dev, CS5535_CABLE_DETECT, &bit);
	w80 = bit & 1;		/* Remember if we have a 80 wire cable */
	printk("\n!!W80 = %d", w80);
#if defined(DISPLAY_CS5535_TIMINGS) && defined(DEBUG)
	{
		unsigned long val1, val2, flags;

		spin_lock_irqsave(&ide_lock, flags);
		/* all CPUs (there should only be one CPU with this chipset) */

		rdmsr(ATAC_GLD_MSR_CAP, val1, val2);
		DPRINTK("ATAC_GLD_MSR_CAP=0x%.8X\n", (unsigned int) val1);
		rdmsr(ATAC_GLD_MSR_CONFIG, val1, val2);
		DPRINTK("ATAC_GLD_MSR_CONFIG=0x%.8X\n", (unsigned int) val1);
		rdmsr(ATAC_GLD_MSR_SMI, val1, val2);
		DPRINTK("ATAC_GLD_MSR_SMI=0x%.8X\n", (unsigned int) val1);
		rdmsr(ATAC_GLD_MSR_ERROR, val1, val2);
		DPRINTK("ATAC_GLD_MSR_ERROR=0x%.8X\n", (unsigned int) val1);
		rdmsr(ATAC_GLD_MSR_PM, val1, val2);
		DPRINTK("ATAC_GLD_MSR_PM=0x%.8X\n", (unsigned int) val1);
		rdmsr(ATAC_GLD_MSR_DIAG, val1, val2);
		DPRINTK("ATAC_GLD_MSR_DIAG=0x%.8X\n", (unsigned int) val1);
		rdmsr(ATAC_IO_BAR, val1, val2);
		DPRINTK("ATAC_IO_BAR=0x%.8X.%.8X\n", (unsigned int) val2,
			(unsigned int) val1);
		rdmsr(ATAC_RESET, val1, val2);
		DPRINTK("ATAC_RESET=0x%.8X\n", (unsigned int) val1);
		rdmsr(ATAC_CH0D0_PIO, val1, val2);
		DPRINTK("ATAC_CH0D0_PIO=0x%.8X\n", (unsigned int) val1);
		rdmsr(ATAC_CH0D0_DMA, val1, val2);
		DPRINTK("ATAC_CH0D0_DMA=0x%.8X\n", (unsigned int) val1);
		rdmsr(ATAC_CH0D1_PIO, val1, val2);
		DPRINTK("ATAC_CH0D1_PIO=0x%.8X\n", (unsigned int) val1);
		rdmsr(ATAC_CH0D1_DMA, val1, val2);
		DPRINTK("ATAC_CH0D1_DMA=0x%.8X\n", (unsigned int) val1);
		rdmsr(ATAC_PCI_ABRTERR, val1, val2);
		DPRINTK("ATAC_PCI_ABRTERR=0x%.8X\n", (unsigned int) val1);
	}

	spin_unlock_irqrestore(&ide_lock, flags);
#endif				/* DISPLAY_CS5535_TIMINGS */

	return 0;
}

unsigned int __init
ata66_cs5535(ide_hwif_t * hwif)
{
	u8 bit;

	pci_read_config_byte(hwif->pci_dev, CS5535_CABLE_DETECT, &bit);

	return (bit & 1);
}

/**
 * \ingroup ide config
 * \brief
 *  This gets invoked by the IDE driver once for each channel. It
 *  performs channel-specific pre-initialization before drive probing.
 *
 * \param  ide_hwif_t *hwif
 *         
 */
void __init
init_hwif_cs5535(ide_hwif_t * hwif)
{
	unsigned long dummy;
	u32 timings;
	hwif->autodma = 0;
	unsigned long dfltpio = (cs5535_pio_cmd_timings[0] << 16) |
	    cs5535_pio_dta_timings[0];

	int dmastatus = 0;

	if (!bmide_dev)
		DPRINTK("OOPS - bad timing\n");
	else {
		unsigned long bibma = pci_resource_start(bmide_dev, 4);
		dmastatus = inb_p((u16) bibma + 0x02);
	}

	if (hwif->mate)
		hwif->serialized = hwif->mate->serialized = 1;

	hwif->tuneproc = &cs5535_tuneproc;
	hwif->speedproc = &cs5535_set_drive;

	/* Check the registers, and set up the PIO registers if nessesary */
	/* Note that this will default to whatever the BIOS set them to be */

	/* Drive 0 */

	rdmsr(ATAC_CH0D0_PIO, timings, dummy);
	if (CS5535_BAD_PIO(timings)) {
		/* PIO timings not initialized? */
		wrmsr(ATAC_CH0D0_PIO, dfltpio, 0);

		if (!hwif->drives[0].autotune)
			hwif->drives[0].autotune = 1;
	}

	/* Drive 1 */

	rdmsr(ATAC_CH0D1_PIO, timings, dummy);
	if (CS5535_BAD_PIO(timings)) {
		/* PIO timings not initialized? */
		wrmsr(ATAC_CH0D1_PIO, dfltpio, 0);
		if (!hwif->drives[1].autotune)
			hwif->drives[1].autotune = 1;

	}

	/* We want to take special care not to mess with the settings
	   from the BIOS.  If the DMA registers are already set, just
	   assume those are correct, and go from there */

	rdmsr(ATAC_CH0D0_DMA, timings, dummy);

	if (!CS5535_BAD_DMA(timings)) {
		if (dmastatus & 0x20)
			hwif->dmaproc(ide_dma_on, &(hwif->drives[0]));

		hwif->drives[0].autotune = 2;
	}

	rdmsr(ATAC_CH0D1_DMA, timings, dummy);

	if (!CS5535_BAD_DMA(timings)) {
		if (dmastatus & 0x40)
			hwif->dmaproc(ide_dma_on, &(hwif->drives[1]));

		hwif->drives[1].autotune = 2;
	}
	/* w80 will be 1 if a 80 wire cable was detected */

	hwif->udma_four = w80;

	hwif->dmaproc = &cs5535_dmaproc;
	if (!noautodma)
		hwif->autodma = 1;
}

/**
 *  init_dma_cs5535     -   set up for DMA
 *  @hwif: interface
 *  @dmabase: DMA base address
 *
 *  FIXME: this can go away
 */

/**
 * \ingroup ide config
 * \brief
 * wrapper for call of ide_setup_dma 
 *  
 * \param  ide_hwif_t *hwif
 *         
 */
void __init
init_dma_cs5535(ide_hwif_t * hwif, unsigned long dmabase)
{
	ide_setup_dma(hwif, dmabase, 8);
}
