/*
 * drivers/i2c/i2c-tps65010.c
 *
 * Device driver to the TI TPS65010 Power Management device
 *
 * Author: MontaVista Software, Inc. <source@mvista.com>
 *
 * 2004 (c) MontaVista Software, Inc. This program is free software;
 * you can redistribute it and/or modify it under the terms of the GNU
 * General Public License version 2 as published by the Free Software
 * Foundation.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/miscdevice.h>
#include <linux/fcntl.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/spinlock.h>
#include <linux/sysctl.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/tps65010.h>

struct i2c_driver tps65010_driver;
struct i2c_client *tps65010_i2c_client;

static unsigned short ignore[] = { I2C_CLIENT_END };
static unsigned short normal_addr[] = { TPS65010_I2C_ADDR, I2C_CLIENT_END };
static unsigned short normal_addr_range[] = { 0x48, 0x49,
					I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
	.normal_i2c		= normal_addr,
	.normal_i2c_range	= normal_addr_range,
	.probe			= ignore,
	.probe_range		= ignore,
	.ignore			= ignore,
	.ignore_range		= ignore,
	.force			= ignore,
};

static int tps65010_probe(struct i2c_adapter *adap);
static int tps65010_detach(struct i2c_client *client);
static int tps65010_command(struct i2c_client *client, unsigned int cmd,
			   void *arg);

struct i2c_driver tps65010_driver = {
	.name		= "TPS65010",
	.id		= I2C_DRIVERID_TPS65010,
	.flags		= I2C_DF_NOTIFY,
	.attach_adapter	= tps65010_probe,
	.detach_client	= tps65010_detach,
	.command	= tps65010_command
};

static int
tps65010_attach(struct i2c_adapter *adap, int addr, unsigned short flags,
		int kind)
{
	struct i2c_client *c;

	c = (struct i2c_client *) kmalloc(sizeof (*c), GFP_KERNEL);

	if (!c)
		return -ENOMEM;

	strcpy(c->name, "TPS65010");
	c->id = tps65010_driver.id;
	c->flags = 0;
	c->addr = addr;
	c->adapter = adap;
	c->driver = &tps65010_driver;
	c->data = NULL;

	tps65010_i2c_client = c;

	return i2c_attach_client(c);
}

static int
tps65010_probe(struct i2c_adapter *adap)
{
	return i2c_probe(adap, &addr_data, tps65010_attach);
}

static int
tps65010_detach(struct i2c_client *client)
{
	i2c_detach_client(client);
	kfree(tps65010_i2c_client);
	return 0;
}

/* No commands defined */
static int 
tps65010_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
	return 0;
}

unsigned char 
tps65010_read(u8 subaddr)
{
	u8 buf = 0;

	i2c_master_send(tps65010_i2c_client, &subaddr, 1);
	i2c_master_recv(tps65010_i2c_client, &buf, 1);

	return buf;
}

int
tps65010_write(u8 buf, u8 subaddr)
{
	char tmpbuf[2];

	tmpbuf[0] = subaddr;	/* register number */
	tmpbuf[1] = buf;	/* register data */

	i2c_master_send(tps65010_i2c_client, &tmpbuf[0], 2);
	return 0;
}
static __init int
tps65010_init(void)
{
	int retval = 0;

	retval = i2c_add_driver(&tps65010_driver);

	if (retval < 0) {
		printk(KERN_ERR "i2c-tps65010: i2c_add_driver() failed\n");
		return retval;
	}

	printk(KERN_INFO "TPS65010 driver successfully loaded\n");

	return retval;
}

static __exit void
tps65010_exit(void)
{
	i2c_del_driver(&tps65010_driver);
}

module_init(tps65010_init);
module_exit(tps65010_exit);

EXPORT_SYMBOL(tps65010_read);
EXPORT_SYMBOL(tps65010_write);

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_LICENSE("GPL");
