/*
 * i2c-mpc.c 
 * non-CPM I2C driver for the Motorola MPC85xx, MPC824x, and MPC107
 * Derived from Wolfgang Denk's MPC8240 I2C driver, and IBM 4xx I2C driver
 * 
 * (C) Copyright 2002,2003, 2004 Motorola Inc
 * Authors: Xianghua Xiao, Andy Fleming
 *
 * Maintainer: Kumar Gala (galak@motorola.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/config.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <linux/i2c.h>
#include <linux/i2c-id.h>
#include <linux/list.h>
#include <asm/ocp.h>

#include "i2c-mpc.h"

/* Function Declarations */
static int wait_for_txcomplete(struct mpc_i2c_private* priv);
static int wait_for_bus(struct mpc_i2c_private* priv);
static void init_i2c_registers(struct mpc_i2c_private *priv);

/* Issues an i2c START command */
static inline void mpc_i2c_start(struct mpc_i2c_private *priv)
{
	priv->write(&priv->regs->i2ccr, MPC_I2CCR_MSTA, MPC_I2CCR_MSTA);
}

/* Issues an i2c STOP command */
static inline void mpc_i2c_stop(struct mpc_i2c_private *priv)
{
	priv->write(&priv->regs->i2ccr, 0, MPC_I2CCR_MSTA);
}

/* I2C interrupt handler */
static void mpc_i2c_interrupt(int irq, void *dev_id, struct pt_regs * regs)
{
	struct mpc_i2c_private *priv= (struct mpc_i2c_private *)dev_id;

	priv->write(&priv->regs->i2csr, 0, MPC_I2CSR_MIF);
	priv->pending = 1;

	wake_up_interruptible(&priv->i2c_waitq);
}

/* Write val to addr, changing only the bits specified in mask */
static void mpc_i2c_write32(volatile u8 *addr, u8 val, u8 mask)
{
	u32 tmp;

	tmp = in_le32((u32 *)addr);
	out_le32((u32 *)addr, (u32)((tmp & ~mask) | (val & mask)));
}

static u8 mpc_i2c_read32(volatile u8 *addr)
{
	u8 val = (u8)in_le32((u32 *)addr);
	return val;
}

static void mpc_i2c_write8(volatile u8 *addr, u8 val, u8 mask)
{
	u8 tmp;

	tmp = in_8(addr);

	out_8(addr, (tmp & ~mask) | (val & mask));
}

static u8 mpc_i2c_read8(volatile u8 *addr)
{
	u8 val = in_8(addr);
	return val;
}

/*
 * Whenever we initiate a transaction, the first byte clocked
 * onto the bus after the start condition is the address (7 bit) of the
 * device we want to talk to.  This function manipulates the address specified
 * so that it makes sense to the hardware when written to the I2C peripheral.
 *
 * Note: 10 bit addresses are not supported in this driver
 */
static int mpc_doAddress(struct mpc_i2c_private *priv,
		struct i2c_msg *msg) 
{
	volatile struct mpc_i2c *regs = priv->regs;
	unsigned short flags = msg->flags;
	u8 i2ccrval = MPC_I2CCR_MTX;
	u8 addr = msg->addr << 1;
	int err = 0;

	/* Modify the MTX bit and the R/~W bit based on whether this is
	 * a read or a write */
	if (flags & I2C_M_RD)
		addr |= 1;

	/* Modify the MTX and R/~W bits if this device needs inversion */
	if (flags & I2C_M_REV_DIR_ADDR)
		addr ^= 1;

	if(priv->poll == 0)
		i2ccrval |= MPC_I2CCR_MIEN;

	/* Send the address and R/~W indicator, and enable interrupts */
	priv->write(&regs->i2ccr, i2ccrval, MPC_I2CCR_MTX | MPC_I2CCR_MIEN);

	priv->write(&regs->i2cdr, addr, MPC_I2C_FULLREG);

	/* Wait for the address to finish being sent */
	err = wait_for_txcomplete(priv);

	if(err)
		return err;

	/* Return if we didn't get the ack for the address byte */
	if((priv->read(&regs->i2csr) & MPC_I2CSR_RXAK)) {
		mpc_i2c_stop(priv);

		return -EREMOTEIO;
	}

	return err;
}

/* Reports device capabilities to kernel */
static u32 mpc_func(struct i2c_adapter *adap)
{
	return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL; 
}


/*
 * Wait for master transfer to complete.
 * It puts current process to sleep until we get interrupt or timeout expires.
 * Returns the number of transferred bytes or error (<0)
 */
static int wait_for_txcomplete(struct mpc_i2c_private* priv)
{
	volatile struct mpc_i2c *regs = priv->regs;
	int ret = 0;

	if (priv->poll == 0) {
		/* Interrupt mode */
		wait_queue_t wait;
		init_waitqueue_entry(&wait, current);

		add_wait_queue(&priv->i2c_waitq, &wait);
		set_current_state(TASK_INTERRUPTIBLE);

		if (0 == priv->pending)
			schedule_timeout(MPC_INT_TIMEOUT * HZ);

		set_current_state(TASK_RUNNING);
		remove_wait_queue(&priv->i2c_waitq, &wait);

		/* Make sure we weren't interrupted externally,
		 * and that we didn't time out */
		if (signal_pending(current)){
			printk(KERN_DEBUG "signal pending\n");
			ret = -ERESTARTSYS;
		} else if (0 == priv->pending){
			printk(KERN_DEBUG "timed out status:%x ctl:%x\n",
					priv->read(&regs->i2csr),
					priv->read(&regs->i2ccr));
			ret = -ETIMEDOUT;
		}

		/* Ok, it's safe to clear the pending interrupt now */
		priv->pending = 0;
	} else {
		/* Polling mode */
		unsigned long x = jiffies + priv->adap.timeout * HZ;

		/* Keep going until the interrupt field is set */
		while (!(priv->read(&regs->i2csr) & MPC_I2CSR_MIF)) {
			if (time_after(jiffies, x)) {
				ret = -ETIMEDOUT;
				break;
			}

			if (signal_pending(current)) {
				ret = -ERESTARTSYS;
				break;
			}
			schedule();
		}

		/* If the transmit completed, clear the interrupt bit */
		if ((priv->read(&regs->i2csr) & MPC_I2CSR_MIF)) {
			priv->write(&regs->i2csr, 0, MPC_I2CSR_MIF);
			ret = 0;
		}
	}

	if (ret < 0) {
		mpc_i2c_stop(priv);
	}

	return ret;
}


/* Resets the device, in the case that something happens
 * that shouldn't (such as losing arbitration).  Basically,
 * just for error recovery */
static void mpc_i2c_reset(struct mpc_i2c_private *priv)
{
	volatile struct mpc_i2c *regs = priv->regs;

	/* Put the controller into reset */
	priv->write(&regs->i2ccr, 0, MPC_I2CCR_MEN);

	/* Make it the master */
	priv->write(&regs->i2ccr, MPC_I2CCR_MSTA, MPC_I2CCR_MSTA);

	/* Reenable it */
	priv->write(&regs->i2ccr, MPC_I2CCR_MEN, MPC_I2CCR_MEN);

	/* Dummy read the i2cdr */
	priv->read(&regs->i2cdr);

	/* Put us back in to a wait mode (STOP) */
	priv->write(&regs->i2ccr, 0, MPC_I2CCR_MSTA);

	/* Reinitialize the registers */
	init_i2c_registers(priv);
}


/* Wait for the BUS to free up */
static int wait_for_bus(struct mpc_i2c_private *priv)
{
	wait_queue_t wait;
	volatile struct mpc_i2c *regs = priv->regs;
	int timeout = MPC_TIMEOUT;
	int delaytime = HZ/1000 + 1;

	init_waitqueue_entry(&wait, current);

	add_wait_queue(&priv->i2c_waitq, &wait);
	set_current_state(TASK_INTERRUPTIBLE);

	while((priv->read(&regs->i2csr) & MPC_I2CSR_MBB) && timeout--) {
		/* If we're here, that means something is wrong.
		 * We only support being the master here, and all
		 * transactions should be done.  Reset, and hope
		 * that fixes it */
		mpc_i2c_reset(priv);

		schedule_timeout(delaytime);
	}

	set_current_state(TASK_RUNNING);
	remove_wait_queue(&priv->i2c_waitq, &wait);

	return (timeout <= 0);
}


/* Perform one i2c_msg transfer.  It could be a read, or a write.
 * This function assumes that the address byte has been sent
 * already, and only handles the message contents */
static int mpc_i2c_xfer_bytes(struct mpc_i2c_private* priv, struct i2c_msg* pm)
{
	volatile struct mpc_i2c *regs = priv->regs;
	char *buf = pm->buf;
	int len = pm->len;
	int i;
	int ret = 0;

	if(pm->flags & I2C_M_RD) {
		/* Change to read mode */
		priv->write(&regs->i2ccr, 0, MPC_I2CCR_MTX);

		/* If there is only one byte, we need to TXAK now */
		if(len == 1)
			priv->write(&regs->i2ccr, 0, MPC_I2CCR_TXAK);

		/* Do a dummy read, to initiate the first read */
		priv->read(&regs->i2cdr);
	}

	for(i = 0; i < len; ++i) {
		/* If this is a read, read a byte, otherwise, write a byte */
		if(pm->flags & I2C_M_RD) {
			/* Wait for the previous read to finish */
			ret = wait_for_txcomplete(priv);

			if (ret)
				return ret;

			/* If this is the 2nd to last byte, send
			 * the TXAK signal */
			if(i == len - 2) {
				priv->write(&regs->i2ccr, 0, MPC_I2CCR_TXAK);
			}

			/* If this is the last byte, send STOP */
			if (i == len - 1)
				mpc_i2c_stop(priv);

			buf[i] = priv->read(&regs->i2cdr);

		} else {
			/* Otherwise, it's a write */
			priv->write(&regs->i2cdr, buf[i], MPC_I2C_FULLREG);

			/* Wait for the transaction to complete */
			ret = wait_for_txcomplete(priv);

			if(ret)
				return ret;

			/* Return error if we didn't get the ack */
			if((priv->read(&regs->i2csr) & MPC_I2CSR_RXAK)) {
				printk(KERN_DEBUG "NO ACK!\n");
				ret = -EREMOTEIO;
			}
		}

		/* If there was an error, abort */
		if(ret)
			return ret;
	}

	return ret;
}


/* Transfer a list of messages.  Any combinations of reads and
 * writes are possible, and the limit on the number of messages
 * is determined solely by the kernel.  This will perform a
 * RESTART command between each message. */
static int mpc_xfer(struct i2c_adapter *adap, struct i2c_msg msgs[], int num)
{
	int i;
	struct mpc_i2c_private *priv = adap->data;
	volatile struct mpc_i2c *regs = priv->regs;
	int err = 0;

	if (0 == num)
		return 0;

	/* Wait for the BUS to free */
	err = wait_for_bus(priv);

	/* Exit if the BUS stayed busy */
	if (err) {
		printk(KERN_INFO "mpc-i2c%d: bus was busy\n", priv->index);
		return -EREMOTEIO;
	}

	/* Clear interrupts */
	priv->write(&regs->i2csr, 0, MPC_I2CSR_MIF);

	mpc_i2c_start(priv);

	/* Send each message, chaining them together with repeat STARTs */
	for(i = 0; i < num && !err; ++i) {
		err = mpc_doAddress(priv, &msgs[i]);

		if (err)
			return err;

		err = mpc_i2c_xfer_bytes(priv, &msgs[i]);

		if (err)
			return err;

		/* If this is not the last packet, repeat START */
		if (i < num - 1) {
			priv->write(&regs->i2ccr, MPC_I2CCR_RSTA, MPC_I2CCR_RSTA);
		}
	}

	mpc_i2c_stop(priv);

	return err < 0 ? err : num;
}

static struct i2c_algorithm mpc_algo = {
	.name = "MPC algorithm",
	.id = I2C_ALGO_MPC,
	.master_xfer = mpc_xfer,
	.smbus_xfer = NULL,
	.slave_send = NULL,
	.slave_recv = NULL,
	.algo_control = NULL,
	.functionality  = mpc_func
};


/* Initialize i2c registers to reasonable defaults */
static void init_i2c_registers(struct mpc_i2c_private *priv)
{
	volatile struct mpc_i2c *regs = priv->regs;

	/* Disable interrupts */
	priv->write(&regs->i2ccr, 0, MPC_I2CCR_MIEN);

	/* Set the slave address */
	priv->write(&regs->i2cadr, I2C_MPC_ADDR, MPC_I2CADR_MASK);

	/* Make clock run maximum slow.	*/
	priv->write(&regs->i2cfdr, MPC_I2C_CLKSPD, MPC_I2CFDR_MASK);

	if (priv->flags & MPC_I2C_SEPARATE_DFSRR) {
		/* Set the default digital filter sampling rate. */
		priv->write(&regs->i2cdfsrr, MPC_I2C_DFSR, MPC_I2CDFSRR);
	} else {
		/* The MPC107 has the dfsrr in the same register
		 * as the control register, so we would handle
		 * that case here.
		 */
		panic("mpc-i2c%d: This driver does not support the MPC107, yet!\n",
				priv->index);
	}

	priv->write(&regs->i2ccr, 0, MPC_I2CCR_MSTA);
	priv->write(&regs->i2ccr, MPC_I2CCR_MTX, MPC_I2CCR_MTX);

	/* Clear interrupt bit */
	priv->write(&regs->i2csr, 0, MPC_I2CSR_MIF);
}

/*
 * Scan bus for valid 7-bit addresses (ie things that ACK on 1 byte read)
 */
static void __init mpc_i2c_scan_bus(struct mpc_i2c_private* priv)
{
	int found = 0;
	char dummy;
	struct i2c_msg msg = {
		.buf   = &dummy,
		.len   = sizeof(dummy),
		.flags = I2C_M_RD
	};

	printk(KERN_INFO "mpc-i2c%d: scanning bus...\n", priv->index);

	for (msg.addr = 0; msg.addr < 0x7f; ++msg.addr) {
		if (mpc_xfer(&priv->adap, &msg, 1) == 1) {
			++found;
			printk(KERN_INFO " 0x%02x found\n", msg.addr);
		}
	}

	printk(KERN_INFO "mpc-i2c%d: %d device(s) detected\n", 
			priv->index, found);
}

/* Dumb function that increments the module use count */
static void mpc_i2c_inc_use(struct i2c_adapter *adap)
{
	MOD_INC_USE_COUNT;
}

/* Dumb function that decrements the module use count */
static void mpc_i2c_dec_use(struct i2c_adapter *adap)
{
	MOD_DEC_USE_COUNT;
}


/* Bring the i2c up and running.  This is called by OCP */
static int mpc_i2c_probe(struct ocp_device *ocp)
{
	struct mpc_i2c_private *priv;
	volatile struct mpc_i2c *regs;
	struct i2c_adapter *adap;
	struct ocp_mpc_i2c_data *additions;
	int err;

	/* Allocate the private data structure */
	priv = kmalloc(sizeof(*priv), GFP_KERNEL);

	if (NULL == priv) {
		printk(KERN_ERR "mpc-i2c: could not allocate device data structure\n");
		return -ENOMEM;
	}

	/* Initialize the private data structure */
	memset(priv, 0, sizeof(*priv));
	priv->index = ocp->def->index;

	priv->regs = ioremap(ocp->def->paddr, sizeof(struct mpc_i2c));

	if (NULL == priv->regs) {
		err = -ENXIO;
		goto regs_fail;
	}

	regs = priv->regs;

	priv->irq = ocp->def->irq;

	additions = (struct ocp_mpc_i2c_data *)ocp->def->additions;
	priv->flags = additions->flags;

	/* Set up the write and read functions */
	if (priv->flags & MPC_I2C_32BIT) {
		priv->write = &mpc_i2c_write32;
		priv->read = &mpc_i2c_read32;
	} else {
		priv->write = &mpc_i2c_write8;
		priv->read = &mpc_i2c_read8;
	}

	priv->poll = 0;

	ocp_set_drvdata(ocp, priv);

	init_i2c_registers(priv);

	/* Initialize the wait queue */
	init_waitqueue_head(&priv->i2c_waitq);

	adap = &priv->adap;
	strcpy(adap->name, "MPC I2C");
	adap->data = priv;
	adap->id = I2C_HW_OCP | mpc_algo.id;
	adap->algo = &mpc_algo;
	adap->inc_use = &mpc_i2c_inc_use;
	adap->dec_use = &mpc_i2c_dec_use;
	adap->client_register = NULL;
	adap->client_unregister = NULL;
	adap->timeout = MPC_TIMEOUT;
	adap->retries = MPC_RETRIES;

	if ((err = i2c_add_adapter(adap)) != 0) {
		printk(KERN_CRIT "mpc-i2c%d: failed to register i2c adapter\n", priv->index);
		goto add_fail;
	}

	/* Register the interrupt */
	if ((err = request_irq(priv->irq, mpc_i2c_interrupt, 0, "MPC I2C", priv)) != 0) {
		printk(KERN_ERR "mpc-i2c%d: request_irq %d failed\n",
				priv->index, priv->irq);
		priv->poll = 1;
	} else {
		priv->poll = 0;
	}

	/* Enable the I2C Interface */
	priv->write(&regs->i2ccr, MPC_I2CCR_MEN, MPC_I2CCR_MEN);

	/* Scan the bus */
	mpc_i2c_scan_bus(priv);

	return 0;

	/* Failure handling */
add_fail:
	ocp_set_drvdata(ocp, 0);
	iounmap((void *)priv->regs);
regs_fail:
	kfree(priv);

	return err;
}


/* Stop the i2c device, and unregister it */
static void mpc_i2c_remove(struct ocp_device *ocp)
{
	struct mpc_i2c_private *priv = (struct mpc_i2c_private *)ocp_get_drvdata(ocp);

	if(i2c_del_adapter(&priv->adap) == 0) {
		if(0 == priv->poll) {
			/* Turn off the interrupts */
			priv->write(&priv->regs->i2ccr, 0, MPC_I2CCR_MIEN);

			free_irq(priv->irq, priv);
		}

		iounmap((void *)priv->regs);
		kfree(priv);
	}
}


static struct ocp_device_id mpc_i2c_ids[] =
{
	{ .vendor = OCP_ANY_ID, .function = OCP_FUNC_IIC },
	{ .vendor = OCP_VENDOR_INVALID }
};

static struct ocp_driver mpc_i2c_driver =
{
	.name       = "i2c",
	.id_table   = mpc_i2c_ids,
	.probe      = mpc_i2c_probe,
	.remove     = mpc_i2c_remove,
};

/* Called when the kernel boots, or module is loaded, depending
 * on whether or not it is a module */
static int __init mpc_i2c_init(void)
{
	if (ocp_register_driver(&mpc_i2c_driver) == 0) {
		ocp_unregister_driver(&mpc_i2c_driver);
		return -ENODEV;
	}

	return 0;
}

/* Called when the module exits (or not at all) */
static void __exit mpc_i2c_exit(void)
{
	ocp_unregister_driver(&mpc_i2c_driver);
}

MODULE_LICENSE("GPL");

module_init(mpc_i2c_init);
module_exit(mpc_i2c_exit);
