/******************************************************************************

	i2c-dbmx2.c
	driver for on-chip I2C of Motorola DragonBall MX2
	supports i2c-dev interface

	Author: MontaVista Software, Inc. <source@mvista.com>
	Copyright (c) 2003 MontaVista Software, Inc.

	Based on: i2c-aa-dbmx.c from Motorola Dragonball MX2 ADS BSP
	Copyright (C) 2002 Motorola GSG-China

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*******************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/fs.h>

#include <asm/irq.h>
#include <asm/arch/gpio.h>
#include <asm/io.h>
#include <asm/delay.h>

#include <linux/i2c-id.h>
#include <asm/arch/i2c.h>

#define MODULE_NAME "MX2 I2C ALGO/ADAP"

/*initialization progress indicators*/

static char mx2_i2c_initstate_adap;

static void mx2_i2c_inc(struct i2c_adapter *adap);
static void mx2_i2c_dec(struct i2c_adapter *adap);
static int mx2_i2c_xfer_link(struct i2c_adapter *i2c_adap, struct i2c_msg msgs[],
			int num);
static u32 mx2_i2c_func(struct i2c_adapter *adap);

static int __init dbmx2_i2c_init(void);
static void __init dbmx2_i2c_exit(void);

static struct i2c_algorithm mx2_i2c_algo = {
	.name = MODULE_NAME " algo",
	.id = I2C_ALGO_OCP,
	.master_xfer = mx2_i2c_xfer_link,
	.functionality = mx2_i2c_func
};

static struct i2c_adapter mx2_i2c_adap = {
	.name = MODULE_NAME " adap",
	.id = I2C_ALGO_OCP | I2C_HW_OCP,
	.algo = &mx2_i2c_algo,
	.inc_use = mx2_i2c_inc,
	.dec_use = mx2_i2c_dec
};

/*send/receive data*/
static int
mx2_i2c_xfer_link(struct i2c_adapter *i2c_adap, struct i2c_msg msgs[], int num)
{
	return mx2_i2c_xfer(msgs, num);
}

/*report type of I2C*/
static u32
mx2_i2c_func(struct i2c_adapter *adap)
{
#ifdef MX2_I2C_HIGH_SPEED
	return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL | I2C_FUNC_HIGH_SPEED;
#else
	return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL;
#endif
}

static void
mx2_i2c_inc(struct i2c_adapter *adap)
{
	MOD_INC_USE_COUNT;
}

static void
mx2_i2c_dec(struct i2c_adapter *adap)
{
	MOD_DEC_USE_COUNT;
}

/*initialize I2C*/
static int __init
dbmx2_i2c_init(void)
{
	int tmp;

	pr_debug("ver. 1.0 built on %s %s\n", __TIME__, __DATE__);

	if (!mx2_initstate_i2c) {
		printk(KERN_ERR MODULE_NAME ": I2C function is unavailable\n");
		return -ENODEV;
	}

	/* add the I2C adapter/algorithm driver to the linux kernel */
	tmp = i2c_add_adapter(&mx2_i2c_adap);
	if (tmp) {
		printk(KERN_ERR MODULE_NAME ": failed to add adapter\n");
		dbmx2_i2c_exit();
		return tmp;
	}
	mx2_i2c_initstate_adap = 1;

	return 0;
}

/*deinitialize*/
static void __init
dbmx2_i2c_exit(void)
{
	if (mx2_i2c_initstate_adap)
		i2c_del_adapter(&mx2_i2c_adap);
}

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("DBMX2 I2C driver 1.0");
MODULE_LICENSE("GPL");
module_init(dbmx2_i2c_init);
module_exit(dbmx2_i2c_exit);
