/******************************************************************************

	i2c-dbmx1.c
	driver for on-chip I2C of Motorola DragonBall MX1
	compatible with i2c-dev interface

	Author: MontaVista Software, Inc. <source@mvista.com>
	Copyright (c) 2003 MontaVista Software, Inc.

	Original code: i2c-aa.c from Motorola Dragonball MX1 ADS BSP 0.3.4
	Copyright 2002, 2003 Motorola, Inc. All Rights Reserved.

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
	
	Modifications:
	Nov 2003 - ver 1.1, MontaVista Software, Inc: removed an unbalanced enable-irq

*******************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/ioport.h>

#include <linux/i2c.h>
#include <linux/i2c-id.h>

#include <asm/arch/platform.h>
#include <asm/irq.h>
#include <asm/arch/mx1ads-gpio.h>
#include <asm/io.h>

/*for convenient access to I2C registers*/
#define mx1i2c_iadr_in inl(IO_ADDRESS((I2C_BASE+I2C_IADR)))
#define mx1i2c_ifdr_in inl(IO_ADDRESS((I2C_BASE+I2C_IFDR)))
#define mx1i2c_i2cr_in inl(IO_ADDRESS((I2C_BASE+I2C_I2CR)))
#define mx1i2c_i2sr_in inl(IO_ADDRESS((I2C_BASE+I2C_I2SR)))
#define mx1i2c_i2dr_in inl(IO_ADDRESS((I2C_BASE+I2C_I2DR)))

#define mx1i2c_iadr_out(x) outl(x,IO_ADDRESS((I2C_BASE+I2C_IADR)))
#define mx1i2c_ifdr_out(x) outl(x,IO_ADDRESS((I2C_BASE+I2C_IFDR)))
#define mx1i2c_i2cr_out(x) outl(x,IO_ADDRESS((I2C_BASE+I2C_I2CR)))
#define mx1i2c_i2sr_out(x) outl(x,IO_ADDRESS((I2C_BASE+I2C_I2SR)))
#define mx1i2c_i2dr_out(x) outl(x,IO_ADDRESS((I2C_BASE+I2C_I2DR)))

/*control register bit definitions*/
#define MX1I2C_ENABLE (1<<7)
#define MX1I2C_INTEN (1<<6)
#define MX1I2C_MASTER (1<<5)
#define MX1I2C_TRANSMIT (1<<4)
#define MX1I2C_NOACK (1<<3)
#define MX1I2C_REPSTART (1<<2)

/*status register bit definitions*/
#define MX1I2C_DATAREADY (1<<7)
#define MX1I2C_ADDRASSLA (1<<6)
#define MX1I2C_BUSBUSY (1<<5)
#define MX1I2C_LOSTARB (1<<4)
#define MX1I2C_SLATRANS (1<<2)
#define MX1I2C_INTPEND (1<<1)
#define MX1I2C_ACKRCVD 1

/*
 * I2C clock = system clock / 960  (exactly 100k for 96M BCLK)
 * defined in table 29-4(29.5.2) of DBMX1 user's manual
 */
#define MX1I2C_DEFAULT_FREQ 		0x17
#define MX1I2C_TIMEOUT HZ

#undef MX1I2C_WAIT_FOR_BUS /*enable bus waiting code from old i2c-aa for debugging */

DECLARE_WAIT_QUEUE_HEAD(mx1i2c_wait);

/*initialization progress indicators*/
static int mx1i2c_initstate_i2c;
static int mx1i2c_initstate_gpio;
static int mx1i2c_initstate_adap;
static int mx1i2c_initstate_irq;

#ifdef MX1I2C_WAIT_FOR_BUS
static int mx1i2c_wait_for_bus_busy(void);
static void mx1i2c_wait_for_bus_idle(void);
#endif

static void mx1i2c_isr(int irq, void *dev_id, struct pt_regs *reg);
static int mx1i2c_wait_for_end_of_tx(void);

static void mx1i2c_inc(struct i2c_adapter *adap);
static void mx1i2c_dec(struct i2c_adapter *adap);
static int mx1i2c_xfer(struct i2c_adapter *i2c_adap, struct i2c_msg msgs[],
		       int num);
static u32 mx1i2c_func(struct i2c_adapter *adap);

int __init dbmx1_i2c_init(void);
void __init dbmx1_i2c_exit(void);

static struct i2c_algorithm mx1i2c_algo = {
	.name = "DBMX1 I2C algo",
	.id = I2C_ALGO_OCP,
	.master_xfer = mx1i2c_xfer,
	.functionality = mx1i2c_func
};

static struct i2c_adapter mx1i2c_adap = {
	.name = "DBMX1 I2C adap",
	.id = I2C_ALGO_OCP | I2C_HW_OCP,
	.algo = &mx1i2c_algo,
	.inc_use = mx1i2c_inc,
	.dec_use = mx1i2c_dec,
};

#ifdef MX1I2C_WAIT_FOR_BUS
/* Waits for I2C bus to become busy (start signal detected).
   Returns 0 if arbitration OK, 1 if arbitration lost. */
static int
mx1i2c_wait_for_bus_busy(void)
{
	unsigned long int val;

	while (!((val = mx1i2c_i2sr_in) & MX1I2C_BUSBUSY)) {
		if (val & MX1I2C_LOSTARB) {
			return 1;
		}
	}
	return 0;
}

/* Waits for I2C bus to become idle (stop signal detected) */
static void
mx1i2c_wait_for_bus_idle(void)
{
	while (mx1i2c_i2sr_in & MX1I2C_BUSBUSY) ;
}
#endif

/* wait for end of transfer*/
static int
mx1i2c_wait_for_end_of_tx(void)
{
	unsigned long tx_status;
	wait_queue_t __wait;

	tx_status = mx1i2c_i2sr_in;
	if (tx_status & MX1I2C_INTPEND) {
		mx1i2c_i2sr_out(tx_status & ~MX1I2C_INTPEND);
		return 0;
	}

	init_waitqueue_entry(&__wait, current);
	add_wait_queue(&mx1i2c_wait, &__wait);
	set_current_state(TASK_UNINTERRUPTIBLE);

	mx1i2c_i2cr_out(mx1i2c_i2cr_in | MX1I2C_INTEN);	/*enable interrupt */

	schedule_timeout(MX1I2C_TIMEOUT);
	current->state = TASK_RUNNING;
	remove_wait_queue(&mx1i2c_wait, &__wait);

	mx1i2c_i2cr_out(mx1i2c_i2cr_in & ~(MX1I2C_INTEN));	/*disable interrupt */

	tx_status = mx1i2c_i2sr_in;
	if (tx_status & MX1I2C_INTPEND) {
		mx1i2c_i2sr_out(tx_status & ~MX1I2C_INTPEND);
		return 0;
	}

	printk(KERN_ERR "I2C-DBMX1 ERROR: I2C transfer timeout\n");
	return -1;		/*timeout happened */
}

/*initialize I2C*/
int __init
dbmx1_i2c_init(void)
{
	int tmp;
	mx1i2c_initstate_i2c = 0;
	mx1i2c_initstate_gpio = 0;
	mx1i2c_initstate_adap = 0;
	mx1i2c_initstate_irq = 0;

	printk(KERN_INFO "DBMX1 I2C Driver ver. 1.1\n");
	tmp = (int) request_region(IO_ADDRESS(I2C_BASE), 0x14, "dbmx1_i2c");
	if (!tmp) {
		printk(KERN_ERR "I2C-DBMX1 ERROR: I2C is already in use\n");
		dbmx1_i2c_exit();
		return -1;
	}
	mx1i2c_initstate_i2c = 1;
	/*
	 * port pins for I2C
	 * PA15 : I2C_DATA
	 * PA16 : I2C_CLK
	 */
	tmp =
	    mx1_register_gpios(PORT_A, 0x00018000, PRIMARY | TRISTATE | OUTPUT);
	if (tmp < 0) {
		printk(KERN_ERR
		       "I2C-DBMX1 ERROR: PORT_A mask 0x18000 is already in use\n");
		dbmx1_i2c_exit();
		return tmp;
	}
	mx1i2c_initstate_gpio = 1;

	/* add the I2C adapter/algorithm driver to the linux kernel */
	tmp = i2c_add_adapter(&mx1i2c_adap);
	if (tmp) {
		printk(KERN_ERR "I2C-DBMX1 ERROR: failed to add adapter\n");
		dbmx1_i2c_exit();
		return tmp;
	}
	mx1i2c_initstate_adap = 1;

	tmp =
	    request_irq(IRQ_I2C, mx1i2c_isr, SA_INTERRUPT, "dbmx1_i2c",
			"i2c_bus");
	if (tmp) {
		printk(KERN_ERR "I2C-DBMX1 ERROR: failed to request irq\n");
		dbmx1_i2c_exit();
		return tmp;
	}
	mx1i2c_initstate_irq = 1;

	return 0;
}

/*deinitialize*/
void __init
dbmx1_i2c_exit(void)
{
	if (mx1i2c_initstate_irq)
		free_irq(IRQ_I2C, "i2c_bus");
	if (mx1i2c_initstate_adap)
		i2c_del_adapter(&mx1i2c_adap);
	if (mx1i2c_initstate_gpio)
		mx1_unregister_gpios(PORT_A, 0x00018000);
	if (mx1i2c_initstate_i2c)
		release_region(IO_ADDRESS(I2C_BASE), 0x14);
}

/*send/receive data*/
static int
mx1i2c_xfer(struct i2c_adapter *i2c_adap, struct i2c_msg msgs[], int num)
{
	int i, j, count = 0;
	unsigned long int addr;

#ifdef MX1I2C_WAIT_FOR_BUS
      start:
#endif

	/*due to I2C hardware bugs we cannot turn off and on
	   master mode without full initialization */
	mx1i2c_i2cr_out(0);
	mx1i2c_ifdr_out(MX1I2C_DEFAULT_FREQ);
	mx1i2c_i2cr_out(MX1I2C_ENABLE);

	mx1i2c_i2cr_out(mx1i2c_i2cr_in | MX1I2C_MASTER);	/*send start signal */

#ifdef MX1I2C_WAIT_FOR_BUS
	if (mx1i2c_wait_for_bus_busy()) {
		/*arbitration lost */
		mx1i2c_i2cr_out(mx1i2c_i2cr_in & ~MX1I2C_MASTER);	/*send stop signal */
		mx1i2c_wait_for_bus_idle();
		goto start;
	}
#endif

	for (i = 0; i < num; i++) {	/* deal with  message i */
		if (msgs[i].len > 0) {	/*sanity check - message length */
			/*prepare address */
			addr = (msgs[i].addr << 1);
			if (msgs[i].flags & I2C_M_RD)
				addr |= 1;
			if (msgs[i].flags & I2C_M_REV_DIR_ADDR)
				addr ^= 1;

			mx1i2c_i2cr_out(mx1i2c_i2cr_in & ~MX1I2C_NOACK);	/*turn on ACK */
			/*transmit I2C address */
			mx1i2c_i2cr_out(mx1i2c_i2cr_in | MX1I2C_TRANSMIT);
			mx1i2c_i2dr_out(addr);
			mx1i2c_wait_for_end_of_tx();

			if (msgs[i].flags & I2C_M_RD) {

				/*transmit data */

				mx1i2c_i2cr_out(mx1i2c_i2cr_in &
						~MX1I2C_TRANSMIT);

				/*if second before last txfer in the last msg - disable ACK */
				if ((msgs[i].len == 1) && (i == num - 1))
					mx1i2c_i2cr_out(mx1i2c_i2cr_in |
							MX1I2C_NOACK);

				msgs[i].buf[0] = mx1i2c_i2dr_in;	/*dummy read to start clock */

				if (msgs[i].len > 2)
					for (j = 0; j < msgs[i].len - 2; j++) {
						mx1i2c_wait_for_end_of_tx();
						msgs[i].buf[j] = mx1i2c_i2dr_in;
					}

				mx1i2c_wait_for_end_of_tx();

				/*second before last byte, unless length is 1 byte */
				if (msgs[i].len > 1) {
					if (i == num - 1)
						mx1i2c_i2cr_out(mx1i2c_i2cr_in | MX1I2C_NOACK);	/*unack in the last msg */
					msgs[i].buf[msgs[i].len - 2] =
					    mx1i2c_i2dr_in;

					mx1i2c_wait_for_end_of_tx();
				}

				if (i == num - 1) {	/*the last message done, stop the bus */
					mx1i2c_i2cr_out(mx1i2c_i2cr_in & ~MX1I2C_MASTER);	/*send stop signal */

#ifdef MX1I2C_WAIT_FOR_BUS
					mx1i2c_wait_for_bus_idle();
#endif
				} else
					mx1i2c_i2cr_out(mx1i2c_i2cr_in | MX1I2C_REPSTART);	/*not the last message, repeat start */

				/* read the last byte */
				msgs[i].buf[msgs[i].len - 1] = mx1i2c_i2dr_in;

			} /* end of if(READ) */
			else {	/* write data */

				for (j = 0; j < msgs[i].len; j++) {
					mx1i2c_i2dr_out(msgs[i].buf[j]);
					mx1i2c_wait_for_end_of_tx();
				}

				if (i == num - 1) {	/*the last message done, stop the bus */

					/*I2C hardware bug workaround - dummy repeat start.
					   Without this code the I2C will not send out data againg */
					mx1i2c_i2cr_out(mx1i2c_i2cr_in |
							MX1I2C_REPSTART);
					mx1i2c_i2dr_out(0);
					mx1i2c_wait_for_end_of_tx();
					/*end of hardware bug workaround */

					mx1i2c_i2cr_out(mx1i2c_i2cr_in & ~MX1I2C_MASTER);	/*send stop signal */
#ifdef MX1I2C_WAIT_FOR_BUS
					mx1i2c_wait_for_bus_idle();
#endif
				} else
					mx1i2c_i2cr_out(mx1i2c_i2cr_in | MX1I2C_REPSTART);	/*not the last message, repeat start */

			}	/* end of else(WRITE) */
			count += msgs[i].len;
		}

	}			/* for */

	/* Clear the I2CR to disable I2C module */
	mx1i2c_i2cr_out(0);
	return count;
}

/*interrupt handler*/
static void
mx1i2c_isr(int irq, void *dev_id, struct pt_regs *reg)
{
	mx1i2c_i2cr_out(mx1i2c_i2cr_in & ~(MX1I2C_INTEN));	/*disable interrupt */
	wake_up(&mx1i2c_wait);
	return;
}

/*report type of I2C*/
static u32
mx1i2c_func(struct i2c_adapter *adap)
{
	return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL;
}

static void
mx1i2c_inc(struct i2c_adapter *adap)
{
	MOD_INC_USE_COUNT;
}

static void
mx1i2c_dec(struct i2c_adapter *adap)
{
	MOD_DEC_USE_COUNT;
}

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("DBMX1 I2C driver 1.1");
MODULE_LICENSE("GPL");
module_init(dbmx1_i2c_init);
module_exit(dbmx1_i2c_exit);
