/* 

drivers/i2c/i2c-adap-mv64360.c 

TWSI (i2c) adapter driver for MV64360 Two-Wire Serial 
Interface adapter

Copyright 2003 Artesyn Communication Products, LLC

Author: Tim Montgomery <timm@artesyncp.com>
Derived mainly from the Artesyn i2c adapter driver 

Also based on:
i2c-adap-ibm_ocp.c, i2c-elecktor.c
Copyright 2000-2002 MontaVista Software Inc.
Copyright (C) 1995-97 Simon G. Vogl
              1998-99 Hans Berglund

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>

#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/i2c-id.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/slab.h>

#include <linux/pci.h>

#include <asm/irq.h>
#include <asm/io.h>
#include <asm/mv64360.h>
#include <asm/semaphore.h>

#include "i2c-mv64360.h"
#define IIC_VER "2003.07.31"

/* --- setting states on the bus with the right timing: --------------- */

#define IIC_DEFAULT_TIMEOUT (3*HZ)
#define IIC_DEFAULT_RETRY    3

/* constant - TWSI integral with MV64360 */

#define MV64360_TWSI_IRQ 37

static struct i2c_adapter *mv64360_i2c;

static int general_call_in_progress = 0;

static void
 mv64360_i2c_handler(int this_irq, void *i2c, struct pt_regs *regs);

#ifdef CONFIG_KATANA
extern int katana_get_proc(void);
extern u_char katana_i2c_initialized;
#endif

/******************************************************************************
 * I2C (TWSI) Hardware Section
 ******************************************************************************/

/****************************************************************************
*
* mv64360_i2c_hardware_init - initialize the i2c hardware
*
* This function initializes the MV64360 TWSI (i2c) hardware
*
* RETURNS: 0 on success, -1 on failure
*/

int
mv64360_i2c_hardware_init(struct i2c_adapter *adap)
{				/* which i2c adapter */
	MV64360_TWSI_DEVICE *dev = (MV64360_TWSI_DEVICE *) adap->data;
	int control;

	/* reset the interface */
	mv_reg_write(dev->base_addr, MV64360_TWSI_SOFT_RESET, 0);

   /********************************************************************** 
    *   Set clock division
    *
    * the TWSI clock rate is set according to the following equation:
    *
    *                       SysClk Frequency
    *  SCL Frequency =   ---------------------
    *                     10 (M + 1) * 2^(N+1)
    *
    * The default value of M and N are four, yeilding an SCL frequency of
    *
    *  for 133Mhz SysClk:  83.3 KHz
    *
    *  for 100Mhz SysClk:  62.5 Khz
    * 
    *  M and N can be adjusted in order to obtain a max SCL rate of 100Khz.
    *
    **/

	/* set M to 8, N to 3 for ~100Khz rate @ 133Mhz system clock */
	mv_reg_write(dev->base_addr, MV64360_TWSI_BAUD_RATE, ((8 << 3) | 3));

	/* Clear slave address */
	mv_reg_write(dev->base_addr, MV64360_TWSI_SLAVE_ADDR, 0);

	/* Clear slave address */
	mv_reg_write(dev->base_addr, MV64360_TWSI_EXT_SLAVE_ADDR, 0);

	/* Enable bus and bring it to a known state */
	mv_reg_write(dev->base_addr, MV64360_TWSI_CONTROL,
		(MV64360_TWSI_CONTROL_TWSI_EN | MV64360_TWSI_CONTROL_STOP));

	/* TWSI controller clears stop bit once the stop condition has been 
	 * issued on the bus.
	 */

	do {
		control = mv_reg_read(dev->base_addr, MV64360_TWSI_CONTROL);
	}
	while ((control & MV64360_TWSI_CONTROL_STOP) ==
		MV64360_TWSI_CONTROL_STOP);

	if (dev->interrupt_mode) {
		/* install interrupt handler */
		if (request_irq(MV64360_TWSI_IRQ, mv64360_i2c_handler,
				SA_INTERRUPT,
				"MV64360 Two-Wire Serial Interface",
				adap) < 0) {
			printk(KERN_ERR "Failed to register TWSI IRQ\n");
			return (-1);
		} else {
			mv_set_reg_bits(dev->base_addr, MV64360_TWSI_CONTROL,
				MV64360_TWSI_CONTROL_INT_EN);
			printk("Enabled TWSI at IRQ %d\n", MV64360_TWSI_IRQ);
		}
	} else {
		printk("Running TWSI in polled mode\n");
	}

	/* initialize master transfer synchronization semaphore */
	sema_init(&dev->master_sem, 1);
	down(&dev->master_sem);

	pr_debug("mv64360_i2c_hardware_init: Initialized IIC on MV64360\n");

	dev->slave_enabled = 0;
	dev->slave_addr_mode = 7;	/* default slave address mode to 7bit */
	dev->initialized = 1;

	return (0);
}

/****************************************************************************
*
* i2c_analyze_status - analyze status code
*
* This function decodes the TWSI status code into an appropriate message 
*
* RETURNS: N/A
*/

static void
i2c_analyze_status(int status)
{				/* status code */

	switch (status) {
	case 0x00:
		printk(KERN_INFO "I2C: bus error\n");
		break;

	case 0x08:
		printk(KERN_INFO "I2C: start condition transmitted\n");
		break;

	case 0x10:
		printk(KERN_INFO "I2C: repeat start condition transmitted\n");
		break;

	case 0x18:
		printk(KERN_INFO
			"I2C: address + write bit transmitted, ack received\n");
		break;

	case 0x20:
		printk(KERN_INFO
			"I2C: address + write bit transmitted, ack not received\n");
		break;

	case 0x28:
		printk(KERN_INFO
			"I2C: master transmitted data byte, ack received\n");
		break;

	case 0x30:
		printk(KERN_INFO
			"I2C: master transmitted data byte, ack not received\n");
		break;

	case 0x38:
		printk(KERN_INFO
			"I2C: master lost arbitration during address or transfer\n");
		break;

	case 0x40:
		printk(KERN_INFO
			"I2C: address + read bit transmitted, ack recieved\n");
		break;

	case 0x48:
		printk(KERN_INFO
			"I2C: address + read bit transmitted, ack not recieved\n");
		break;

	case 0x50:
		printk(KERN_INFO
			"I2C: master received read data, ack transmitted\n");
		break;

	case 0x58:
		printk(KERN_INFO
			"I2C: master received read data, ack not transmitted\n");
		break;

	case 0x60:
		printk(KERN_INFO
			"I2C: slave received slave address, ack transmitted\n");
		break;

	case 0x68:
		printk(KERN_INFO
			"I2C: master lost arbitration during address transmit,\n"
			" address is targeted to slave (write access), ack transmitted\n");
		break;

	case 0x70:
		printk(KERN_INFO
			"I2C: general call received, ack transmitted\n");
		break;

	case 0x78:
		printk(KERN_INFO
			"I2C: master lost arbitration during address transmit,\n"
			" general call address received, ack transmitted\n");
		break;

	case 0x80:
		printk(KERN_INFO
			"I2C: slave received write data after receiving slave address,\n"
			" ack transmitted\n");
		break;

	case 0x88:
		printk(KERN_INFO
			"I2C: slave received write data after receiving slave address,\n"
			" ack not transmitted\n");
		break;

	case 0x90:
		printk(KERN_INFO
			"I2C: slave received write data after receiving general call,\n"
			" ack transmitted\n");
		break;

	case 0x98:
		printk(KERN_INFO
			"I2C: slave received write data after receiving general call,\n"
			" ack not transmitted\n");
		break;

	case 0xa0:
		printk(KERN_INFO
			"I2C: slave received stop or repeated start condition\n");
		break;

	case 0xa8:
		printk(KERN_INFO
			"I2C: slave received address + read bit, ack transmitted\n");
		break;

	case 0xb0:
		printk(KERN_INFO
			"I2C: master lost arbitration during address transmit,\n"
			" address is targeted to the slave (read access),"
			" ack transmitted\n");
		break;

	case 0xb8:
		printk(KERN_INFO
			"I2C: slave transmitted read data, ack received\n");
		break;

	case 0xc0:
		printk(KERN_INFO
			"I2C: slave transmitted read data, ack not received\n");
		break;

	case 0xc8:
		printk(KERN_INFO
			"I2C: slave transmitted last read byte, ack received\n");
		break;

	case 0xd0:
		printk(KERN_INFO
			"I2C: second address + write bit transmitted, ack received\n");
		break;

	case 0xd8:
		printk(KERN_INFO
			"I2C: second address + write bit transmitted, ack not received\n");
		break;

	case 0xe0:
		printk(KERN_INFO
			"I2C: second address + read bit transmitted, ack received\n");
		break;

	case 0xe8:
		printk(KERN_INFO
			"I2C: second address + read bit transmitted, ack not received\n");
		break;

	case 0xf8:
		printk(KERN_INFO "I2C: no relevant status\n");
		break;

	default:
		printk(KERN_ERR "I2C: unknown status code 0x%x\n", status);
		break;
	}
}

/****************************************************************************
*
* mv64360_i2c_master_7bit_addr_set - set i2c address for master transaction
*
* This function writes the 7bit address or first part of a 10bit address out
* on the i2c bus
*
* RETURNS: 0 on success, -1 on failure 
*/

static int
mv64360_i2c_master_7bit_addr_set(MV64360_TWSI_DEVICE * dev)
{				/* which TWSI device */
	struct i2c_msg *msg = dev->msg;
	unsigned short flags = msg->flags;
	unsigned char addr;

	if ((flags & I2C_M_TEN)) {
		pr_debug("\nusing 10bit addr: %x\n", msg->addr);

		/* a ten bit address */

		addr = ((msg->addr & 0x0300) >> 7) | 0xf0 |
			(flags & I2C_M_RD ? 0x01 : 0x00);

		/* write address to data register */

		mv_reg_write(dev->base_addr, MV64360_TWSI_DATA, addr);

		pr_debug(" Ah ");

	} else {

		pr_debug("using 7bit addr: %x\n", msg->addr);

		/* normal 7bit address */

		addr = (msg->addr << 1);

		if (flags & I2C_M_RD) {
			addr |= 0x01;
		}

		if (flags & I2C_M_REV_DIR_ADDR) {
			/* toggle with XOR */

			addr ^= 0x01;
		}

		mv_reg_write(dev->base_addr, MV64360_TWSI_DATA, addr);

		pr_debug(" A ");

	}

	return (0);
}

/****************************************************************************
*
* mv64360_i2c_master_wr_byte - write a byte of data on the i2c bus 
*
* This function is the low-level byte-write primitive for writing data to the
* i2c bus 
*
* RETURNS: N/A
*/

static void
mv64360_i2c_master_wr_byte(MV64360_TWSI_DEVICE * dev)
{
	pr_debug("D ");

	/* write the next byte from the i2c message buffer */

	mv_reg_write(dev->base_addr, MV64360_TWSI_DATA,
		dev->msg->buf[dev->bytes_txfrd++]);
}

/****************************************************************************
*
* mv64360_i2c_master_2nd_addr_byte - write 2nd byte of data on the i2c bus 
*
* This function writes the second byte of data out on the i2c bus; this data
* can be either actual i2c data, or the second byte of a 10bit address.
*
* RETURNS: 0 on success
*/

static int
mv64360_i2c_master_2nd_addr_byte(MV64360_TWSI_DEVICE * dev,	/* which TWSI */
	u32 * ctrl)
{				/* TWSI ctrl reg ptr */
	struct i2c_msg *msg = dev->msg;
	unsigned short flags = msg->flags;

	if ((flags & I2C_M_TEN)) {
		/* put low eight bits of 10 bit address */

		mv_reg_write(dev->base_addr, MV64360_TWSI_DATA,
			(msg->addr & 0xff));
	} else {
		/* set up read */

		if (flags & I2C_M_RD) {
			if (msg->len == 1) {
				/* if we are reading one byte, don't ack */

				*ctrl &= ~(MV64360_TWSI_CONTROL_ACK);
			} else {
				*ctrl |= MV64360_TWSI_CONTROL_ACK;
			}
		} else {
			/* do write */

			mv64360_i2c_master_wr_byte(dev);
		}
	}
	return (0);
}

/****************************************************************************
*
* mv64360_i2c_master_wr - write to i2c bus
*
* This function is the high-level write to the i2c bus
*
* RETURNS: 0 on success
*/

static int
mv64360_i2c_master_wr(MV64360_TWSI_DEVICE * dev,	/* which TWSI device */
	int *message_complete)
{				/* is the message complete */
	if (dev->bytes_txfrd >= dev->bytes_to_txfr) {
		*message_complete = TRUE;
	} else {
		mv64360_i2c_master_wr_byte(dev);
		*message_complete = FALSE;
	}
	return (0);
}

/****************************************************************************
*
* mv64360_i2c_master_rd - read from i2c bus
*
* This function is the high-level read from the i2c bus
*
* RETURNS: 0 on success
*/

static int
mv64360_i2c_master_rd(MV64360_TWSI_DEVICE * dev,	/* which TWSI device */
	u32 * ctrl)
{				/* TWSI ctrl reg ptr */
	dev->msg->buf[dev->bytes_txfrd++] =
		(unsigned char) mv_reg_read(dev->base_addr, MV64360_TWSI_DATA);

	if ((dev->bytes_txfrd + 1) >= dev->bytes_to_txfr) {
		*ctrl &= ~(MV64360_TWSI_CONTROL_ACK);
	}
	return (0);
}

/****************************************************************************
*
* mv64360_i2c_master_rd_noack - read from i2c bus w/o ack
*
* This function is the high-level read from the i2c bus, without an 
* acknowledge (signals end of read)
*
* RETURNS: 0 on success
*/

static int
mv64360_i2c_master_rd_noack(MV64360_TWSI_DEVICE * dev)
{				/* which TWSI device */
	dev->msg->buf[dev->bytes_txfrd++] =
		(unsigned char) mv_reg_read(dev->base_addr, MV64360_TWSI_DATA);

	return (0);
}


/****************************************************************************
*
* mv64360_i2c_master_rd_setup - setup master read
*
* This function prepares to receive the first byte in a master read
*
* RETURNS: 0 on success
*/

static int
mv64360_i2c_master_rd_setup(MV64360_TWSI_DEVICE * dev,	/* which TWSI device */
	u32 * ctrl)
{				/* TWSI ctrl reg ptr */
	if (dev->bytes_to_txfr == 1) {
		/* if we are reading one byte, don't ack */

		*ctrl &= ~(MV64360_TWSI_CONTROL_ACK);
	} else {
		*ctrl |= MV64360_TWSI_CONTROL_ACK;
	}
	return (0);
}

/****************************************************************************
*
* mv64360_i2c_slave_rd_data_send - sends data to a slave read
*
* This function sends slave data to the i2c bus.  If a callback fcn 
* has been installed, it calls that function to obtain the data from
* higher level code, otherwise, a failure is returned
*
* RETURNS: 0 on success, -1 on failure
*/

static int
mv64360_i2c_slave_rd_data_send(struct i2c_adapter *adap)
{				/* which i2c adapter */
	MV64360_TWSI_DEVICE *dev = (MV64360_TWSI_DEVICE *) adap->data;
	int retval;
	u8 data;

	/* if a callback is installed, call it */

	if (adap->algo->slave_send) {
		retval = adap->algo->slave_send(adap, &data,
			dev->slave_byte_num);

		if (retval == -1) {
			retval = -1;
		} else {
			retval = 0;
		}

	} else {		/* otherwise, failure */

		retval = -1;
	}

	/* 
	 * write data byte to the controller - if there was an error with the 
	 * user callback, the data written will be junk 
	 */

	mv_reg_write(dev->base_addr, MV64360_TWSI_DATA, data);

	dev->slave_byte_num++;

	return (retval);
}

/****************************************************************************
*
* mv64360_slave_wr_data_recv - receives data from a slave write
*
* This function receives slave data from the i2c bus.  If a callback fcn 
* has been installed, it calls that function to pass the data on to higher
* level code, otherwise, a failure is returned
*
* RETURNS: 0 on success, -1 on failure
*/

static int
mv64360_i2c_slave_wr_data_recv(struct i2c_adapter *adap)
{				/* which i2c adapter */
	MV64360_TWSI_DEVICE *dev = (MV64360_TWSI_DEVICE *) adap->data;
	int retval;
	u8 data;

	data = (u8) mv_reg_read(dev->base_addr, MV64360_TWSI_DATA);

	if (adap->algo->slave_recv) {
		retval = adap->algo->slave_recv(adap, &data,
			dev->slave_byte_num);
	} else {
		retval = -1;
	}

	dev->slave_byte_num++;

	return (retval);
}

/****************************************************************************
*
* mv64360_i2c_general_call_data_recv - handles i2c general call
*
* This function returns a failure if the general call is received.  This
* driver has no mechanism to callback on a general call condition
*
* RETURNS: -1
*/

static int
mv64360_i2c_general_call_data_recv(struct i2c_adapter *adap)
{				/* which i2c adapter */
	MV64360_TWSI_DEVICE *dev = (MV64360_TWSI_DEVICE *) adap->data;
	u8 data;

	data = (u8) mv_reg_read(dev->base_addr, MV64360_TWSI_DATA);

	/* we have no mechanism for a general call callback at this time */

	printk("I2C General Call Received: 0x%02x\n", data);

	dev->slave_byte_num++;

	return (-1);
}

/****************************************************************************
*
* mv64360_i2c_handler - i2c interrupt handler
*
* This function handles i2c interrupts
*
* RETURNS: N/A
*/

static void
mv64360_i2c_handler(int this_irq,	/* our interrupt vector */
	void *i2c,		/* our i2c device */
	struct pt_regs *regs)
{				/* pt_regs pointer */
	struct i2c_adapter *adap = (struct i2c_adapter *) i2c;
	MV64360_TWSI_DEVICE *dev = (MV64360_TWSI_DEVICE *) adap->data;

	int current_status, message_complete = FALSE, master_sem_give = FALSE;

	u32 ctrl, dummy, i2c_status;

	/* read control register */

	ctrl = mv_reg_read(dev->base_addr, MV64360_TWSI_CONTROL);

	/* read status register */

	i2c_status = mv_reg_read(dev->base_addr, MV64360_TWSI_STATUS);

	/* check IFLG for status code change */

	if ((ctrl & MV64360_TWSI_CONTROL_INT_FLAG) ==
		MV64360_TWSI_CONTROL_INT_FLAG) {
		/* we recieved a status change; process */

		switch (i2c_status) {
		case MV64360_TWSI_STATUS_BUS_ERROR:

			current_status = -1;
			printk("TWSI bus error\n");
			dev->bus_error = TRUE;
			dev->slave_enabled = FALSE;
			master_sem_give = TRUE;
			ctrl &= ~(MV64360_TWSI_CONTROL_TWSI_EN |
				MV64360_TWSI_CONTROL_INT_EN |
				MV64360_TWSI_CONTROL_ACK);
			break;

			/* Write first (and possibly only) part of I2C address */

		case MV64360_TWSI_STATUS_START_COND_TX_OK:	/* STATE 0x08 */

			current_status = mv64360_i2c_master_7bit_addr_set(dev);
			break;


		case MV64360_TWSI_STATUS_RPT_START_COND_TX_OK:	/* STATE 0x10 */

			current_status = mv64360_i2c_master_7bit_addr_set(dev);
			break;

		case MV64360_TWSI_STATUS_ADDR_WR_TX_OK_ACK:	/* STATE 0x18 */

			current_status =
				mv64360_i2c_master_2nd_addr_byte(dev, &ctrl);
			break;

			/* Address + write bit transmitted, not acknowledged */

		case MV64360_TWSI_STATUS_ADDR_WR_TX_OK_NOACK:	/* STATE 0x20 */

			current_status = -1;
			message_complete = TRUE;
			master_sem_give = TRUE;
			break;

			/* Write data to I2C device */

		case MV64360_TWSI_STATUS_MASTER_TX_DATA_ACK:	/* STATE 0x28 */

			current_status =
				mv64360_i2c_master_wr(dev, &message_complete);
			if (message_complete) {
				master_sem_give = TRUE;
			}
			break;

			/* Master transmitted data byte, not acknowledged */

		case MV64360_TWSI_STATUS_MASTER_TX_DATA_NOACK:	/* STATE 0x30 */

			current_status = -1;
			message_complete = TRUE;
			master_sem_give = TRUE;
			break;

			/* Master lost arbitration during address or data transfer */

		case MV64360_TWSI_STATUS_MASTER_LOST_ARB:	/* STATE 0x38 */

			current_status = -1;
			if (dev->slave_enabled) {
				ctrl |= MV64360_TWSI_CONTROL_ACK;
			}
			master_sem_give = TRUE;
			break;

			/*
			 * Write second part of 10-bit RX address or set up first data
			 * byte read
			 */

		case MV64360_TWSI_STATUS_ADDR_RD_TX_OK_ACK:	/* STATE 0x40 */

			current_status =
				mv64360_i2c_master_2nd_addr_byte(dev, &ctrl);
			break;

			/* Address + read bit transmitted, not acknowledged */

		case MV64360_TWSI_STATUS_ADDR_RD_TX_OK_NOACK:	/* STATE 0x48 */

			current_status = -1;
			message_complete = TRUE;
			master_sem_give = TRUE;
			break;

			/* Master read data with acknowledge */

		case MV64360_TWSI_STATUS_MASTER_RX_DATA_ACK:	/* STATE 0x50 */

			current_status = mv64360_i2c_master_rd(dev, &ctrl);
			break;

			/* Master read data without acknowledge */

		case MV64360_TWSI_STATUS_MASTER_RX_DATA_NOACK:	/* STATE 0x58 */

			current_status = mv64360_i2c_master_rd_noack(dev);
			message_complete = TRUE;
			master_sem_give = TRUE;
			break;


			/* Slave write address received */

		case MV64360_TWSI_STATUS_SLAVE_RX_ADDR_ACK:	/* STATE 0x60 */

			current_status = 0;
			dev->slave_byte_num = 0;
			break;

			/* Master lost arbitration during address transmit, slave write */

		case MV64360_TWSI_STATUS_MASTER_LOST_ARB_ACK:	/* STATE 0x68 */

			current_status = -1;
			master_sem_give = TRUE;
			break;

			/* Slave general call received */

		case MV64360_TWSI_STATUS_GENERAL_CALL_ACK_TX:	/* STATE 0x70 */

			current_status = 0;
			dev->slave_byte_num = 0;
			break;

			/* Master lost arbitration during address transmit, general call */

		case MV64360_TWSI_STATUS_MSTR_LOST_ARB_GEN_CALL_RX:	/* STATE 0x78 */

			current_status = -1;
			master_sem_give = TRUE;
			break;

			/* Slave received write data, acknowledge transmitted */

		case MV64360_TWSI_STATUS_SLV_RX_WR_SLV_ADRS_ACK_TX:	/* STATE 0x80 */

			current_status = mv64360_i2c_slave_wr_data_recv(adap);
			if (current_status != 0) {
				ctrl &= MV64360_TWSI_CONTROL_ACK;
			}
			break;

			/* Slave received write data, acknowledge not transmitted */

		case MV64360_TWSI_STATUS_SLV_RX_WR_SLV_ADRS_NO_ACK:	/* STATE 0x88 */

			dev->slave_byte_num = -1;
			current_status = mv64360_i2c_slave_wr_data_recv(adap);
			ctrl |= MV64360_TWSI_CONTROL_ACK;
			break;


			/* Slave received write data (general call), ack transmitted */

		case MV64360_TWSI_STATUS_SLV_RX_WR_GEN_CALL_ACK_TX:	/* STATE 0x90 */

			current_status =
				mv64360_i2c_general_call_data_recv(adap);
			if (current_status != 0) {
				ctrl &= MV64360_TWSI_CONTROL_ACK;
			}
			break;

			/* Slave received stop or repeated start */

		case MV64360_TWSI_STATUS_SLV_RX_WR_GEN_CALL_NO_ACK:	/* STATE 0x98 */

			dev->slave_byte_num = -1;
			current_status =
				mv64360_i2c_general_call_data_recv(adap);
			ctrl |= MV64360_TWSI_CONTROL_ACK;
			break;


			/* Slave received stop or repeated start */

		case MV64360_TWSI_STATUS_SLV_RX_STOP_RPT_START:	/* STATE 0xA0 */

			dev->slave_byte_num = -1;

			if (dev->prev_state ==
				MV64360_TWSI_STATUS_SLV_RX_WR_GEN_CALL_ACK_TX) {
				current_status =
					mv64360_i2c_general_call_data_recv
					(adap);
				general_call_in_progress = TRUE;
			} else {
				current_status =
					mv64360_i2c_slave_wr_data_recv(adap);
				general_call_in_progress = FALSE;
			}

			ctrl |= MV64360_TWSI_CONTROL_ACK;
			break;


			/* Slave read address received */

		case MV64360_TWSI_STATUS_SLV_RX_ADRS_RD_BIT_ACK_TX:	/* STATE 0xA8 */

			dev->slave_byte_num = 0;
			current_status = 0;
			break;

			/* Master lost arbitration during address transmit, slave read */

		case MV64360_TWSI_STATUS_MSTR_LOST_ARB_SLV_RD:	/* STATE 0xB0 */

			current_status = -1;
			master_sem_give = TRUE;
			break;

			/* Slave transmitted read data, acknowledge received */

		case MV64360_TWSI_STATUS_SLV_TX_RD_ACK_RCVD:	/* STATE 0xB8 */

			if (dev->status == 0) {
				current_status =
					mv64360_i2c_slave_rd_data_send(adap);
			} else {
				current_status = -1;
			}


			/* Slave transmitted read data, acknowledge not received */

		case MV64360_TWSI_STATUS_SLV_TX_RD_NO_ACK:	/* STATE 0xC0 */

			dev->slave_byte_num = -1;
			current_status = mv64360_i2c_slave_rd_data_send(adap);
			break;

			/* Second part of 10-bit TX address ack'd, write first data byte */

		case MV64360_TWSI_STATUS_2ND_ADDR_WR_BIT_ACK:	/* STATE 0xD0 */

			current_status =
				mv64360_i2c_master_wr(dev, &message_complete);
			break;

			/* Second part of 10-bit TX address not acknowledged */

		case MV64360_TWSI_STATUS_2ND_ADDR_WR_BIT_TX_NO_ACK:	/* STATE 0xD8 */

			current_status = -1;
			message_complete = TRUE;
			master_sem_give = TRUE;
			break;


			/* Second part of 10-bit RX address ack'd, write first data byte */

		case MV64360_TWSI_STATUS_2ND_ADDR_RD_BIT_ACK:	/* STATE 0xE0 */

			current_status =
				mv64360_i2c_master_rd_setup(dev, &ctrl);
			break;

			/* Second part of 10-bit RX address not acknowledged */

		case MV64360_TWSI_STATUS_2ND_ADDR_RD_BIT_NO_ACK:	/* STATE 0xE8 */

			current_status = -1;
			message_complete = TRUE;
			master_sem_give = TRUE;
			break;

		default:

			current_status = -1;
			break;
		}

		/* clear the interrupt */

		ctrl &= ~MV64360_TWSI_CONTROL_INT_FLAG;

		/* set STOP bit if all messages have been sent */

		if (message_complete) {
			ctrl |= MV64360_TWSI_CONTROL_STOP;
		}


		if (dev->interrupt_mode) {
			if (master_sem_give) {
				if (dev->slave_enabled) {
					ctrl |= MV64360_TWSI_CONTROL_ACK;
				} else {
					ctrl &= ~MV64360_TWSI_CONTROL_ACK;
				}

				/* signal the transaction to end the master block copy routine */

				up(&dev->master_sem);

			}

			/* update the control register */

			mv_reg_write(dev->base_addr, MV64360_TWSI_CONTROL,
				ctrl);
			dummy = mv_reg_read(dev->base_addr,
				MV64360_TWSI_CONTROL);
		}

		else
		{
			/* update the control register */

			mv_reg_write(dev->base_addr, MV64360_TWSI_CONTROL,
				ctrl);
			dummy = mv_reg_read(dev->base_addr,
				MV64360_TWSI_CONTROL);

			if (master_sem_give) {
				/* signal the transaction end to the polling block copy routine */
				dev->poll_finished = 1;
			}
		}

		dev->status = current_status;
		dev->prev_state = i2c_status;

	}
}

/******************************************************************************
 * I2C Algorithm Section
 ******************************************************************************/

/****************************************************************************
*
* mv64360_i2c_adapter_get - get the i2c_adapter pointer 
*
* This function retreives an i2c adapter ptr for use within kernel space
*
* RETURNS: struct i2c_adapter ptr - pointer to i2c adapter
*/

struct i2c_adapter
*
mv64360_i2c_adapter_get(int adapter)
{				/* which adapter to get */
	/* currently, we support only the first adapter */

	if (adapter == 0) {
		return (mv64360_i2c);
	} else {
		return (NULL);
	}
}

/****************************************************************************
*
* mv64360_i2c_txfr - perform an i2c transfer
*
* This function performs an i2c transfer, processing an array of i2c messages
*
* RETURNS: 0 on success, -1 on failure
*/

int
mv64360_i2c_txfr(struct i2c_adapter *adap,	/* which i2c adapter */
	struct i2c_msg msgs[],	/* ptr to array of messages to send */
	int num_msgs)
{				/* number of messages to send */
	MV64360_TWSI_DEVICE *dev = (MV64360_TWSI_DEVICE *) adap->data;
	int delay, status, curr_msg, retries, timeout;
	u32 ctrl;

	if (num_msgs == 0) {
		/* no messages; just return */

		return (0);
	}

	retries = 0;

	do {
		/* wait for any pending transfers to complete */

		delay = jiffies + adap->timeout;
		timeout = 0;

		do {
			status = mv_reg_read(dev->base_addr,
				MV64360_TWSI_STATUS);

			if (jiffies > delay) {
				pr_debug("[bus busy timeout] last status read: %x\n" " [bus busy timeout] ", status);
				i2c_analyze_status(status);
				printk(KERN_ERR
					"MV64360 TWSI: bus busy timeout\n");
				timeout = 1;
				break;
			}

		} while (status != MV64360_TWSI_STATUS_NO_STATUS);

		/* proceed if we haven't timed out */

		if (timeout != 1) {

			dev->bus_error = 0;

			for (curr_msg = 0; curr_msg < num_msgs; curr_msg++) {
				/* set ACK */
				mv_set_reg_bits(dev->base_addr,
					MV64360_TWSI_CONTROL,
					MV64360_TWSI_CONTROL_ACK);

				dev->msg = &msgs[curr_msg];

				/* check for overflow on message size (msg len is a signed short) */
				if (dev->msg->len < 0) {
					printk(KERN_ERR
						"I2C: message too long\n");
					return (-1);
				}

				dev->bytes_txfrd = 0;
				dev->bytes_to_txfr = dev->msg->len;
				dev->last_msg =
					((curr_msg + 1) >= num_msgs) ? 1 : 0;

				/* delay between previous stop and new start condition  
				 * (MV64360 constraint) 
				 */

				udelay(5);

				dev->poll_finished = 0;

				/* issue start on i2c */

				pr_debug("S ");

				mv_set_reg_bits(dev->base_addr,
					MV64360_TWSI_CONTROL,
					MV64360_TWSI_CONTROL_START);

				/* block until compete */

				if (dev->interrupt_mode) {
					down(&dev->master_sem);
				} else {
					do {
						ctrl = mv_reg_read(dev->
							base_addr,
							MV64360_TWSI_CONTROL);

						if ((ctrl & MV64360_TWSI_CONTROL_INT_FLAG) == MV64360_TWSI_CONTROL_INT_FLAG) {
							mv64360_i2c_handler(0,
								adap, NULL);
						}
					}
					while ((dev->poll_finished == 0)
						&& (dev->status == 0));
				}

				/* check for error */

				if (dev->status == -1) {
					i2c_analyze_status(dev->prev_state);
				}
			}
		}
	} while ((retries++ < adap->retries) &&
		((dev->status == -1) || (timeout == 1)));

	if (retries >= adap->retries) {
		printk(KERN_ERR "I2C: max retries exceeded\n");
		return (-1);
	} else {
		return (curr_msg);
	}
}

/****************************************************************************
*
* i2c_set_local_slave_addr - set i2c local slave address
*
* This function sets the slave address of the MV64360 i2c controller.  The
* local slave address mode should be set prior to calling this function.
*
* RETURNS: 0 on success, -1 if address is out of range
*/

int
i2c_set_local_slave_addr(struct i2c_adapter *adap)
{				/* which i2c adapter */
	MV64360_TWSI_DEVICE *dev = adap->data;
	u32 slave_value, ext_slave_value;

	slave_value = mv_reg_read(dev->base_addr, MV64360_TWSI_SLAVE_ADDR);
	ext_slave_value =
		mv_reg_read(dev->base_addr, MV64360_TWSI_EXT_SLAVE_ADDR);

	if (dev->slave_addr_mode == 7) {
		if ((dev->slave_addr < 0x00) || (dev->slave_addr > 0xFE)) {
			return (-1);
		}

		pr_debug("setting local slave address to: 0x%02x\n",
			dev->slave_addr);

		slave_value =
			((slave_value & ~MV64360_TWSI_SLAVE_ADDR_SADDR_MASK) |
			(dev->slave_addr & MV64360_TWSI_SLAVE_ADDR_SADDR_MASK));
		ext_slave_value =
			ext_slave_value & ~MV64360_TWSI_EXT_SLAVE_ADDR_MASK;
	} else {
		if ((dev->slave_addr < 0x00) || (dev->slave_addr > 0x3FF)) {
			return (-1);
		}
		slave_value = ((slave_value & ~MV64360_TWSI_SLAVE_ADDR_SADDR_MASK) | (0x000000F0) |	/* 10 bit addr flag */
			((dev->slave_addr & 0x300) >> 7));
		ext_slave_value = ((ext_slave_value &
				~MV64360_TWSI_EXT_SLAVE_ADDR_MASK) |
			(dev->slave_addr & MV64360_TWSI_EXT_SLAVE_ADDR_MASK));
	}

	mv_reg_write(dev->base_addr, MV64360_TWSI_SLAVE_ADDR, slave_value);
	mv_reg_write(dev->base_addr, MV64360_TWSI_EXT_SLAVE_ADDR,
		ext_slave_value);

	return (0);

}

/****************************************************************************
*
* i2c_slave_enable - enable i2c slave interface
*
* This function enables the i2c slave interface.  The local slave address and
* local slave address mode should be set before calling this function 
*
* RETURNS: 0 on success, -1 on timeout
*/

int
i2c_slave_enable(struct i2c_adapter *adap)
{				/* which i2c adapter */
	MV64360_TWSI_DEVICE *dev = adap->data;
	unsigned long delay, status, ctrl;

	/* wait for any pending transfers to complete */

	delay = jiffies + (1 * HZ);

	do {
		status = mv_reg_read(dev->base_addr, MV64360_TWSI_STATUS);

		if (jiffies > delay) {
			pr_debug("[bus busy timeout] last status read: %x\n"
				" [bus busy timeout] ", status);
			i2c_analyze_status(status);
			printk(KERN_ERR "MV64360 TWSI: bus busy timeout\n");
			return (-1);
		}

	} while (status != MV64360_TWSI_STATUS_NO_STATUS);

	ctrl = mv_reg_read(dev->base_addr, MV64360_TWSI_STATUS);
	ctrl |= MV64360_TWSI_CONTROL_ACK;
	mv_reg_write(dev->base_addr, MV64360_TWSI_CONTROL_ACK, ctrl);

	dev->slave_enabled = 1;

	printk("TWSI i2c slave interface enabled\nLocal Slave Address: %x\n",
		mv_reg_read(dev->base_addr, MV64360_TWSI_SLAVE_ADDR));

	return (0);
}

/****************************************************************************
*
* i2c_slave_disable - disables i2c slave interface
*
* This function disables the i2c slave interface.
*
* RETURNS: 0 on success, -1 on timeout
*/

int
i2c_slave_disable(struct i2c_adapter *adap)
{				/* which i2c adapter */
	MV64360_TWSI_DEVICE *dev = adap->data;
	unsigned long delay, status, ctrl;

	/* wait for any pending transfers to complete */

	delay = jiffies + (1 * HZ);

	do {
		status = mv_reg_read(dev->base_addr, MV64360_TWSI_STATUS);

		if (jiffies > delay) {
			pr_debug("[bus busy timeout] last status read: %x\n"
				" [bus busy timeout] ", status);
			i2c_analyze_status(status);
			printk(KERN_ERR "MV64360 TWSI: bus busy timeout\n");
			return (-1);
		}

	} while (status != MV64360_TWSI_STATUS_NO_STATUS);

	ctrl = mv_reg_read(dev->base_addr, MV64360_TWSI_STATUS);
	ctrl &= ~MV64360_TWSI_CONTROL_ACK;
	mv_reg_write(dev->base_addr, MV64360_TWSI_CONTROL_ACK, ctrl);

	dev->slave_enabled = 0;

	return (0);
}

/****************************************************************************
*
* i2c_algo_control - ioctl() interface for i2c algorithm
*
* This function implements an ioctl-like interface for controlling i2c
* paramters.
*
* RETURNS: 0 on success, -1 on timeout
*/

static int
i2c_algo_control(struct i2c_adapter *adap,	/* which i2c adapter */
	unsigned int cmd,	/* ioctl command */
	unsigned long arg)
{				/* ioctl argument */
	MV64360_TWSI_DEVICE *dev = adap->data;
	int retval;

	switch (cmd) {
		/* set our local slave address (the address to which we will respond to
		   master transactions on i2c */

	case I2C_LOCAL_SLAVE_ADDR:
		dev->slave_addr = arg;
		retval = i2c_set_local_slave_addr(adap);
		break;

		/* set our local slave address mode (7 or 10 bit) */

	case I2C_LOCAL_SLAVE_ADDR_MODE:
		if ((arg != 7) && (arg != 10)) {
			retval = -EINVAL;
		} else {
			dev->slave_addr_mode = arg;
			retval = 0;
		}
		break;

		/* enable the slave interface */

	case I2C_LOCAL_SLAVE_ENABLE:
		if (arg) {
			retval = i2c_slave_enable(adap);
		} else {
			retval = i2c_slave_disable(adap);
		}
		retval = 0;
		break;

	default:
		return (-EINVAL);
	}

	return (retval);
}

/****************************************************************************
*
* i2c_func - query i2c driver functionality
*
* This function returns a set of flags that describe the i2c functionality
* implmented by this driver.
*
* RETURNS: functionality
*/

static u32
i2c_func(struct i2c_adapter *adap)
{				/* which i2c adapter */
	return ((I2C_FUNC_SMBUS_EMUL | I2C_FUNC_10BIT_ADDR));
}

/* dummy callbacks for debug */

#ifdef DEBUG

static int
slave_send(struct i2c_adapter *adap, char *buf, int arg)
{
	printk("I2C: slave requesting data; no callback installed\n");
	return (0);
}

static int
slave_recv(struct i2c_adapter *adap, char *buf, int arg)
{
	printk("I2C: received slave data; no callback installed\n");
	return (0);
}
#endif


/******************************************************************************
 * algorithm structure
 ******************************************************************************/

struct i2c_algorithm katana_iic_algo = {
	.name		= "Marvell MV64360 I2C algorithm",
	/* we don't have a valid i2c algo number - please update as necessary */
	.id		= I2C_ALGO_EXP,
	.master_xfer	= mv64360_i2c_txfr,
#ifdef DEBUG
	.slave_send 	= slave_send,	/* slave_send    */
	.slave_recv 	= slave_recv,	/* slave_recv    */
#endif
	.algo_control	= i2c_algo_control,	/* ioctl         */
	.functionality	= i2c_func,		/* functionality */
};

/******************************************************************************
 * I2C Adapter Section
 ******************************************************************************/

/****************************************************************************
*
* mv64360_i2c_release - releases irq and memory associated with this adapter
*
* RETURNS: N/A
*/

static void
mv64360_i2c_release(struct i2c_adapter *adap)
{
	disable_irq(MV64360_TWSI_IRQ);

	free_irq(MV64360_TWSI_IRQ, 0);

	kfree(adap->data);

	kfree(adap);
}

/****************************************************************************
*
* mv64360_i2c_reg - dummy function for register device
*
* RETURNS: 0
*/

static int
mv64360_i2c_reg(struct i2c_client *client)
{				/* which i2c client */
	return (0);
}

/****************************************************************************
*
* mv64360_i2c_unreg - dummy function for unregister device
*
* RETURNS: 0
*/

static int
mv64360_i2c_unreg(struct i2c_client *client)
{				/* which i2c client */
	return (0);
}

/****************************************************************************
*
* mv64360_i2c_inc_use - increment module use count
*
* RETURNS: N/A
*/

static void
mv64360_i2c_inc_use(struct i2c_adapter *adap)
{				/* which i2c adapter */
	MOD_INC_USE_COUNT;
}

/****************************************************************************
*
* mv64360_i2c_dec_use - decrement module use count
*
* RETURNS: N/A
*/

static void
mv64360_i2c_dec_use(struct i2c_adapter *adap)
{				/* which i2c adapter */
	MOD_DEC_USE_COUNT;
}

/****************************************************************************
*
* mv64360_i2c_add_bus - connect adapter structure algorithm fcns
*
* RETURNS: 1 on success, or an error flag on failure
*/

int
mv64360_i2c_add_bus(struct i2c_adapter *adap,	/* adapter to be init'd */
	u32 base_addr,		/* base address of TWSI device */
	int int_mode)
{				/* 1 for interrupt mode, 0 for polled */
	MV64360_TWSI_DEVICE *dev;
	static int primary_adapter = 1;
	int err;

	/* local adapter */
	if (primary_adapter) {
		mv64360_i2c = adap;
		primary_adapter = 0;
	}

	adap->data = kmalloc(sizeof(MV64360_TWSI_DEVICE), GFP_KERNEL);

	if (adap->data == NULL) {
		printk(KERN_ERR "%s: couldn't allocate private data\n",
			adap->name);
		return (-ENOMEM);
	}

	memset(adap->data, 0, sizeof(MV64360_TWSI_DEVICE));

	strcpy(adap->name, "MV64360 TWSI (i2c) adapter");

	adap->id |= katana_iic_algo.id;
	adap->algo = &katana_iic_algo;
	adap->timeout = IIC_DEFAULT_TIMEOUT;
	adap->retries = IIC_DEFAULT_RETRY;
	adap->id = I2C_HW_MV64360;
	adap->inc_use = mv64360_i2c_inc_use;
	adap->dec_use = mv64360_i2c_dec_use;
	adap->client_register = mv64360_i2c_reg;
	adap->client_unregister = mv64360_i2c_unreg;

	dev = (MV64360_TWSI_DEVICE *) adap->data;

	dev->initialized = 0;

	dev->base_addr = base_addr;
	dev->interrupt_mode = int_mode;

	err = mv64360_i2c_hardware_init(adap);

	if (err < 0) {
		printk(KERN_ERR "%s: mv64360_i2c_hardware_init() failed\n",
			adap->name);
		kfree(adap->data);
		return (err);
	}

	err = i2c_add_adapter(adap);

	if (err < 0) {
		printk(KERN_ERR "%s: i2c_add_adapter() failed\n", adap->name);
		kfree(adap->data);
		return (err);
	}

	printk("i2c-adap-mv64360.o: hw routines for %s registered.\n",
		adap->name);

	return (1);
}

/****************************************************************************
*
* mv64360_i2c_del_bus - disconnects adapter structure algorithm fcns
*
* RETURNS: 1 on success
*/

int
mv64360_i2c_del_bus(struct i2c_adapter *adap)
{
	int err;

	err = i2c_del_adapter(adap);

	if (err < 0) {
		return (err);
	}

	pr_debug("i2c-algo-iic.o: adapter unregistered: %s\n", adap->name);

	return (1);
}

/****************************************************************************
*
* mv64360_i2c_init - init mv64360_i2c module
*
* This function is the top-level init for the I2C driver.  It starts up the 
* calls through the i2c heirarchy for both local and PCI interfaces
*
* RETURNS: 0 on succes, < 0 on failure
*/

static int __init
mv64360_i2c_init(void)
{
	struct i2c_adapter *adap;
	int err;
	extern ulong katana_get_time(void);
	extern int katana_set_time(ulong);

	printk(KERN_INFO "mv64360_i2c_init: "
		"MV64360 Two Wire Serial Interface Adapter Module\n");

	printk("Adding Local i2c adapter: ");

	adap = kmalloc(sizeof(struct i2c_adapter), GFP_KERNEL);

	if (!adap) {
		return (-ENOMEM);
	}

	memset(adap, 0, sizeof(struct i2c_adapter));

#ifndef CONFIG_KATANA
	/* add bus for local interface in interrupt mode */

	err = mv64360_i2c_add_bus(adap, mv64360_base, 1);
#else

	if (katana_get_proc() == 0) {
		/* add bus for local interface in interrupt mode */

		err = mv64360_i2c_add_bus(adap, mv64360_base, 1);
	} else {
		/* add bus for local interface in interrupt mode */

		err = mv64360_i2c_add_bus(adap, mv64360_base, 0);
	}

	if (err >= 0) {
		katana_i2c_initialized = 1;	/* i2c driver initialized */
	}
#endif

	if (err < 0) {
		printk(KERN_ERR "mv64360_i2c_add_bus() failed\n");
		kfree(adap);
		return (err);
	}

	return (0);
}

/****************************************************************************
*
* mv64360_i2c_exit - clean up mv64360_i2c module
*
* This function is called when the module is unloaded to remove it from
* the i2c layer
*
* RETURNS: N/A
*/

static void
mv64360_i2c_exit(void)
{
	mv64360_i2c_del_bus(mv64360_i2c);
	mv64360_i2c_release(mv64360_i2c);
}

EXPORT_SYMBOL(i2c_set_local_slave_addr);
EXPORT_SYMBOL(i2c_slave_enable);
EXPORT_SYMBOL(i2c_slave_disable);

EXPORT_SYMBOL(mv64360_i2c_adapter_get);
EXPORT_SYMBOL(mv64360_i2c_add_bus);
EXPORT_SYMBOL(mv64360_i2c_del_bus);

/*
 * If modules is NOT defined when this file is compiled, then the MODULE_*
 * macros will resolve to nothing
 */
MODULE_AUTHOR("Artesyn Communication Products, LLC <www.artesyncp.com>");
MODULE_DESCRIPTION("I2C adapter/algo routines for MV64360 TWSI adapter");

MODULE_LICENSE("GPL");

module_init(mv64360_i2c_init);
module_exit(mv64360_i2c_exit);
