/*
 * xilinx_lcd.c
 *
 * Virtex2Pro 2x16 character LCD driver.
 * 
 * Author: MontaVista Software, Inc.
 *         source@mvista.com
 *
 * 2004 (c) MontaVista, Software, Inc.  This file is licensed under the terms
 * of the GNU General Public License version 2.  This program is licensed
 * "as is" without any warranty of any kind, whether express or implied.
 */

/*
 * Based on:
 *   Simple driver for a memory-mapped 44780-style LCD display.
 *   Configured for CMB-VR7701/Rockhopper
 *   2003 (c) MontaVista Software, Inc.
 * Which is in turn based on:
 *   Copyright 2001 Bradley D. LaRonde <brad@ltc.com>
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/delay.h>	/* udelay() */
#include <asm/io.h>

#include "xparameters.h"
#include "xilinx_lcd.h"

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("Xilinx character LCD driver");
MODULE_LICENSE("GPL");

#define LCD_GREETING "MontaVista Linux****************"

#define XILINX_LCD_MINOR 200

#define XILINX_LCD_NAME "xilinx_char_lcd"

#define XILINX_LCD_MAX_POS	32	/* cursor position is 0 to XILINX_LCD_MAX_POS-1 */

static int lcd_open(struct inode *inode, struct file *file);
static ssize_t lcd_write(struct file *file, const char *s, size_t size,
			 loff_t * ppos);
static loff_t lcd_llseek(struct file *file, loff_t pos, int);
static int lcd_release(struct inode *inode, struct file *file);

/* Misc device structures */
static struct file_operations lcd_fops = {
      llseek:lcd_llseek,
      write:lcd_write,
      open:lcd_open,
      release:lcd_release
};

static struct miscdevice lcd_dev = {
	XILINX_LCD_MINOR,
	XILINX_LCD_NAME,
	&lcd_fops
};

/*
 * Driver data
 */
static u32 BaseAddress, SavedAddress;
static int lcd_position;	/* LCD cursor position, '-1' means "invalid" */

/*
 * Private helpers
 */

static inline void
lcd_wait(void)
{
	udelay(200);		/* assuming Fosc > 50kHz */
}

/*
 * "Clear Display" and "Return Home" commands take more time to complete
 * than others.
 */
static inline void
lcd_wait_long(void)
{
	int i;

	for (i = 0; i < 30; i++)
		lcd_wait();
}

static void
lcd_command(unsigned char c)
{
	writew(c | HD44780_INSTR, BaseAddress);
	lcd_wait();
}

static void
lcd_writechar(unsigned char c)
{
	writew(c | HD44780_WR_DATA, BaseAddress);
	lcd_wait();
}

static void
lcd_reset(void)
{
	int i;

	lcd_command(HD44780_8BIT_2LINE);
	lcd_command(HD44780_MODE_INC);
	lcd_command(HD44780_CLR_DISPLAY);
	lcd_wait_long();
	lcd_command(HD44780_DISPLAY_OFF);
	lcd_command(HD44780_CURSOR_LINE);
}

static void
lcd_seek(int pos)
{
	int i;

	if (lcd_position == -1)
		lcd_reset();
	if (pos == lcd_position)
		return;

	lcd_command(HD44780_RET_HOME);
	lcd_wait_long();
	for (i = 0; i < pos; i++) {
		lcd_command(HD44780_RIGHT);
		if (i == XILINX_LCD_MAX_POS / 2) {
			int j;

			for (j = 0; j < (40 - XILINX_LCD_MAX_POS / 2); j++)
				lcd_command(HD44780_RIGHT);
		}
	}
	lcd_position = pos;
}

/*
 * Public functions
 */

static loff_t
lcd_llseek(struct file *file, loff_t pos, int orig)
{
	if (orig == 1)		/* SEEK_CUR */
		pos = file->f_pos + pos;
	else if (orig != 0)	/* SEEK_SET */
		return -EINVAL;

	if (pos < XILINX_LCD_MAX_POS && pos >= 0)
		return pos;
	else
		return -EINVAL;
}

static ssize_t
lcd_write(struct file *file, const char *s, size_t size, loff_t * ppos)
{
	int i;

	/* Position the cursor */
	if (*ppos >= XILINX_LCD_MAX_POS || *ppos < 0)
		return -EFAULT;
	lcd_seek(*ppos);

	/* Print characters */
	for (i = 0; i < size; i++) {
		lcd_writechar(*s++);
		lcd_position++;
		if (lcd_position == XILINX_LCD_MAX_POS / 2) {
			/* skip unvisible positions by writing spaces to them */
			int j;

			for (j = 0; j < (40 - XILINX_LCD_MAX_POS / 2); j++)
				lcd_writechar(' ');
		}
		if (lcd_position == XILINX_LCD_MAX_POS) {
			/* do not go beyond the visible area */
			++i;
			break;
		}
	}

	*ppos = lcd_position;
	return i;
}

static int
lcd_open(struct inode *inode, struct file *file)
{
	MOD_INC_USE_COUNT;
	return 0;
}

static int
lcd_release(struct inode *inode, struct file *file)
{
	MOD_DEC_USE_COUNT;
	lcd_seek(XILINX_LCD_MAX_POS);
	return 0;
}

static int
lcd_init(void)
{
	int err;
	loff_t start = 0;
	static const unsigned long remap_size
	    =
	    XPAR_OPB_LCD_INTERFACE_0_HIGHADDR -
	    XPAR_OPB_LCD_INTERFACE_0_BASEADDR + 1;

	SavedAddress = XPAR_OPB_LCD_INTERFACE_0_BASEADDR;
	BaseAddress = (u32) ioremap(SavedAddress, remap_size);
	printk(KERN_INFO "%s at 0x%08X (%ld bytes) mapped to 0x%08X\n",
	       XILINX_LCD_NAME, SavedAddress, remap_size, BaseAddress);

	lcd_reset();
	lcd_write(NULL, LCD_GREETING, sizeof (LCD_GREETING) - 1, &start);

	/* Register the lcd driver */
	err = misc_register(&lcd_dev);
	if (err != 0) {
		printk(KERN_ERR "%s: Could not register driver (error %d).\n",
		       lcd_dev.name, err);
	}

	return err;
}

static void __exit
lcd_cleanup(void)
{
	lcd_command(HD44780_DISPLAY_OFF);
	iounmap((void *) SavedAddress);
	misc_deregister(&lcd_dev);
}

EXPORT_NO_SYMBOLS;

module_init(lcd_init);
module_exit(lcd_cleanup);
