/******************************************************************************
	ts-mxlads.c
	driver for Motorola DragonBall MXL ADS Touchscreen support

	Will not work on a MX1ADS board, good only for a MXL ADS board.
	This driver uses external MXB7843 touchscreen controller connected to SPI1

	Author: MontaVista Software, Inc. <source@mvista.com>
	Copyright (c) 2003 MontaVista Software, Inc.

	This program is based on spi.c from Motorola DBMX1 ADS BSP Ver. 0.3.4
	Copyright (C) 2001 Motorola Motorola Semiconductors HK Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
	
	Modifications:
	Nov 2003 - ver 1.1, MontaVista Software, Inc: changed to use misc device

********************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <asm/uaccess.h>	/* get_user,copy_to_user */

#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/string.h>
#include <linux/init.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <linux/errno.h>
#include <linux/tqueue.h>
#include <linux/wait.h>
#include <linux/miscdevice.h>

#include <asm/irq.h>
#include <asm/arch/hardware.h>
#include <asm/arch/irqs.h>
#include <asm/arch/mx1ads-gpio.h>

#ifdef CONFIG_PM
#include <linux/pm.h>
#endif

#include "ts-dbmx1.h"

#define MXLTS_PLLCLK2 5

#define MX1TS_SCAN_DELTA 11	/*the minimal difference in reading in order for data to be recorded */

//#define SPI_DEBUG 3
#ifdef SPI_DEBUG
static int debug = SPI_DEBUG;
//MODULE_PARM(debug, "i");
#define debugprintk( level, args... ) if( (level) <= debug ) printk( args )
#else
#define debugprintk( level, args... )
#endif
#define ENTER() debugprintk( 4, "%s: entered\n", __FUNCTION__ )
#define LEAVE() debugprintk( 4, "%s: leaving\n", __FUNCTION__ )

#define MXLTS_X_CONFIG_BYTE 0xD0	/*this byte is sent to get sampling data from X channel */
#define MXLTS_Y_CONFIG_BYTE 0x90	/*this byte is sent to get sampling data from Y channel */
#define MXLTS_CONF_POWER_ON 0	/*power channel on mask */
#define MXLTS_CONF_POWER_OFF 1	/*power channel off mask */

#define SPI_RATE_512  (7<<13)
#define SPI_RATE_256  (6<<13)
#define SPI_RATE_128  (5<<13)
#define SPI_RATE_64   (4<<13)
#define SPI_RATE_32   (3<<13)
#define SPI_RATE_16   (2<<13)
#define SPI_RATE_8    (1<<13)
#define SPI_RATE_4    (0<<13)

#define SPI_RDY_IGNORE ( 0<<11)
#define SPI_RDY_01     ( 1<<11)
#define SPI_RDY_10     ( 2<<11)
#define SPI_RDY_11     ( 3<<11)

#define SPI_SLAVE   	(0<<10)
#define SPI_MASTER 	(1<<10)

#define SPI_EN		(1<<9)

#define SPI_XCH		(1<<8)
#define SPI_NO_XCH      (0<<8)

#define SPI_SS_ACTIVE_LOW  (1<<7)
#define SPI_SS_ACTIVE_HIGH (0<<7)

#define SPI_SS_STAY_LOW_BETWEEN_BURSTS (1<<6)
#define SPI_SS_INSERT_PULSES           (0<<6)
#define SPI_RX_AFTER_BITCOUNT          (1<<6)
#define SPI_RX_AFTER_SS                (0<<6)

#define SPI_PHA0                       (0<<5)
#define SPI_PHA1                       (1<<5)

#define SPI_CLK_ACTIVE_HIGH            (0<<4)
#define SPI_CLK_ACTIVE_LOW             (1<<4)

#define SPI_BASE SPI1_BASE
#define SPI_REG( offset ) IO32_RAW( ( IO_ADDRESS( SPI_BASE ) + (offset) ) )

#define TPNL_IRQ			GPIO_INT_PORTD	/* this is for Port D */

#define TPNL_INTR_MODE			SA_INTERRUPT|SA_SHIRQ
#define TPNL_PEN_UPVALUE		-999

#define SPI_RR_INT		0x00000008
#define SPI_TF_INT		0x00000004
#define SPI_TH_INT		0x00000002
#define SPI_TE_INT		0x00000001

/*this function is used to wake up and terminate a thread*/
#define __mx1ts_wait_event(wq, condition) 					\
do {									\
	wait_queue_t __wait;						\
	init_waitqueue_entry(&__wait, current);				\
	add_wait_queue(&wq, &__wait);					\
		set_current_state(TASK_UNINTERRUPTIBLE);		\
		if (!(condition))						\
		schedule();						\
	current->state = TASK_RUNNING;					\
	remove_wait_queue(&wq, &__wait);				\
} while (0)

#define mx1ts_wait_event(wq, condition) 					\
do {									\
	if (condition)	 						\
		break;							\
	__mx1ts_wait_event(wq, condition);					\
} while (0)

static int spi_open(struct inode *inode, struct file *filp);
static int spi_release(struct inode *inode, struct file *filp);
static int spi_fasync(int fd, struct file *filp, int mode);
static int spi_ioctl(struct inode *inode, struct file *filp,
		     unsigned int cmd, unsigned long arg);
static ssize_t spi_read(struct file *filp, char *buf, size_t count, loff_t * l);
static unsigned int spi_poll(struct file *filp, struct poll_table_struct *wait);

static int getblock(ts_event_t * *block, int size);
static ts_event_t *getdata(void);
static void add_x_y(unsigned x, unsigned y, unsigned flag);

#ifdef CONFIG_PM
static int mxlts_pm_callback(struct pm_dev *pmdev,
			     pm_request_t rqst, void *data);
#endif

static void mxlts_reg_init(void);
static void mxlts_reg_clear(void);
void __init mxlts_exit(void);
int __init mxlts_init(void);

#define TOUCHPAN_BIT 31

#define touchpan_enable_int( )   mx1_gpio_unmask_intr( PORT_D, TOUCHPAN_BIT )
#define touchpan_disable_int( )  mx1_gpio_mask_intr( PORT_D, TOUCHPAN_BIT)
#define touchpan_clear_int(  )   mx1_gpio_clear_intr( PORT_D, TOUCHPAN_BIT )

#define NODATA()	( rptr==wptr )

DECLARE_WAIT_QUEUE_HEAD(spi_wait);
DECLARE_WAIT_QUEUE_HEAD(mxlts_scan_wait);
DECLARE_WAIT_QUEUE_HEAD(mxlts_scan_lock);
static struct fasync_struct *ts_fasync;
static int mxlts_busy = 0;
static unsigned short rptr;
static unsigned short wptr;

static int mxlts_initstate_region = 0;
static int mxlts_initstate_irq = 0;
static int mxlts_initstate_gpio_C_inp = 0;
static int mxlts_initstate_gpio_C_outp = 0;
static int mxlts_initstate_gpio_D_inp = 0;
static int mxlts_initstate_thread = 0;
static int mxlts_initstate_misc_dev = 0;

static DECLARE_COMPLETION(mxlts_scan_thread_exit);

static int mxlts_scan_thread_terminating = 0;

static int mxlts_scan_thread(void *data);

static ts_event_t *mxlts_buffer;
#define BUFLEN	250
#define BUFSIZE (BUFLEN*sizeof(ts_event_t))
#define NEXTI(i)	{i=(i==BUFLEN-1)?0:i+1;}
#define GETNEXTI(i)	((i==BUFLEN-1)?0:i+1)
#define BLKEND(i)	((i)==0)

#define DSIZE()		((wptr<rptr)?(BUFLEN-rptr):(wptr-rptr))

static struct file_operations spi_fops = {
	open:spi_open,
	release:spi_release,
	read:spi_read,
	poll:spi_poll,
	ioctl:spi_ioctl,
	fasync:spi_fasync,
};

#ifdef CONFIG_PM
static struct pm_dev *mxlts_pmdev;
#endif

static void
touchpan_readdev(__u32 * x, __u32 * y, int power_mode)
{
	__u32 x_upper, x_lower, y_upper, y_lower;

	int was_waiting1 = 0;
	int was_waiting2 = 0;

	ENTER();

	SPI_REG(SPI_CONT) &= ~SPI_XCH;
	SPI_REG(SPI_CONT) |= SPI_EN;

/*get X and Y channel data*/
	SPI_REG(SPI_TXD) = MXLTS_X_CONFIG_BYTE | power_mode;
	SPI_REG(SPI_TXD) = 0;
	SPI_REG(SPI_TXD) = MXLTS_Y_CONFIG_BYTE | power_mode;
	SPI_REG(SPI_TXD) = 0;
	SPI_REG(SPI_TXD) = 0;
	/* exchange data */
	SPI_REG(SPI_CONT) |= SPI_XCH;
	/* wait for SPI_TE_INT bit */
	while (!(SPI_REG(SPI_INT) & SPI_TE_INT)) {
		was_waiting1++;
		continue;
	}
	/*    Reset SPI IRQ bit */
	SPI_REG(SPI_INT) &= SPI_TE_INT;
	/* wait for SPI_XCH bit */
	SPI_REG(SPI_CONT) &= ~SPI_XCH;
	x_upper = SPI_REG(SPI_RXD);	/* dummy read */
	x_upper = SPI_REG(SPI_RXD);
	x_lower = SPI_REG(SPI_RXD);
	y_upper = SPI_REG(SPI_RXD);
	y_lower = SPI_REG(SPI_RXD);
	SPI_REG(SPI_CONT) &= (~SPI_EN);

	if ((was_waiting1) || (was_waiting2))
		debugprintk(4, "was waiting1 %d 2 %d\n", was_waiting1,
			    was_waiting2);

	debugprintk(4, "%s: x_upper = %x, x_lower = %x\n", __FUNCTION__,
		    x_upper, x_lower);
	debugprintk(4, "%s: y_upper = %x, y_lower = %x\n", __FUNCTION__,
		    y_upper, y_lower);

	*x = (((x_upper << 5) & 0xFE0) | ((x_lower >> 3) & 0x1F));
	*y = (((y_upper << 5) & 0xFE0) | ((y_lower >> 3) & 0x1F));

	debugprintk(4, "%s: coordinates are (%08X,%08X)\n", __FUNCTION__,
		    *x, *y);
	LEAVE();
}

static void
touchpan_readdata(__u32 * x, __u32 * y)
{
	int i;
#define NUM_OF_SAMPLE  (2)
#define MAX_POS_DIFF   (2)
	ENTER();

	for (*x = *y = i = 0; i < NUM_OF_SAMPLE; i++) {
		__u32 xp, yp;
		touchpan_disable_int();
		touchpan_readdev(&xp, &yp, MXLTS_CONF_POWER_ON);
		touchpan_enable_int();
		if ((xp < 100) || (xp > 5000) || (yp < 100) || (yp > 5000)) {
			*x = TPNL_PEN_UPVALUE;
			*y = TPNL_PEN_UPVALUE;
			return;
		}
		*x += xp;
		*y += yp;
	}

	*x /= NUM_OF_SAMPLE;
	*y /= NUM_OF_SAMPLE;

	LEAVE();

	return;
}

static void
touchpan_isr(int irq, void *dev_id, struct pt_regs *regs)
{
	if (mx1_gpio_intr_status_bit(PORT_D, TOUCHPAN_BIT)) {
		wake_up(&mxlts_scan_lock);
		touchpan_clear_int();
	}
}

static int
mxlts_scan_thread(void *data)
{
	__u32 newX = TPNL_PEN_UPVALUE, newY = TPNL_PEN_UPVALUE;
	__u32 oldX = TPNL_PEN_UPVALUE, oldY = TPNL_PEN_UPVALUE;
	int pen_data_saved = 0;
	daemonize();
	reparent_to_init();
	strcpy(current->comm, "mxlts_scan");

	for (;;) {
		mx1ts_wait_event(mxlts_scan_lock,
				 mxlts_scan_thread_terminating);
		if (mxlts_scan_thread_terminating)
			break;
		/*if we happen to be here, the pen has been down and a reading is available */
	      repeat_scan:touchpan_readdata(&newX, &newY);
		if (((oldX > (newX + MX1TS_SCAN_DELTA))
		     || (oldY > (newY + MX1TS_SCAN_DELTA))
		     || ((oldX + MX1TS_SCAN_DELTA) < newX)
		     || ((oldY + MX1TS_SCAN_DELTA) < newY)
		     || (!pen_data_saved)) && (newX != TPNL_PEN_UPVALUE)) {
			/* report input event */
			add_x_y(newX, newY, PENDOWN);
			oldX = newX;
			oldY = newY;
			pen_data_saved = 1;
			wake_up_interruptible(&spi_wait);

			if (ts_fasync) {
				kill_fasync(&ts_fasync, SIGIO, POLL_IN);
			}
		}

		if (mx1_gpio_get_bit(PORT_D, TOUCHPAN_BIT) == 1) {	/*pen still down */

			sleep_on_timeout(&mxlts_scan_wait, 2);
			if (mxlts_scan_thread_terminating)
				break;
			if (mx1_gpio_get_bit(PORT_D, TOUCHPAN_BIT) == 1)	/*pen is still down */
				goto repeat_scan;
			else {	/*pen already up */
				if (pen_data_saved) {
					add_x_y(0, 0, PENUP);
					oldX = TPNL_PEN_UPVALUE;
					oldY = TPNL_PEN_UPVALUE;
					pen_data_saved = 0;
					wake_up_interruptible(&spi_wait);
					if (ts_fasync) {
						kill_fasync(&ts_fasync,
							    SIGIO, POLL_IN);
					}
				}
				continue;
			}

		} else {	/*pen already up */
			if (pen_data_saved) {
				add_x_y(0, 0, PENUP);
				oldX = TPNL_PEN_UPVALUE;
				oldY = TPNL_PEN_UPVALUE;
				pen_data_saved = 0;
				wake_up_interruptible(&spi_wait);
				if (ts_fasync) {
					kill_fasync(&ts_fasync, SIGIO, POLL_IN);
				}
			}
		}
	}
	complete_and_exit(&mxlts_scan_thread_exit, 0);
	return 0;
}

/**************************************************
Misc functions for buffer
***************************************************/

/******************************************************************************
 * Function Name: add_x_y
 *
 * Input: 		i	:
 * Value Returned:	void	:
 *
 * Description: add pen data to buffer
 *
 *****************************************************************************/
static void
add_x_y(unsigned x, unsigned y, unsigned flag)
{
	ENTER();
	if (GETNEXTI(wptr) == rptr) {
		return;
	}
	debugprintk(2, "%s: Added %d, %d, flag = %X\n",
		    __FUNCTION__, x, y, flag);
	mxlts_buffer[wptr].x = x;
	mxlts_buffer[wptr].y = y;
	mxlts_buffer[wptr].pressure = flag;
	NEXTI(wptr);
	LEAVE();
}

/******************************************************************************
 * Function Name: getblock
 *
 * Description: read a block of 'touch' data from buffer, read count shall less
 *	than size,but shall be as more as possible. the data shall not really copy
 *  to upper layer,untill copy_to_user is invoked.
 *****************************************************************************/
static int
getblock(ts_event_t * *block, int size)
{
	int cnt, rd;
	unsigned long flags;
	ts_event_t *p;

	if (NODATA()) {
		return 0;
	}

	save_flags(flags);
	rd = (DSIZE() * sizeof (ts_event_t) >=
	      size) ? size / sizeof (ts_event_t) : DSIZE();

	for (*block = p = getdata(), cnt = 1; (NULL != p) && (cnt < rd); cnt++) {
		if (rptr == 0)
			break;
		p = getdata();
	}

	return (cnt) * sizeof (ts_event_t);

}

/******************************************************************************
 * Function Name: getdata
 *
 * Input: 		void	:
 * Value Returned:	ts_event_t	: a 'touch' event data format
 *
 * Description: read a 'touch' event from data buffer
 *
 *****************************************************************************/
static ts_event_t *
getdata(void)
{
	ts_event_t *data;

	if (NODATA()) {
		return NULL;
	}

	data = &(mxlts_buffer[rptr]);
	NEXTI(rptr);
	return data;
}

static void
mxlts_reg_init(void)
{
	__u32 x, y;
	ENTER();
	mxlts_reg_clear();
	SPI_REG(SPI_RESET) = 1;
	udelay(10);

	debugprintk(4,
		    "%s: writing %08X to SPI_CONT ( expected 0xE607 ) \n",
		    __FUNCTION__,
		    SPI_RATE_512 | SPI_RDY_IGNORE | SPI_MASTER |
		    SPI_EN | SPI_SS_ACTIVE_HIGH |
		    SPI_SS_INSERT_PULSES | SPI_PHA0 | SPI_CLK_ACTIVE_HIGH | 7);

	SPI_REG(SPI_CONT) =
	    (SPI_RATE_512 | SPI_RDY_IGNORE | SPI_MASTER | SPI_EN |
	     SPI_SS_ACTIVE_HIGH | SPI_SS_INSERT_PULSES | SPI_PHA0 |
	     SPI_CLK_ACTIVE_HIGH | 7);

	wptr = 0;
	rptr = 0;
	touchpan_readdev(&x, &y, MXLTS_CONF_POWER_ON);	/*set up power mode */

	touchpan_clear_int();
	touchpan_enable_int();
}

static void
mxlts_reg_clear(void)
{
	touchpan_disable_int();
	SPI_REG(SPI_CONT) = 0;
}

/******************************************************************************
 * Function Name: spi_open
 *
 * Input: 		inode	:
 * 			filp	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: allocate resource when open the inode
 *
 *****************************************************************************/
static int
spi_open(struct inode *inode, struct file *filp)
{
	MOD_INC_USE_COUNT;
	if (!mxlts_busy)
		mxlts_reg_init();
	mxlts_busy++;
	return 0;
}

/******************************************************************************
 * Function Name: spi_release
 *
 * Input: 		inode	:
 * 			filp	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: release resource when close the inode
 *
 *****************************************************************************/
static int
spi_release(struct inode *inode, struct file *filp)
{
	spi_fasync(-1, filp, 0);
	mxlts_busy--;
	if (!mxlts_busy)
		mxlts_reg_clear();
	MOD_DEC_USE_COUNT;
	return 0;
}

/******************************************************************************
 * Function Name: spi_fasync
 *
 * Input: 		fd	:
 * 			filp	:
 * 			mode	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: provide fasync functionality for select system call
 *
 *****************************************************************************/
static int
spi_fasync(int fd, struct file *filp, int mode)
{
	return (fasync_helper(fd, filp, mode, &ts_fasync));
}

/******************************************************************************
 * Function Name: spi_ioctl
 *
 * Input: 		inode	:
 * 			filp	:
 * 			cmd	: command for ioctl
 * 			arg	: parameter for command
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: ioctl for this device driver
 *
 *****************************************************************************/
static int
spi_ioctl(struct inode *inode,
	  struct file *filp, unsigned int cmd, unsigned long arg)
{
	return -EIO;
}

/******************************************************************************
 * Function Name: spi_poll
 *
 * Input: 		filp	:
 * 			wait	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: support poll and select
 *
 *****************************************************************************/
static unsigned int
spi_poll(struct file *filp, struct poll_table_struct *wait)
{
	poll_wait(filp, &spi_wait, wait);
	return (NODATA())? 0 : (POLLIN | POLLRDNORM);
}

static ssize_t
spi_read(struct file *filp, char *buf, size_t count, loff_t * l)
{
	int nonBlocking = filp->f_flags & O_NONBLOCK;
	ts_event_t *ev;
	int cnt = 0;

	ENTER();

	if (nonBlocking) {
		if (NODATA())
			return -EINVAL;

		cnt = getblock(&ev, count);
		if (cnt > count) {
			debugprintk(1, KERN_ERR "Error read spi buffer\n");
			return -EINVAL;
		}

		copy_to_user(buf, (char *) ev, cnt);
		LEAVE();
		return cnt;

	} else {

		/* check , when woken, there is a complete event to read */
		for (;;) {
			if (!NODATA()) {
				cnt = getblock(&ev, count);
				if (cnt > count) {
					debugprintk(1,
						    KERN_ERR
						    "Error read spi buffer\n");
					return -EINVAL;
				}
				LEAVE();
				/* returns length to be copied otherwise errno -Exxx */
				copy_to_user(buf, (char *) ev, cnt);
				return cnt;
			} else {
				debugprintk(3,
					    "%s: waiting for data...\n",
					    __FUNCTION__);
				interruptible_sleep_on(&spi_wait);
				if (signal_pending(current)) {
					LEAVE();
					return -ERESTARTSYS;
				}
			}
		}
	}
}

#ifdef CONFIG_PM
/*power management event handling*/
static int
mxlts_pm_callback(struct pm_dev *pmdev, pm_request_t rqst, void *data)
{
	__u32 x, y;
	switch (rqst) {
	case PM_SUSPEND:
		if (mxlts_busy) {
			touchpan_disable_int();
			touchpan_readdev(&x, &y, MXLTS_CONF_POWER_OFF);	/*set up power mode */
			mxlts_reg_clear();
		}
		break;

	case PM_RESUME:
		if (mxlts_busy)
			mxlts_reg_init();
		break;
	}
	return 0;
}
#endif

static struct miscdevice mxlts_misc_dev = {
	minor:TS_MINOR,
	name:TS_NAME,
	fops:&spi_fops,
};

int __init
mxlts_init(void)
{
	int tmp;
	unsigned long tmp_reg;
	mxlts_buffer = 0;
	mxlts_initstate_region = 0;
	mxlts_initstate_irq = 0;
	mxlts_initstate_gpio_C_inp = 0;
	mxlts_initstate_gpio_C_outp = 0;
	mxlts_initstate_gpio_D_inp = 0;
	mxlts_initstate_thread = 0;
	mxlts_initstate_misc_dev = 0;

	printk("DBMXL ADS Touchscreen Driver, Ver. 1.1\n");
	debugprintk(0, "compiled %s %s\n", __TIME__, __DATE__);

	tmp = (int) request_region(IO_ADDRESS(SPI_BASE), 0x20, "mxlads_ts");
	if (!tmp) {
		printk(KERN_ERR "%s: MXL SPI1 already in use\n", __FILE__);
		mxlts_exit();
		return -EPERM;
	}
	mxlts_initstate_region = 1;

	if ((tmp = misc_register(&mxlts_misc_dev)) < 0) {
		printk(KERN_ERR
		       "%s: failed to register misc device, code %d\n",
		       __FILE__, tmp);
		mxlts_exit();
		return tmp;
	}
	mxlts_initstate_misc_dev = 1;

	if ((tmp = mx1_register_gpio(PORT_C, 16, PRIMARY | INPUT)) < 0) {
		printk(KERN_ERR
		       "%s: could not register GPIO C pin 16\n", __FILE__);
		mxlts_exit();
		return tmp;
	}
	mxlts_initstate_gpio_C_inp = 1;

	if ((tmp = mx1_register_gpios(PORT_C, (1 << 15) | (1 << 14) | (1 << 13)
				      | (1 << 17), PRIMARY | OUTPUT)) < 0) {
		printk(KERN_ERR
		       "%s: could not register GPIO C pins 13,14,15,17\n",
		       __FILE__);
		mxlts_exit();
		return tmp;
	}
	mxlts_initstate_gpio_C_outp = 1;

	if ((tmp = mx1_register_gpio(PORT_D, TOUCHPAN_BIT, GPIO | INPUT)) < 0) {
		printk(KERN_ERR
		       "%s: could not register GPIO D pins 7,8,10,%d\n",
		       __FILE__, TOUCHPAN_BIT);
		mxlts_exit();
		return tmp;
	}
	mxlts_initstate_gpio_D_inp = 1;
	mx1_gpio_config_intr(PORT_D, TOUCHPAN_BIT, POSITIVE_EDGE);

	if (NULL == (mxlts_buffer = (ts_event_t *) vmalloc(BUFSIZE))) {
		printk(KERN_ERR
		       "%s: not enough kernel memory for spi data buffer\n",
		       __FILE__);
		mxlts_exit();
		return -1;
	}

	/*Request for Touch panel ISR handler */
	if ((tmp = request_irq(TPNL_IRQ,
			       (void *) touchpan_isr,
			       TPNL_INTR_MODE, "mxlts", "mxlts")) < 0) {
		printk(KERN_ERR
		       "%s: failed to request_irq %d, code %d\n",
		       __FILE__, TPNL_IRQ, tmp);
		mxlts_exit();
		return tmp;
	}
	mxlts_initstate_irq = 1;

	if (((tmp_reg =
	      inl(IO_ADDRESS((PLL_PCDR + PLL_BASE)))) & PCDR_PCLKDIV2_MASK)
	    != ((MXLTS_PLLCLK2 << PCDR_PCLKDIV2_BIT) & PCDR_PCLKDIV2_MASK)) {
		printk(KERN_WARNING
		       "%s: changing PERCLK2 from 0x%x to 0x%x\n",
		       __FILE__,
		       (unsigned
			int) ((tmp_reg & PCDR_PCLKDIV2_MASK) >>
			      PCDR_PCLKDIV2_BIT), MXLTS_PLLCLK2);
		tmp_reg &= ~PCDR_PCLKDIV2_MASK;
		tmp_reg |=
		    (MXLTS_PLLCLK2 << PCDR_PCLKDIV2_BIT) & PCDR_PCLKDIV2_MASK;
		outl(tmp_reg, IO_ADDRESS((PLL_PCDR + PLL_BASE)));
	}
	/*init thread */
	mxlts_scan_thread_terminating = 0;
	tmp =
	    kernel_thread(&mxlts_scan_thread, NULL,
			  CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
	if (tmp < 0) {
		printk(KERN_ERR "%s: could not start thread\n", __FILE__);
		mxlts_exit();
		return tmp;
	}
	mxlts_initstate_thread = 1;

#ifdef CONFIG_PM
	mxlts_pmdev =
	    pm_register(PM_UNKNOWN_DEV, PM_SYS_UNKNOWN, mxlts_pm_callback);

	if (!mxlts_pmdev)
		printk(KERN_WARNING
		       "%s: failed to init power management\n", __FILE__);
#endif
	return 0;
}

void __init
mxlts_exit(void)
{
#ifdef CONFIG_PM
	pm_unregister(mxlts_pmdev);
#endif

	if (mxlts_initstate_thread) {
		mxlts_scan_thread_terminating = 1;
		wake_up(&mxlts_scan_lock);
		wait_for_completion(&mxlts_scan_thread_exit);
	}

	if (mxlts_initstate_irq) {
		disable_irq(TPNL_IRQ);
		free_irq(TPNL_IRQ, "mxlts");
	}

	if (mxlts_buffer)
		vfree(mxlts_buffer);

	if (mxlts_initstate_gpio_D_inp)
		mx1_unregister_gpio(PORT_D, TOUCHPAN_BIT);

	if (mxlts_initstate_gpio_C_outp)
		mx1_unregister_gpios(PORT_C,
				     (1 << 15) | (1 << 14) | (1 <<
							      13) | (1 << 17));

	if (mxlts_initstate_gpio_C_inp)
		mx1_unregister_gpio(PORT_C, 16);

	if (mxlts_initstate_misc_dev)
		misc_deregister(&mxlts_misc_dev);

	if (mxlts_initstate_region)
		release_region(IO_ADDRESS(SPI_BASE), 0x20);
}

module_init(mxlts_init);
module_exit(mxlts_exit);

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("MXLADS Touchscreen Driver");
MODULE_LICENSE("GPL");
