/******************************************************************************
	ts-dbmx1.c
	driver for Motorola DragonBall MX1 ASP - Touchscreen support

	NOTE: Voice Input/Output support is in vio-dbmx1.c

	Author: MontaVista Software, Inc. <source@mvista.com>
	Copyright (c) 2003 MontaVista Software, Inc.

	This program is based on asp.c from Motorola DBMX1 ADS BSP Ver. 0.3.4
	Author: Chen Ning
	Date of Creation:   10 DEC,2001
	Copyright (C) 2001 Motorola Semiconductors HK Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
	
	Modifications:
	Nov 2003 - ver 1.1, MontaVista Software, Inc: changed to use misc device

********************************************************************************/

#include <linux/module.h>
#include <linux/version.h>

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <asm/uaccess.h>	/* get_user,copy_to_user */

#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/string.h>
#include <linux/init.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <linux/errno.h>
#include <linux/tqueue.h>
#include <linux/wait.h>
#include <linux/miscdevice.h>

#ifdef CONFIG_PM
#include <linux/pm.h>
#endif

#include <asm/irq.h>
#include <asm/arch/hardware.h>
#include <asm/arch/irqs.h>

#include <asm/arch/platform.h>

#include "ts-dbmx1.h"

#define MX1TS_PLLCLK2 5

#ifdef DEBUG_MX1TS
#define debugprintk(fmt, args...) printk("%s: " fmt, __FUNCTION__ , ## args)
#else
#define debugprintk(fmt, args...)
#endif

#ifdef _ASP_DEBUG
#define TRACE printk
#else
#define TRACE  1?0:printk
#endif

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

/*register access macros*/
#define mx1_asp_reg_out(r,x) outl(x,IO_ADDRESS((ASP_BASE + r)))
#define mx1_asp_reg_in(r) inl(IO_ADDRESS((ASP_BASE + r)))

#define WRITEREG mx1_asp_reg_out
#define READREG  mx1_asp_reg_in

#define ASP_FIFO_SIZE	12

#define DEV_IRQ_NAME	"PenDevice"
#define DEV_IRQ_ID	    "PenDevice"

static int mx1ts_initstate_region_asp;
static int mx1ts_initstate_region_ts;
static int mx1ts_initstate_irq_5;
static int mx1ts_initstate_irq_pen;
static int mx1ts_initstate_irq_touch;
static int mx1ts_initstate_misc_dev;

/* local buffer for store data
 * speed  200 f/s
 *
 * a new feature is read block, assume the buffer is a continuous buffer
 * so, if there is enough data, read data from current position to
 * buffer end, then next time, from buffer head to data end. -- the buffer
 * is a loop buffer.
 */
static ts_event_t *mx1ts_buffer;
#define BUFLEN	250
#define BUFSIZE (BUFLEN*sizeof(ts_event_t))
#define NEXTI(i)	{i=(i==BUFLEN-1)?0:i+1;}
#define GETNEXTI(i)	((i==BUFLEN-1)?0:i+1)
#define BLKEND(i)	((i)==0)

#if 1
#define DSIZE()		((wptr<rptr)?(BUFLEN-rptr):(wptr-rptr))
#else
#define DSIZE()		((wptr<rptr)?(BUFLEN-rptr+wptr):(wptr-rptr))
#endif

DECLARE_WAIT_QUEUE_HEAD(ts_wait);
static struct fasync_struct *ts_fasync;

static int _gTplIsAutoCalMode = 1;

static u16 _gTplAutoZero = 0;
static u16 _gTplRangeX = 0;
static u16 _gTplRangeY = 0;

#ifdef CONFIG_PM
static struct pm_dev *mx1ts_pmdev;
#endif

static void disable_auto_sample(void);
static void enable_auto_sample(void);

static void clear_pen_interrupt_flag(void);

static void clear_data_interrupt_flag(void);
static void clear_pen_FIFO(void);

static void disable_data_interrupt(void);

/* functions and interface */
static int asp_open(struct inode *inode, struct file *filp);
static int asp_release(struct inode *inode, struct file *filp);
static int asp_fasync(int fd, struct file *filp, int mode);
static ssize_t asp_read(struct file *, char *, size_t, loff_t * l);
static int asp_ioctl(struct inode *inode,
		     struct file *filp, unsigned int cmd, unsigned long arg);
static unsigned int asp_poll(struct file *filp, struct poll_table_struct *wait);

static void asp_interrupt(int, void *, struct pt_regs *);	/* ISR : */

int __init mx1ts_init(void);
void __init mx1ts_cleanup(void);

#ifdef CONFIG_PM
static int mx1ts_pm_callback(struct pm_dev *pmdev,
			     pm_request_t rqst, void *data);
#endif

static struct file_operations ts_fops = {
	open:asp_open,
	release:asp_release,
	read:asp_read,
	poll:asp_poll,
	ioctl:asp_ioctl,
	fasync:asp_fasync,
};

/* local prototype */
static void mx1ts_asp_reg_init(void);
static void mx1ts_reg_init(void);
static void mx1ts_asp_reg_clear(void);
static void mx1ts_reg_clear(void);
static int getblock(ts_event_t * *block, int size);
static ts_event_t *getdata(void);
/*
static void addpressure(unsigned short i);
static void addy(unsigned short i);
static void addx(unsigned short i);
*/

static void enable_pen_touch_interrupt(void);
static void disable_pen_touch_interrupt(void);
static void enable_pen_up_interrupt(void);
static void disable_pen_up_interrupt(void);

static void start_auto_cal(void);
static void stop_auto_cal(void);
static void asp_post_process(int *px, int *py);

static void add_x_y(unsigned x, unsigned y, unsigned flag);

static unsigned short rptr;
static unsigned short wptr;
#define NODATA()	(rptr==wptr)

static char mx1ts_busy = 0;

static int
read_pen_fifo(void)
{
	unsigned long reg;
	int x = -1;

	reg = READREG(ASP_ISTATR);
	if (reg & 0x03)
		x = READREG(ASP_PADFIFO);
	return x;
}

/******************************************************************************
 * Function Name: asp_release
 *
 * Input: 		inode	:
 * 			filp	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: release resource when close the inode
 *
 * Modification History:
 * 	10 DEC,2001, Chen Ning
 *****************************************************************************/
static int
asp_release(struct inode *inode, struct file *filp)
{
	debugprintk("\n");
	asp_fasync(-1, filp, 0);
	mx1ts_busy--;
	if (!mx1ts_busy)
		mx1ts_reg_clear();
	MOD_DEC_USE_COUNT;
	return 0;
}

/******************************************************************************
 * Function Name: asp_open
 *
 * Input: 		inode	:
 * 			filp	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: allocate resource when open the inode
 *
 * Modification History:
 * 	10 DEC,2001, Chen Ning
 *****************************************************************************/
static int
asp_open(struct inode *inode, struct file *filp)
{
	debugprintk("\n");
	MOD_INC_USE_COUNT;
	if (!mx1ts_busy)
		mx1ts_reg_init();
	mx1ts_busy++;
	return 0;
}

/******************************************************************************
 * Function Name: asp_fasync
 *
 * Input: 		fd	:
 * 			filp	:
 * 			mode	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: provide fasync functionality for select system call
 *
 * Modification History:
 * 	10 DEC,2001, Chen Ning
 *****************************************************************************/
static int
asp_fasync(int fd, struct file *filp, int mode)
{
	return (fasync_helper(fd, filp, mode, &ts_fasync));
}

/******************************************************************************
 * Function Name: asp_ioctl
 *
 * Input: 		inode	:
 * 			filp	:
 * 			cmd	: command for ioctl
 * 			arg	: parameter for command
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: ioctl for this device driver
 *
 * Modification History:
 * 	10 DEC,2001, Chen Ning
 *****************************************************************************/
static int
asp_ioctl(struct inode *inode,
	  struct file *filp, unsigned int cmd, unsigned long arg)
{
	return -EIO;
}

/******************************************************************************
 * Function Name: asp_poll
 *
 * Input: 		filp	:
 * 			wait	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: support poll and select
 *
 * Modification History:
 * 	10 DEC,2001, Chen Ning
 *****************************************************************************/
static unsigned int
asp_poll(struct file *filp, struct poll_table_struct *wait)
{
	poll_wait(filp, &ts_wait, wait);

	return (NODATA())? 0 : (POLLIN | POLLRDNORM);
}

/******************************************************************************
 * Function Name: asp_read
 *
 * Input: 		filp	: the file
 * 			buf	: data buffer
 * 			count	: number of chars to be readed
 * 			l	: offset of file
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: read device driver
 *
 * Modification History:
 * 	10 DEC,2001, Chen Ning
 *****************************************************************************/
static ssize_t
asp_read(struct file *filp, char *buf, size_t count, loff_t * l)
{
	int nonBlocking = filp->f_flags & O_NONBLOCK;
	ts_event_t *ev;
	int cnt = 0;

	if (nonBlocking) {
		if (!NODATA()) {
			/* returns length to be copied otherwise errno -Exxx */

			cnt = getblock(&ev, count);
			if (cnt > count) {
				printk(KERN_ERR
				       "%s: error reading buffer nonblocking\n",
				       __FILE__);
				return -EINVAL;
			}

			__copy_to_user(buf, (char *) ev, cnt);
			return cnt;
		} else
			return -EINVAL;
	} else {

		/* check , when woken, there is a complete event to read */
		while (1) {
			if (!NODATA()) {

				cnt = getblock(&ev, count);
				if (cnt > count) {
					printk(KERN_ERR
					       "%s: error reading buffer\n",
					       __FILE__);
					return -EINVAL;
				}
				/* returns length to be copied otherwise errno -Exxx */

				__copy_to_user(buf, (char *) ev, cnt);
				return cnt;
			} else {
				interruptible_sleep_on(&ts_wait);
				if (signal_pending(current))
					return -ERESTARTSYS;
			}
		}
	}
}

/******************************************************************************
 * Function Name: asp_interrupt
 *
 * Input: 		irq	: interrupt number
 * 			dev_id	: device driver name
 * 			regs	: registers
 * Value Returned:	void	:
 *
 * Description: interrupt handler for ASP device
 *
 * Modification History:
 * 	
 *****************************************************************************/
static void
asp_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	static int x = -1, y = -1, az = -1, u = -1;
	unsigned long reg, ctl, cmp;
	int i = 0;

	if (strcmp((char *) dev_id, DEV_IRQ_ID)) {
		return;
	}

	reg = READREG(ASP_ISTATR);
	cmp = READREG(ASP_CMPCNTL);
	ctl = READREG(ASP_ACNTLCR);

	if ((reg & 0x40) && !(ctl & 0x4000)) {	/* user start touch the panel */
		clear_pen_FIFO();

		_gTplIsAutoCalMode = TRUE;	/* Enter Auto Cal Mode */
		start_auto_cal();
		clear_pen_interrupt_flag();
		disable_pen_touch_interrupt();
		enable_pen_up_interrupt();

		x = -1;
		y = -1;
	} else if ((reg & 0x400)) {	/* user pen up interrupt */

		disable_auto_sample();
		disable_data_interrupt();
		clear_pen_interrupt_flag();
		clear_pen_FIFO();

		enable_pen_touch_interrupt();
		disable_pen_up_interrupt();

		if ((x != -1) && (y != -1)) {
			add_x_y(x, y, PENUP);
		}
		wake_up_interruptible(&ts_wait);
		if (ts_fasync)
			kill_fasync(&ts_fasync, SIGIO, POLL_IN);
	} else if ((reg & 0x03) && (ctl & 0x4000)) {	/* pen data  interrupt */

		clear_pen_interrupt_flag();

		/*  Read data from Tpl/ASP FIFO to RingBuffer */
		x = -1;
		y = -1;
		i = 0;

		/* For Auto Calibrate */
		if (_gTplIsAutoCalMode) {
			az = read_pen_fifo();
			x = read_pen_fifo();
			y = read_pen_fifo();
			if (az < x && az < y) {	/*  valid data set */
				_gTplAutoZero = az & 0xFFFF;
				_gTplRangeX = x & 0xFFFF;
				_gTplRangeY = y & 0xFFFF;

				_gTplIsAutoCalMode = FALSE;	/* Exit Auto Cal Mode */
				stop_auto_cal();
				enable_auto_sample();

				return;
			} else {
				/* Invalid data set */
				start_auto_cal();
			}
			return;
		}

		for (i = 0; i < ASP_FIFO_SIZE / 3; i++) {
			if (READREG(ASP_ISTATR) & 0x1) {
				az = read_pen_fifo();	/* read AUTO Zero */
				if (az > _gTplAutoZero + 0x200)	/* search synchronization header */
					continue;

				x = -1;
				y = -1;

				x = read_pen_fifo();	/* read x; */
				y = read_pen_fifo();	/* read y; */

				u = read_pen_fifo();	/* read U; */

				asp_post_process(&x, &y);
				if (x != -1 && y != -1)
					add_x_y(x, y, PENDOWN);
			}
		}

		wake_up_interruptible(&ts_wait);
		if (ts_fasync)
			kill_fasync(&ts_fasync, SIGIO, POLL_IN);
		clear_data_interrupt_flag();
		clear_pen_interrupt_flag();
	}

	else {
		x = -1;
		y = -1;
		clear_pen_interrupt_flag();
	}

}

static void
start_auto_cal(void)
{
	unsigned long reg;
	clear_pen_FIFO();
	reg = READREG(ASP_ACNTLCR);

	reg &= ~0x3000;
	reg |= 0x1000;		/* set mode is 01 //must set mod 10 for pen up detect */

	reg |= 0x8000;		/* enable AutoZero */
	reg |= 0x4000000;	/* enable auto calibrate; */
	reg |= 0x4000;		/* enable auto sample */
	reg |= 0x2;

	WRITEREG(ASP_ACNTLCR, reg);

	reg = READREG(ASP_ICNTLR);
	reg &= 0xFFFFB8C;
	reg |= 0x02 | 0x400;	/* enable FIFO full and Pen Up; */
	WRITEREG(ASP_ICNTLR, reg);
}

static void
stop_auto_cal(void)
{
	unsigned long reg;
	reg = READREG(ASP_ACNTLCR);
	reg &= ~0x4000000;	/* Disable auto Calibrate; */
	WRITEREG(ASP_ACNTLCR, reg);
}

static void
asp_post_process(int *px, int *py)
{
	u32 x;
	u32 y;
	if (_gTplAutoZero == 0) {
		*px = -1;
		return;
	}
	if (_gTplRangeX == 0) {
		*px = -1;
		return;
	}
	if (_gTplRangeY == 0) {
		*px = -1;
		return;

	}
	x = (*px) & 0xFFFF;
	y = (*py) & 0xFFFF;
	if (x < _gTplAutoZero || y < _gTplAutoZero) {
		*px = -1;
		return;
	}
	x = (u32) ((x - _gTplAutoZero) << 16);
	x /= (u32) (_gTplRangeX - _gTplAutoZero);
	y = (u32) ((y - _gTplAutoZero) << 16);
	y /= (u32) (_gTplRangeY - _gTplAutoZero);

	*px = x >> 4;		/* same value S16 and U16 */
	*py = y >> 4;

}

static void
enable_auto_sample(void)
{
	unsigned long data;

	clear_pen_FIFO();

	data = READREG(ASP_ACNTLCR);
	data &= ~0x3000;
	data |= 0x2000;		/* set the 0x10 mode */
	WRITEREG(ASP_ACNTLCR, data);
	data |= 0x8000;		/* enable auto zero */
	WRITEREG(ASP_ACNTLCR, data);
	data |= 0x4000;		/* enable auto sample */
	WRITEREG(ASP_ACNTLCR, data);

	data |= 0x02;		/* enable PEN A/D */
	WRITEREG(ASP_ACNTLCR, data);

	data = READREG(ASP_ICNTLR);
	data |= 0x3;		/* enable pen data ready and full interrupt */
	WRITEREG(ASP_ICNTLR, data);
}

static void
disable_auto_sample(void)
{
	unsigned long data;

	data = READREG(ASP_ACNTLCR);
	data &= 0x0ffffbfff;	/* disable auto */
	WRITEREG(ASP_ACNTLCR, data);

	data &= 0x0ffff0fff;
	WRITEREG(ASP_ACNTLCR, data);

	data &= 0x0fffffffd;	/* disable PEN A/D */
	WRITEREG(ASP_ACNTLCR, data);

	data = READREG(ASP_ICNTLR);
	data &= ~0x3;		/* disable pen data ready and full interrupt */
	WRITEREG(ASP_ICNTLR, data);

}

/* Frank Li
Only Enable Pen Touch Interrupt
*/
static void
enable_pen_touch_interrupt(void)
{
	unsigned long val;
	val = READREG(ASP_ICNTLR);
	val |= 0x30;
	WRITEREG(ASP_ICNTLR, val);

}

static void
disable_pen_touch_interrupt(void)
{
	unsigned long val;

	val = READREG(ASP_ICNTLR);
	val &= ~0x010;
	WRITEREG(ASP_ICNTLR, val);
}

static void
enable_pen_up_interrupt(void)
{
	unsigned long val;
	val = READREG(ASP_ICNTLR);
	val |= 0x400;
	WRITEREG(ASP_ICNTLR, val);
}

static void
disable_pen_up_interrupt(void)
{
	unsigned long val;
	val = READREG(ASP_ICNTLR);
	val &= ~0x400;
	WRITEREG(ASP_ICNTLR, val);
}

static void
clear_pen_interrupt_flag(void)
{
	unsigned long val;

	val = READREG(ASP_ISTATR);
	val |= 0x4C0;
	WRITEREG(ASP_ISTATR, val);
}

static void
clear_data_interrupt_flag(void)
{
	unsigned long val;

	val = READREG(ASP_ISTATR);

	if (val & 0xc0) {
		val = 0xc0;
		WRITEREG(ASP_ISTATR, val);
	}
}

static void
clear_pen_FIFO(void)
{
	unsigned long val;

	int i;

	for (i = 0; i < 12; i++) {
		val = READREG(ASP_ISTATR);
		if (val & 0x03)
			val = READREG(ASP_PADFIFO);
	}
}

static void
disable_data_interrupt(void)
{
	unsigned long val;

	val = READREG(ASP_ICNTLR);

	val &= 0xfffd;
	WRITEREG(ASP_ICNTLR, val);
}

static void
mx1ts_asp_reg_init(void)
{
	mx1_asp_reg_out(ASP_ACNTLCR, (1UL << 25));	/*enable clock */

	mx1_asp_reg_out(ASP_ACNTLCR, (mx1_asp_reg_in(ASP_ACNTLCR) | (1UL << 23)));	/*software reset */
	for (;;)
		if (!(mx1_asp_reg_in(ASP_ACNTLCR) & (1UL << 23)))
			break;	/* wait for sw reset ready */

	mx1_asp_reg_out(ASP_ACNTLCR, (mx1_asp_reg_in(ASP_ACNTLCR) | 0x1));	/*enable  BandGap */
	for (;;)
		if (mx1_asp_reg_in(ASP_ISTATR) & (1UL << 9))
			break;	/* wait for bandgap ready */
}

static void
mx1ts_asp_reg_clear(void)
{
	mx1_asp_reg_out(ASP_ICNTLR, 0);
	mx1_asp_reg_out(ASP_CLKDIV, 0);
	mx1_asp_reg_out(ASP_ACNTLCR, 0);	/*disable ASP */
}

/******************************************************************************
 * Function Name: mx1ts_reg_init
 *
 * Input: 			:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: initialize ASP hardware
 *
 * Modification History:
 * 	10 DEC,2001, Chen Ning
 *****************************************************************************/
static void
mx1ts_reg_init(void)
{
	unsigned long value;

/* Initialize registers:
Common voice/pen registers:
ASP_ACNTLCR - control
ASP_ICNTLR - dma/int control
ASP_ISTATR - int status
ASP_CLKDIV - clock divider

Pen-related registers:
ASP_PADFIFO - fifo
ASP_PSMPLRG - sample rate control
ASP_CMPCNTL - compare & compare control

Voice-related registers:
ASP_VADFIFO - input fifo
ASP_VDAFIFO - output fifo
ASP_VADGAIN - input FIR/Comb gains/decimations
ASP_VDAGAIN - output FIR/Comb gains/decimations
ASP_VADCOEF - access to input FIR RAM with coefficients
ASP_VDACOEF - access to output FIR RAM with coefficients

*/
	value = READREG(ASP_ACNTLCR);
	value |= 0x00040200;
	WRITEREG(ASP_ACNTLCR, value);	/* set the default value of the conreg */

	value = READREG(ASP_CLKDIV);
	value &= 0xFFE0;
	value |= 0x01;
	WRITEREG(ASP_CLKDIV, value);	/* clock divide ratio 32 */

	value = 0x701f;
	WRITEREG(ASP_PSMPLRG, value);	/* set the sample rate control reg */

	value = 0;
	WRITEREG(ASP_CMPCNTL, value);	/* disable the compare function */

	clear_pen_FIFO();
	wptr = 0;
	rptr = 0;

	enable_pen_touch_interrupt();
}

static void
mx1ts_reg_clear(void)
{
	disable_pen_touch_interrupt();
	disable_data_interrupt();
}

static struct miscdevice mx1ts_misc_dev = {
	minor:TS_MINOR,
	name:TS_NAME,
	fops:&ts_fops,
};

/******************************************************************************
 * Function Name: mx1ts_init
 *
 * Input: 		void	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: device driver initialization
 *
 * Modification History:
 * 	10 DEC,2001, Chen Ning
 *****************************************************************************/
int __init
mx1ts_init(void)
{
	unsigned long tmp_reg;
	int tmp;
	mx1ts_initstate_region_asp = 0;
	mx1ts_initstate_region_ts = 0;
	mx1ts_initstate_irq_5 = 0;
	mx1ts_initstate_irq_pen = 0;
	mx1ts_initstate_irq_touch = 0;
	mx1ts_buffer = 0;
	mx1ts_initstate_misc_dev = 0;

	printk("DBMX1 ASP Touchscreen Driver, Ver. 1.1\n");
	debugprintk("%s %s\n", __TIME__, __DATE__);

	/*NOTE: ASP registers support touchscreen and also sound in a separate driver */
	tmp =
	    (int) request_region(IO_ADDRESS(ASP_ACNTLCR), 0x4, "dbmx1_asp(ts)");
	if (tmp) {
		debugprintk("taking common controls of ASP\n");
		mx1ts_initstate_region_asp = 1;
	}

	/*NOTE: ASP registers support touchscreen and also sound in a separate driver */
	tmp = (int) request_region(IO_ADDRESS(ASP_PADFIFO), 0x4, "dbmx1_ts");
	if (!tmp) {
		printk(KERN_ERR "%s: ASP touchscreen already in use\n",
		       __FILE__);
		mx1ts_cleanup();
		return -EPERM;
	}
	mx1ts_initstate_region_ts = 1;

	if (((tmp_reg =
	      inl(IO_ADDRESS((PLL_PCDR + PLL_BASE)))) & PCDR_PCLKDIV2_MASK)
	    != ((MX1TS_PLLCLK2 << PCDR_PCLKDIV2_BIT) & PCDR_PCLKDIV2_MASK)) {
		printk(KERN_WARNING
		       "%s: changing PERCLK2 from 0x%x to 0x%x\n",
		       __FILE__,
		       (unsigned int) ((tmp_reg & PCDR_PCLKDIV2_MASK) >>
				       PCDR_PCLKDIV2_BIT), MX1TS_PLLCLK2);
		tmp_reg &= ~PCDR_PCLKDIV2_MASK;
		tmp_reg |=
		    (MX1TS_PLLCLK2 << PCDR_PCLKDIV2_BIT) & PCDR_PCLKDIV2_MASK;
		outl(tmp_reg, IO_ADDRESS((PLL_PCDR + PLL_BASE)));
	}

	tmp = request_irq(5, asp_interrupt, SA_SHIRQ, DEV_IRQ_NAME, DEV_IRQ_ID);
	if (tmp < 0) {
		printk(KERN_ERR "%s: error requesting irq 5\n", __FILE__);
		mx1ts_cleanup();
		return tmp;
	}
	mx1ts_initstate_irq_5 = 1;

	tmp = request_irq(PEN_DATA_INT,
			  asp_interrupt, SA_SHIRQ, DEV_IRQ_NAME, DEV_IRQ_ID);
	if (tmp < 0) {
		printk(KERN_ERR "%s: error requesting PEN_DATA_INT\n",
		       __FILE__);
		mx1ts_cleanup();
		return tmp;
	}
	mx1ts_initstate_irq_pen = 1;

	tmp = request_irq(TOUCH_INT,
			  asp_interrupt, SA_SHIRQ, DEV_IRQ_NAME, DEV_IRQ_ID);

	if (tmp < 0) {
		printk(KERN_ERR "%s: error requesting TOUCH_INT\n", __FILE__);
		mx1ts_cleanup();
		return tmp;
	}
	mx1ts_initstate_irq_touch = 1;

	mx1ts_buffer = (ts_event_t *) vmalloc(BUFSIZE);
	if (!mx1ts_buffer) {
		printk(KERN_ERR
		       "%s: failed to allocate memory - %d bytes\n",
		       __FILE__, BUFSIZE);
		mx1ts_cleanup();
		return -EPERM;
	}
	/* register our character device */
	tmp = misc_register(&mx1ts_misc_dev);
	if (tmp < 0) {
		printk(KERN_ERR "%s: error registering device\n", __FILE__);
		mx1ts_cleanup();
		return tmp;
	}
	mx1ts_initstate_misc_dev = 1;

#ifdef CONFIG_PM
	mx1ts_pmdev =
	    pm_register(PM_UNKNOWN_DEV, PM_SYS_UNKNOWN, mx1ts_pm_callback);

	if (!mx1ts_pmdev)
		printk(KERN_WARNING
		       "%s: failed to init power management\n", __FILE__);
#endif

	if (mx1ts_initstate_region_asp)
		mx1ts_asp_reg_init();
	return 0;
}

/******************************************************************************
 * Function Name: mx1ts_cleanup
 *
 * Input: 		void	:
 * Value Returned:	void	:
 *
 * Description: clean up and free all of resource for this MODULE
 *
 * Modification History:
 * 	10 DEC,2001, Chen Ning
 *****************************************************************************/
void __init
mx1ts_cleanup(void)
{
	if (mx1ts_initstate_misc_dev) {

		if (!(check_region(IO_ADDRESS(ASP_VADFIFO), 0x4)))	/*ASP voice not loaded */
			mx1ts_asp_reg_clear();
#ifdef CONFIG_PM
		pm_unregister(mx1ts_pmdev);
#endif
		misc_deregister(&mx1ts_misc_dev);;
	}

	if (mx1ts_buffer)
		vfree(mx1ts_buffer);

	if (mx1ts_initstate_irq_touch) {
		disable_irq(TOUCH_INT);
		free_irq(TOUCH_INT, DEV_IRQ_ID);
	}

	if (mx1ts_initstate_irq_pen) {
		disable_irq(PEN_DATA_INT);
		free_irq(PEN_DATA_INT, DEV_IRQ_ID);
	}

	if (mx1ts_initstate_irq_5) {
		disable_irq(5);
		free_irq(5, DEV_IRQ_ID);
	}

	if (mx1ts_initstate_region_ts)
		release_region(IO_ADDRESS(ASP_PADFIFO), 0x4);

	if (mx1ts_initstate_region_asp)
		release_region(IO_ADDRESS(ASP_ACNTLCR), 0x4);
}

#ifdef CONFIG_PM
/*power management event handling*/
static int
mx1ts_pm_callback(struct pm_dev *pmdev, pm_request_t rqst, void *data)
{
	switch (rqst) {
	case PM_SUSPEND:
		if (mx1ts_busy)
			mx1ts_reg_clear();
		if (!(check_region(IO_ADDRESS(ASP_VADFIFO), 0x4)))	/*ASP voice not loaded */
			mx1ts_asp_reg_clear();
		if (mx1ts_initstate_region_ts)
			release_region(IO_ADDRESS(ASP_PADFIFO), 0x4);

		if (mx1ts_initstate_region_asp)
			release_region(IO_ADDRESS(ASP_ACNTLCR), 0x4);
		mx1ts_initstate_region_ts = 0;
		mx1ts_initstate_region_asp = 0;
		break;

	case PM_RESUME:
		/*NOTE: ASP registers support touchscreen and also sound in a separate driver */
		if (request_region(IO_ADDRESS(ASP_ACNTLCR), 0x4,
				   "dbmx1_asp(ts)")) {
			debugprintk("taking common controls of ASP\n");
			mx1ts_initstate_region_asp = 1;
		}

		/*NOTE: ASP registers support touchscreen and also sound in a separate driver */
		if (!(request_region(IO_ADDRESS(ASP_PADFIFO), 0x4, "dbmx1_ts"))) {
			printk(KERN_ERR
			       "%s: re-request ASP touchscreen region failed from PM\n",
			       __FILE__);
		} else
			mx1ts_initstate_region_ts = 1;

		if (mx1ts_initstate_region_asp)
			mx1ts_asp_reg_init();
		if (mx1ts_busy)
			mx1ts_reg_init();
		break;
	}
	return 0;
}
#endif

/******************************************************************************
 * Function Name: add_x_y
 *
 * Input: 		i	:
 * Value Returned:	void	:
 *
 * Description: add pen data to buffer
 *
 *****************************************************************************/
static void
add_x_y(unsigned x, unsigned y, unsigned flag)
{
	if (GETNEXTI(wptr) == rptr)
		return;
	mx1ts_buffer[wptr].x = x;
	mx1ts_buffer[wptr].y = y;

	mx1ts_buffer[wptr].pressure = flag;

	NEXTI(wptr);		/*  goto next wptr */
}

/******************************************************************************
 * Function Name: getdata
 *
 * Input: 		void	:
 * Value Returned:	ts_event_t	: a 'touch' event data format
 *
 * Description: read a 'touch' event from data buffer
 *
 * Modification History:
 * 	10 DEC,2001, Chen Ning
 *****************************************************************************/
/*  no really read out the data, so no copy */
static ts_event_t *
getdata(void)
{
	ts_event_t *data;

	if (NODATA())
		return NULL;

	data = &(mx1ts_buffer[rptr]);

	NEXTI(rptr);
	return data;
}

/******************************************************************************
 * Function Name: getblock
 *
 * Input: 		block	:
 * 			size	:
 * Value Returned:	int	: count of data size
 *
 * Description: read a block of 'touch' data from buffer, read count shall less
 * 	than size, but shall be as more as possible. the data shall not really
 * 	copy to upper layer, until copy_user is invoked. So 'ZERO COPY' is
 * 	inplemented.
 *
 * Modification History:
 * 	10 DEC,2001, Chen Ning
 *****************************************************************************/
/*  assume continuous buffer */
static int
getblock(ts_event_t * *block, int size)
{
	int cnt = 0, rd;
	unsigned long flags;
	ts_event_t *p;
	if (NODATA())
		return 0;

	save_flags(flags);
	cli();

	if (DSIZE() * sizeof (ts_event_t) >= size) {
		rd = size / sizeof (ts_event_t);
	} else {
		rd = DSIZE();
	}

	*block = p = getdata();	/* get block head */
	cnt++;

	while (p && (cnt < rd)) {
		if (rptr == 0)
			break;
		p = getdata();
		cnt++;
	}
	restore_flags(flags);

	return (cnt) * sizeof (ts_event_t);

}

module_init(mx1ts_init);
module_exit(mx1ts_cleanup);

MODULE_AUTHOR("MontaVista Software, Inc. <source@mvista.com>");
MODULE_DESCRIPTION("DBMX1 ASP Touchscreen Driver");
MODULE_LICENSE("GPL");

/* end of file */
