/*
  Copyright (C) 2003 Takashi SHUDO
  SH-7300 VIO3 driver for Super-H
*/

//#define DDEBUG
#ifdef DDEBUG
#define D(x)	x
#ifdef CONFIG_SH_7751_SOLUTION_ENGINE
#include <asm/hitachi_7751se.h>
#elif defined(CONFIG_SH_MOBILE_SOLUTION_ENGINE)
#include <asm/hitachi_shmse.h>
#else
#include <asm/hitachi_se.h>
#endif
#else
#define D(x)
#endif

#define YC_LINEAR /* When Y of a frame buffer and CrCb are connected
					 and arranged */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/mm.h>
#include <asm/io.h>
#include <asm/irq-sh7300.h>

#include "shm_vio.h"

#define VIO_INTLVL	10	// Interruption priority level

/*
 * wait queue for processes wanting to open
 */
static DECLARE_WAIT_QUEUE_HEAD(wq);

static void init_vio3(void)
{
	// PFC setting
	ctrl_outw(ctrl_inw(PSELA) | 0x8000, PSELA);
	ctrl_outw(0x0000, PACR);
	ctrl_outw(ctrl_inw(PECR) & 0x003F, PECR);
	ctrl_outw(ctrl_inw(PMCR) & 0x3FFF, PMCR);
	ctrl_outw(ctrl_inw(HIZCRA) & 0xBFFF, HIZCRA);
	ctrl_outw(ctrl_inw(DRVCR) | 0x0001, DRVCR);

	// Initialise all registers
	ctrl_outl( 0x00000000, VIO3_CAPSR );
	ctrl_outl( 0x00000000, VIO3_CAPCR );
	ctrl_outl( 0x00000000, VIO3_CAMCR );
	ctrl_outl( 0x00000000, VIO3_CAMOR );
	ctrl_outl( 0x00000000, VIO3_CAPWR );
	ctrl_outl( 0x00000000, VIO3_CSTCR );
	ctrl_outl( 0x00000000, VIO3_CFLCR );
	ctrl_outl( 0x00000000, VIO3_CFSZR );
	ctrl_outl( 0x00000000, VIO3_CDWDR );
	ctrl_outl( 0x00000000, VIO3_CDAYR );
	ctrl_outl( 0x00000000, VIO3_CDACR );
	ctrl_outl( 0x00000000, VIO3_CDMCR );
	ctrl_outl( 0x00000000, VIO3_CDBCR );
	ctrl_outl( 0x00000000, VIO3_CROTR );
	ctrl_outl( 0x00000000, VIO3_VIPSR );
	ctrl_outl( 0x00000000, VIO3_VVSWR );
	ctrl_outl( 0x00000000, VIO3_VVSSR );
	ctrl_outl( 0x00000000, VIO3_VDWDR );
	ctrl_outl( 0x00000000, VIO3_VSARR );
	ctrl_outl( 0x00000000, VIO3_VSAGR );
	ctrl_outl( 0x00000000, VIO3_VSABR );
	ctrl_outl( 0x00000000, VIO3_VDAYR );
	ctrl_outl( 0x00000000, VIO3_VDACR );
	ctrl_outl( 0x00000000, VIO3_VTRCR );
	ctrl_outl( 0x00000000, VIO3_VFLCR );
	ctrl_outl( 0x00000000, VIO3_VFSZR );
	ctrl_outl( 0x00000000, VIO3_VLUTR );
	ctrl_outl( 0x00000000, VIO3_VROTR );
	ctrl_outl( 0x00000000, VIO3_VOSCR );
	ctrl_outl( 0x00000000, VIO3_VOSWR );
	ctrl_outl( 0x00000000, VIO3_VODSR );
	ctrl_outl( 0x00000000, VIO3_VODOR );
	ctrl_outl( 0x00000000, VIO3_VOSAR );
	ctrl_outl( 0x00000000, VIO3_VOAYR );
	ctrl_outl( 0x00000000, VIO3_VOACR );
	ctrl_outl( 0x00000000, VIO3_VHCCR );
	ctrl_outl( 0x00000000, VIO3_VHDSR );
	ctrl_outl( 0x00000000, VIO3_VHDOR );
	ctrl_outl( 0x00000000, VIO3_VHAYR );
	ctrl_outl( 0x00000000, VIO3_VHACR );
	ctrl_outl( 0x00000000, VIO3_FXADR );
	ctrl_outl( 0x00000000, VIO3_DSWPR );
	ctrl_outl( 0x00000000, VIO3_EVTCR );
	ctrl_outl( 0x00000000, VIO3_EVTSR );
	ctrl_outl( 0x00000000, VIO3_STATR );
	ctrl_outl( 0x00000000, VIO3_SRSTR );
}


static void set_camera(void)
{
	/*
	  4:2:2
	  Cb0,Y0,Cr0,Y1
	  VD:High Active
	  HD:High Active
	 */
//	ctrl_outl(0x0000100, VIO3_CAMCR);	// 4:2:2
	ctrl_outl(0x0000004, VIO3_CAMCR);	// 4:2:0
//	ctrl_outl(0x0000000, VIO3_CAMCR);	// 4:2:0

	// offset
	ctrl_outl(((M_HEIGHT<<16) + (M_WIDTH<<1)) & 0x0fff0fff, VIO3_CAMOR);

	// size
	ctrl_outl(((M_HEIGHT<<16) + (M_WIDTH<<1)) & 0x0ffc0ff8, VIO3_CAPWR);

	// filter
	ctrl_outl((RESIZERATIO_1_1<<16) + RESIZERATIO_1_1, VIO3_CFLCR);

	// haba
	ctrl_outl((M_WIDTH & 0x0fff), VIO3_CDWDR);
	
	// clip
	ctrl_outl(((M_HEIGHT<<16) + M_WIDTH) & 0x07fc07fc, VIO3_CFSZR);

	// address inc
	ctrl_outl(0x00000000, VIO3_FXADR);

	// data swap
	ctrl_outl(0x00000013, VIO3_DSWPR);
}


static void start_capture(unsigned long fn)
{
	D(printk("VIO3 Capture\n"));

	ctrl_outl(0x00060d20, VIO3_VTRCR);	// 4:2:0

	// Y,Cr,Cb Data Address
#ifdef YC_REVERSE
	ctrl_outl(0x00000003, VIO3_CROTR);	// 180-degree rotation
	ctrl_outl(Y_BUFF + (VMEM_SIZE * fn) + M_WIDTH*M_HEIGHT -4, VIO3_CDAYR);
#ifdef YC_LINEAR
	ctrl_outl(Y_BUFF + Y_SIZE + (VMEM_SIZE * fn) + (M_WIDTH*M_HEIGHT)/2 -4,
			  VIO3_CDACR);
#else
	ctrl_outl(C_BUFF + (VMEM_SIZE * fn) + (M_WIDTH*M_HEIGHT)/2 -4, VIO3_CDACR);
#endif
#else
	ctrl_outl(0x00000000, VIO3_CROTR);
	ctrl_outl(Y_BUFF + (VMEM_SIZE * fn), VIO3_CDAYR);
#ifdef YC_LINEAR
	ctrl_outl(Y_BUFF + Y_SIZE + (VMEM_SIZE * fn), VIO3_CDACR);
#else
	ctrl_outl(C_BUFF + (VMEM_SIZE * fn), VIO3_CDACR);
#endif
#endif

	// data swap
	ctrl_outl(0x00000053, VIO3_DSWPR);
	
	ctrl_outl((ctrl_inl(VIO3_VIPSR) & ~0x00000010), VIO3_VIPSR);	// VIP

//	ctrl_outl(0x00000204, VIO3_CAPCR);	// RGB only, VDIE Int Enable
//	ctrl_outl(0x00000102, VIO3_CAPCR);	// RGB,YCrCb, VD Int Enable
//	ctrl_outl(0x000001ff, VIO3_CAPCR);	// RGB,YCrCb, All Int Enable
//	ctrl_outl(0x00000202, VIO3_CAPCR);	// RGB only, VD Int Enable

//	ctrl_outl(0x000001fe, VIO3_CAPCR);	// RGB,YCrCb, widthout HD Int Enable
	ctrl_outl(0x000000fe, VIO3_CAPCR);	// YCrCb only, widthout HD Int Enable

	ctrl_outl(0x00000001, VIO3_CAPSR);	// Start
	
	ctrl_outl(ctrl_inl(VIO3_EVTCR) & ~0x08000f01, VIO3_EVTCR);	// VD,HD <- 0
}


static void start_yuv2rgb(unsigned short yuv_fn, unsigned short rgb_fn,
						  unsigned char rot)
{
	D(printk("VIO3 YUV(%d) -> RGB(%d)\n", yuv_fn, rgb_fn));

	// data swap
	ctrl_outl(0x00000013, VIO3_DSWPR);
//	ctrl_outl(0x00000033, VIO3_DSWPR);
	
	ctrl_outl(((M_HEIGHT<<16) + M_WIDTH) & 0x07fc07fc, VIO3_VFSZR);

	ctrl_outl(rot & 0x03, VIO3_VROTR);
	
	// YUV input address
	ctrl_outl(Y_BUFF + (VMEM_SIZE * yuv_fn), VIO3_VSAGR);
#ifdef YC_LINEAR
	ctrl_outl(Y_BUFF + Y_SIZE + (VMEM_SIZE * yuv_fn), VIO3_VSABR);
#else
	ctrl_outl(C_BUFF + (VMEM_SIZE * yuv_fn), VIO3_VSABR);
#endif

	// RGB output address
	switch(rot) {
	case 0x01:
		ctrl_outl(RGB_BUFF+(M_WIDTH*2) -4 + (VMEM_SIZE * rgb_fn), VIO3_VDAYR);
		break;
		
	default:
		ctrl_outl(RGB_BUFF + (VMEM_SIZE * rgb_fn), VIO3_VDAYR);
		break;
	}
	
	
	ctrl_outl((ctrl_inl(VIO3_VIPSR) | 0x00000010), VIO3_VIPSR);

	ctrl_outl(0x000000fe, VIO3_CAPCR);	// RGB,YCrCb, widthout HD Int Enable

	// CDREP=1(4:2:0),TE=1,RY=0
	ctrl_outl(0x00060d21, VIO3_VTRCR);
	
	// Conversion start
	ctrl_outl((ctrl_inl(VIO3_VIPSR) | 0x00000001), VIO3_VIPSR);
	
	ctrl_outl(ctrl_inl(VIO3_EVTCR) & ~0x08000f01, VIO3_EVTCR);
	// VD,HD <- 0
}


void vio_int_hdr(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned long	fact;

	// Clear interrupt factor
	fact = ctrl_inl(VIO3_EVTCR);

	//****************************************
	// FPC()
	//****************************************
	if(fact & 0x08000000) {
		ctrl_outl(~fact, VIO3_EVTCR);
		
		D(printk("VIO3 FPC int\n"));
	}
	
	//****************************************
	// NVD( VD nothing interrupt )
	//****************************************
	if(fact & 0x04000000) {
		ctrl_outl(~fact, VIO3_EVTCR);

		D(printk("VIO3 NVD int\n"));
	}
	
	//****************************************
	// NHD( HD nothing interrupt )
	//****************************************
	if(fact & 0x02000000) {
		ctrl_outl(~fact, VIO3_EVTCR);

		D(printk("VIO3 NHD int\n"));
	}
	
	//****************************************
	// CBOF( CIA buffer overflow interrupt )
	//****************************************
	if(fact & 0x01000000) {
		ctrl_outl(~fact, VIO3_EVTCR);

		D(printk("VIO3 CBOF int\n"));
	}
	
	//****************************************
	// RW( RAM access error interrupt )
	//****************************************
	if(fact & 0x00020000) {
		ctrl_outl(~fact, VIO3_EVTCR);

		D(printk("VIO3 RW int\n"));
	}
	
	//****************************************
	// VBOF( VIP buffer overflow interrupt )
	//****************************************
	if(fact & 0x00010000) {
		ctrl_outl(~fact, VIO3_EVTCR);

		D(printk("VIO3 VBOF int\n"));
	}
	
	//****************************************
	// CPE( Capture end interrupt )
	//****************************************
	if(fact & 0x00000400) {
		ctrl_outl(~fact, VIO3_EVTCR);

		wake_up(&wq);
		
		D(printk("VIO3 Cap END\n"));
	}

	//****************************************
	// RVS(  )
	//****************************************
	if(fact & 0x00000800) {
		ctrl_outl(~fact, VIO3_EVTCR);

		D(printk("VIO3 RVS int\n"));
	}
	
	//****************************************
	// VD( VD interrupt )
	//****************************************
	if(fact & 0x00000200) {
		ctrl_outl(~fact, VIO3_EVTCR);

		D(printk("VIO3 VD int\n"));
	}
	
	//****************************************
	// HD( HD interrupt )
	//****************************************
	if(fact & 0x00000100){
		ctrl_outl(~fact, VIO3_EVTCR);
	}
	
	//****************************************
	// CVE( VIP end interrupt )
	//****************************************
	if(fact & 0x00000001) {

		ctrl_outl(~fact, VIO3_EVTCR);

//		ctrl_outl((ctrl_inl(VIO3_VTRCR)& ~0x00000001), VIO3_VTRCR);// CVIE <- 0

//		ctrl_outl((ctrl_inl(VIO3_CAPCR) & ~0x001100FF), VIO3_CAPCR);

//		wake_up_interruptible_sync(&wq);
//		wake_up_interruptible(&wq);
		wake_up(&wq);
		
		D(printk("VIO3 VIP END int\n"));
	}
}


static void open_vio(void)
{
	// CIA setup
	ctrl_outl(M_WIDTH ,VIO3_VVSWR);
	ctrl_outl(((M_HEIGHT<<16) + M_WIDTH) & 0x07fc07fc ,VIO3_VVSSR);

	// VIP setup
	ctrl_outl(((M_HEIGHT<<16) + M_WIDTH) & 0x07fc07fc, VIO3_VFSZR);
	ctrl_outl(((RESIZERATIO_1_1<<16) + RESIZERATIO_1_1), VIO3_VFLCR);
	ctrl_outl(M_WIDTH*2, VIO3_VDWDR);
	
	ctrl_outw(ctrl_inw(IPRE) | (VIO_INTLVL<<8), IPRE);	// Set priority
	request_irq(VIO_IRQ, vio_int_hdr, SA_INTERRUPT, "vio", NULL);

	ctrl_outb(0x10, IMCR1);					// interrupt mask clear
}

static void release_vio(void)
{
	ctrl_outw(ctrl_inw(IPRE) & ~(VIO_INTLVL<<8), IPRE);	// Reset priority
	ctrl_outb(0x10, IMR1);								// interrupt mask
	
	free_irq(VIO_IRQ, NULL);
}


int eof = 0;

/*
  open
*/
static int vio3_open(struct inode *inode, struct file *filp)
{
	D(printk("<1>VIO3 Open\n"));

	eof = 0;

	open_vio();

	set_camera();

	return 0;
}

/*
  release
*/
static int vio3_release(struct inode *inode, struct file *filp)
{
	D(printk("<1>VIO3 Release\n"));

	release_vio();

	return 0;
}


/*
  read
*/
static int vio3_read(struct file *filp, char *buf, size_t count, loff_t *ppos)
{
	D(volatile unsigned short *p = (volatile unsigned short *)PA_LED);
	int err;
	
	D(printk("<1>VIO3 Read\n"));

//	if(eof != 0) {
//		return 0;
//	}

	D(*p = 0xff00);	// All LED on
	
	start_capture(0x00000000);
	interruptible_sleep_on(&wq);	// !!!
//	wait_event_interruptible(wq, 1);	// !!!

	start_yuv2rgb(0, 0, 0x01);
	interruptible_sleep_on(&wq);	// !!!
	
	D(*p = 0x0000);	// All LED off
	
	err = copy_to_user(buf, "ok", 3);

	if(err != 0) {
		return -EFAULT;
	}
	eof = 1;
	
	return 3;
}


/*
  ioctl
*/
static int vio3_ioctl(struct inode *inode, struct file *filp,
					  unsigned int cmd, unsigned long arg)
{
	D(printk("<1>VIO3 ioctl\n"));
	
	switch(cmd) {
	case VIOCTL_CAPTURE:
		D(printk("VIOCTL_CAPTURE\n"));
		start_capture(arg);
		interruptible_sleep_on(&wq);
		break;

	case VIOCTL_YUV2RGB:
		D(printk("VIOCTL_YUV2RGB\n"));
		start_yuv2rgb(arg>>16, arg & 0xffff, 0);
		interruptible_sleep_on(&wq);
		break;

	default:
		D(printk("vio: unknown ioctl 0x%x\n", cmd));
		return -EINVAL;
	}
	
	return 0;
}


/*
  mmap
*/
static int vio3_mmap(struct file *filp, struct vm_area_struct *vma)
{
	unsigned long offset = (unsigned long)VIOMEM_BASE;

	if(offset >= __pa(VIOMEM_BASE) || (filp->f_flags & O_SYNC)) {
		vma->vm_flags |= VM_IO;
	}
	vma->vm_flags |= VM_RESERVED;

	if(remap_page_range(vma->vm_start, offset, vma->vm_end-vma->vm_start,
						vma->vm_page_prot)) {
		return -EAGAIN;
	}

	return 0;
}


static struct file_operations vio3_fops = {
	NULL,
	NULL,
	vio3_read,
	NULL,
	NULL,
	NULL,
	vio3_ioctl,
	vio3_mmap,
	vio3_open,
	NULL,
	vio3_release,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

//static int vio3_major = 254;
static int vio3_major = 0;

static int __init init_vio(void)
{
	int result;

	result = register_chrdev(vio3_major, "vio", &vio3_fops);
	if(result < 0) {
		printk(KERN_WARNING "vio: can't get major %d\n", vio3_major);
		return result;
	}
	if(vio3_major == 0) {
		vio3_major = result;
	}

	init_vio3();
	
	D(printk("<1>VIO3 Load\n"));
	
	return 0;
}

static void __exit cleanup_vio(void)
{
	unregister_chrdev(vio3_major, "vio");
	
	D(printk("<1>VIO3 Cleanup\n"));
}

MODULE_LICENSE("GPL");
module_init(init_vio);
module_exit(cleanup_vio);
