/*
 * linux/drivers/char/pnx0105_wdt.c
 *
 * Watchdog driver for PNX0105 board
 *
 * Author: MontaVista Software, Inc.
 *         <gdavis@mvista.com> or <source@mvista.com>
 *         Vitaly Wool <vwool@ru.mvista.com>
 *         Adapted for PNX0105
 *
 * 2003 (c) MontaVista Software, Inc. This file is licensed under the
 * terms of the GNU General Public License version 2. This program is
 * licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 * History:
 *
 * 20030527: George G. Davis <gdavis@mvista.com>
 *	Initially based on linux-2.4.19-rmk7-pxa1/drivers/char/sa1100_wdt.c
 *	(c) Copyright 2000 Oleg Drokin <green@crimea.edu>
 *	Based on SoftDog driver by Alan Cox <alan@redhat.com>
 *
 * TODO:
 * 1. Need to disable watchdog when entering chip idle mode.
 */

#include <linux/module.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/reboot.h>
#include <linux/smp_lock.h>
#include <linux/init.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/bitops.h>

/* Watchdog timer registers definition */
#define PNX0105_WDOG_IR          (IO_ADDRESS(WDOG_BASE) + 0x00)
#define PNX0105_WDOG_TCR         (IO_ADDRESS(WDOG_BASE) + 0x04)
#define PNX0105_WDOG_TC          (IO_ADDRESS(WDOG_BASE) + 0x08)
#define PNX0105_WDOG_PR          (IO_ADDRESS(WDOG_BASE) + 0x0C)
#define PNX0105_WDOG_PC          (IO_ADDRESS(WDOG_BASE) + 0x10)
#define PNX0105_WDOG_MCR         (IO_ADDRESS(WDOG_BASE) + 0x14)
#define PNX0105_WDOG_MR0         (IO_ADDRESS(WDOG_BASE) + 0x18)
#define PNX0105_WDOG_MR1         (IO_ADDRESS(WDOG_BASE) + 0x1C)
#define PNX0105_WDOG_EMR         (IO_ADDRESS(WDOG_BASE) + 0x3C)

#define TIMER_MARGIN	19	/* Default is 19 seconds */
#define TIMER_RATE	(12000000/16)

static int timer_margin;	/* in seconds */
static int pre_margin;
static long pnx0105_wdt_users;

static void
pnx0105_wdt_ping(void)
{
	pre_margin = TIMER_RATE * timer_margin;
	outl(2, PNX0105_WDOG_TCR); /* disable and reset WD timer */
  	outl(0x80, PNX0105_WDOG_EMR);
	outl(0x38, PNX0105_WDOG_MCR);
	outl(pre_margin, PNX0105_WDOG_MR1);
	outl(1, PNX0105_WDOG_TCR); /* enable WD timer */
}

/*
 *	Allow only one person to hold it open
 */

static int
pnx0105_wdt_open(struct inode *inode, struct file *file)
{
	if (test_and_set_bit(1, &pnx0105_wdt_users))
		return -EBUSY;

	outl(1, PNX0105_WDOG_TCR); /* enable WD timer */
	outl(0xf, PNX0105_WDOG_PR);
	outl(1, PNX0105_WDOG_PC);
	timer_margin = TIMER_MARGIN;	/* default timeout */

	return 0;
}

static int
pnx0105_wdt_release(struct inode *inode, struct file *file)
{
	/*
	 *      Shut off the timer.
	 */
	pnx0105_wdt_users = 0;
	return 0;
}

static ssize_t
pnx0105_wdt_write(struct file *file, const char *data, size_t len, loff_t * ppos)
{
	/*  Can't seek (pwrite) on this device  */
	if (ppos != &file->f_pos)
		return -ESPIPE;

	/* Refresh LOAD_TIME. */
	if (len) {
		pnx0105_wdt_ping();
		return 1;
	}
	return 0;
}

static int
pnx0105_wdt_ioctl(struct inode *inode, struct file *file,
	       unsigned int cmd, unsigned long arg)
{
	int new_margin;
	static struct watchdog_info ident = {
		.identity		= "PNX0105 Watchdog",
		.options		= WDIOF_SETTIMEOUT,
		.firmware_version	= 0,
	};

	switch (cmd) {
	default:
		return -ENOIOCTLCMD;
	case WDIOC_GETSUPPORT:
		return copy_to_user((struct watchdog_info *) arg, &ident,
				    sizeof (ident));
	case WDIOC_GETSTATUS:
		return put_user(0, (int *) arg);
	case WDIOC_GETBOOTSTATUS:
		return put_user(0, (int *) arg);
	case WDIOC_KEEPALIVE:
		pnx0105_wdt_ping();
		return 0;
	case WDIOC_SETTIMEOUT:
		if (get_user(new_margin, (int *) arg))
			return -EFAULT;
		if (new_margin < 1 || new_margin > 19)
			timer_margin = TIMER_MARGIN;	/* default timeout */
		else
			timer_margin = new_margin;
		pnx0105_wdt_ping();
		/* Fall */
	case WDIOC_GETTIMEOUT:
		return put_user(timer_margin, (int *) arg);
	}
}

static struct file_operations pnx0105_wdt_fops = {
	.owner		= THIS_MODULE,
	.write		= pnx0105_wdt_write,
	.ioctl		= pnx0105_wdt_ioctl,
	.open		= pnx0105_wdt_open,
	.release	= pnx0105_wdt_release,
};

static struct miscdevice pnx0105_wdt_miscdev = {
	.minor	= WATCHDOG_MINOR,
	.name	= "watchdog",
	.fops	= &pnx0105_wdt_fops
};

static int __init
pnx0105_wdt_init(void)
{
	int ret;

	ret = misc_register(&pnx0105_wdt_miscdev);

	if (ret)
		return ret;

	if (timer_margin < 1 || timer_margin > 19)
		timer_margin = 19;

	printk(KERN_INFO "%s: PNX0105 Watchdog Timer: timer margin %d sec\n",
	       pnx0105_wdt_miscdev.name, timer_margin);

	return 0;
}

static void __exit
pnx0105_wdt_exit(void)
{
	misc_deregister(&pnx0105_wdt_miscdev);
}

module_init(pnx0105_wdt_init);
module_exit(pnx0105_wdt_exit);

MODULE_AUTHOR("Vitaly Wool");
MODULE_LICENSE("GPL");
MODULE_PARM(timer_margin, "i");
EXPORT_NO_SYMBOLS;
