/*******************************************************************************
 * (C) Copyright Koninklijke Philips Electronics NV 2004. All rights reserved.
 *
 * You can redistribute and/or modify this software under the terms of
 * version 2 of the GNU General Public License as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *******************************************************************************/
/*------------------------------------------------------------------------------
  Standard include files:
  ------------------------------------------------------------------------------*/
#include "vhtypes.h"

#define PH_OK 0
typedef unsigned int phErrorCode_t; 
/*------------------------------------------------------------------------------
  Project include files:
  ------------------------------------------------------------------------------*/
#include "adc/phadc.h"

/*------------------------------------------------------------------------------
  Adaptation to Linux:
  ------------------------------------------------------------------------------*/
#include <linux/config.h>
#include <linux/spinlock.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/mm.h>
#include <linux/signal.h>
#include <linux/init.h>
#include <linux/kbd_ll.h>
#include <linux/delay.h>
#include <linux/random.h>
#include <linux/poll.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/kbd_kern.h>
#include <linux/smp_lock.h>
#include <linux/timer.h>

#include <asm/keyboard.h>
#include <asm/bitops.h>
#include <asm/uaccess.h>
#include <asm/irq.h>
#include <asm/system.h>

#include <asm/io.h>

/*------------------------------------------------------------------------------
  Local Types and defines:
  ------------------------------------------------------------------------------*/
#ifdef NDEBUG
#define MLPP_PRINTK(...) do {} while(0)
#else
#define MLPP_PRINTK(...) printk(__VA_ARGS__)
#endif /* NDEBUG */

#define KBD_NO_DATA     (-1)
#define KBD_KEYUP       (0x80)
#define KEY_ACCURACY     0x30

#define KBD_CHANNELS     (2)
#define KBD_BUTTONS      (6)
#define NO_B            (-1)
#define PH_B05          0x30
#define PH_B06          0x2e
#define PH_B07          0x32
#define PH_B08          0x13
#define PH_B09          0x20
#define PH_B10          0x19
#define PH_B11          0x1f
#define PH_B12          0x50
#define PH_B13          0x48
#define PH_B14          0x4d
#define PH_B15          0x4b
#define PH_RET          0x1c /* RETURN */

/*------------------------------------------------------------------------------
  Global variables:
  ------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Static Variables:
  ------------------------------------------------------------------------------*/
/* For test purpose, PH_09 is not used and is replaced by PH_RET, which is used as Return key */
static unsigned char ph_key_map[KBD_CHANNELS][KBD_BUTTONS] = {
  {PH_B06, PH_B08, PH_B10, PH_B12, PH_B14, NO_B},
  {PH_B05, PH_B07, PH_B09/*PH_RET*/, PH_B11, PH_B13, PH_B15}
};

static unsigned char ph_keys[128] = {
  0, 0, 0, 0, 0, 0, 0, 0,			      /* 0x00-0x07 */
  0, 0, 0, 0, 0, 0, 0, 0,			      /* 0x08-0x0f */
  0, 0, 0, PH_B08, 0, 0, 0, 0,			      /* 0x10-0x17 */
  0, PH_B10, 0, 0, PH_RET, 0, 0, PH_B11,       	      /* 0x18-0x1f */
  PH_B09, 0, 0, 0, 0, 0, 0, 0,			      /* 0x20-0x27 */
  0, 0, 0, 0, 0, 0, PH_B06, 0,			      /* 0x28-0x2f */
  PH_B05, 0, PH_B07, 0, 0, 0, 0, 0,	              /* 0x30-0x37 */
  0, 0, 0, 0, 0, 0, 0, 0,	                      /* 0x38-0x3f */
  0, 0, 0, 0, 0, 0, 0, 0,	                      /* 0x40-0x47 */
  PH_B13, 0, 0, PH_B15, 0, PH_B14, 0, 0,              /* 0x48-0x4f */
  PH_B12, 0, 0, 0, 0, 0, 0, 0,	                      /* 0x50-0x57 */
  0, 0, 0, 0, 0, 0, 0, 0,	                      /* 0x58-0x5f */
  0, 0, 0, 0, 0, 0, 0, 0,			      /* 0x60-0x67 */
  0, 0, 0, 0, 0, 0, 0, 0,		              /* 0x68-0x6f */
  0, 0, 0, 0, 0, 0, 0, 0,			      /* 0x70-0x77 */
  0, 0, 0, 0, 0, 0, 0, 0			      /* 0x78-0x7f */
};

static unsigned char ph_handle_kbd_event(void);
static unsigned char ph_kbd_read_input(void);
static int ph_read_input(unsigned char buttons[KBD_CHANNELS][KBD_BUTTONS]);

static spinlock_t kbd_controller_lock = SPIN_LOCK_UNLOCKED;
static struct timer_list ph_kbd_timer;

/**
* @brief Every analogue key is assigned to a certain maximum ADC value
*
* Every analogue key is assigned to a certain maximum ADC value.
*/
typedef struct phKeyAssign_st
{
    unsigned long MaxAdcLevel;                     /* Max ADC value for key x */
    unsigned long KeyBitmask;                      /* Bitmask       for key x */
} phKeyAssign_st_t;

/**
* @brief Array contains the maximum ADC value for each possible analogue key 
* on ADC channel 0.
*
* Array contains the maximum ADC value for each possible analogue key 
* on ADC channel 0.
*
* @see phKeyAssign_st.
*
*/
static phKeyAssign_st_t saKeybAdcValuesChannel0[] = 
{
  {0x004e, PH_B06},            
  {0x0101, PH_B08},       
  {0x01bf, PH_B10},          
  {0x027d, PH_B12},   
  {0x0371, PH_B14}                   
};

/**
* @brief Array contains the maximum ADC value for each possible analogue key 
* on ADC channel 1.
*
* Array contains the maximum ADC value for each possible analogue key 
* on ADC channel 1.
*
* @see phKeyAssign_st.
*
*/
static phKeyAssign_st_t saKeybAdcValuesChannel1[] = 
{
  {0x0042, PH_B05},            
  {0x00c1, PH_B07},       
  {0x0154, PH_B09},          
  {0x01eb, PH_B11},   
  {0x028a, PH_B13},          
  {0x0374, PH_B15}         
};

/*------------------------------------------------------------------------------
  Static prototypes:
  ------------------------------------------------------------------------------*/
static inline long int LABS(long int j)
{
  if (j >= 0) return j;
  else return (-j);
}

static int ph_read_input(unsigned char buttons[KBD_CHANNELS][KBD_BUTTONS])
{
  phErrorCode_t error_status;
  UInt32 adcValue0;
  static UInt32 prevadcValue0 = 0x3ff;
  UInt32 adcValue1;
  static UInt32 prevadcValue1 = 0x3ff;
  UInt32 j;
  UInt32 MaxKeys0 =  sizeof(saKeybAdcValuesChannel0)/sizeof(phKeyAssign_st_t);
  UInt32 MaxKeys1 =  sizeof(saKeybAdcValuesChannel1)/sizeof(phKeyAssign_st_t);
  
/* statistic */
#ifdef KBD_STATISTIC
  static UInt32 avg_0 = 0;
  static UInt32 avg_1 = 0;
  static UInt32 N0 = 0;
  static UInt32 N1 = 0;
  UInt32 deb_0 = 0;
  UInt32 deb_1 = 0;
#endif

  error_status = PH_OK;
 
  /* Read out the first ADC keyboard channel (Continuous Mode) */
  error_status = phAdc_ContinuousRead(
				      PH_ADCCHANNEL_0,
				      &adcValue0); 
  MLPP_PRINTK("Read Ch0, %04x\n ",(unsigned int)adcValue0);
  if(error_status != PH_OK)
    return(error_status);
      
  /* Read out the second ADC keyboard channel (Continuous Mode) */
  error_status = phAdc_ContinuousRead(
				      PH_ADCCHANNEL_1,
				      &adcValue1); 
  MLPP_PRINTK("Read Ch1, %04x\n ",(unsigned int)adcValue1);
  if(error_status != PH_OK)
    return(error_status);
   
 
  /* Process ADC value 0 */
  /* Debouncing and filtering */
  if (LABS(prevadcValue0 - adcValue0) < KEY_ACCURACY){
    for(j = 0; j < MaxKeys0; j++)
      {
	if (adcValue0 < saKeybAdcValuesChannel0[j].MaxAdcLevel)
	  {
	    buttons[0][j] = 1;
	    break;
	  }
      }
  } 
  /* else, use previous value */
  else{
    for(j = 0; j < MaxKeys0; j++)
      {
	if (prevadcValue0 < saKeybAdcValuesChannel0[j].MaxAdcLevel)
	  {
	    buttons[0][j] = 1;
	    break;
	  }
      }
  }
#ifdef KBD_STATISTIC
  deb_0 = LABS(prevadcValue0 - adcValue0);
  MLPP_PRINTK("Debouncing: |%x - %x| = %x\n",(unsigned int)prevadcValue0,
	 (unsigned int)adcValue0,(unsigned int)LABS(prevadcValue0 - adcValue0));
  avg_0 = (N0*avg_0 + deb_0)/(N0+1);
  N0++;
  printk("avg_0 = %u\n",(unsigned long)avg_0);
#endif
  prevadcValue0 = adcValue0;
  
  /* Process ADC value 1 */
  if (LABS(prevadcValue1 - adcValue1) < KEY_ACCURACY){
    for(j = 0; j < MaxKeys1; j++)
      {
	if(adcValue1 < saKeybAdcValuesChannel1[j].MaxAdcLevel)
	  {
	    buttons[1][j] = 1;
	    break;
	  }
      }
  }
  /* else, use previous value */
  else{
    for(j = 0; j < MaxKeys1; j++)
      {
	if(prevadcValue1 < saKeybAdcValuesChannel1[j].MaxAdcLevel)
	  {
	    buttons[1][j] = 1;
	    break;
	  }
      }  
  }
#ifdef KBD_STATISTIC
  deb_1 = LABS(prevadcValue1 - adcValue1);
  MLPP_PRINTK("Debouncing: |%x - %x| = %x\n",(unsigned int)prevadcValue1,(unsigned int)adcValue1,(unsigned int)LABS(prevadcValue1 - adcValue1));
  avg_1 = (N1*avg_1 + deb_1)/(N1+1);
  N1++;
  printk("avg_1 = %u\n",(unsigned long)avg_1);
#endif
  prevadcValue1 = adcValue1;

  return(error_status);    
}

static unsigned char ph_kbd_read_input(void)
{
  int i,j;
  phErrorCode_t error_status=PH_OK;
  static unsigned char oldcodes[KBD_CHANNELS][KBD_BUTTONS]={{0,0,0,0,0,0},
							  {0,0,0,0,0,0}};
  unsigned char newcodes[KBD_CHANNELS][KBD_BUTTONS];
  
  memset(newcodes, 0, sizeof(unsigned char) * (KBD_CHANNELS * KBD_BUTTONS));
  MLPP_PRINTK("In ph_kbd_read_input\n ");
  
  error_status = ph_read_input(newcodes);
  if (error_status == PH_OK){
    for (i=0; i<KBD_CHANNELS; i++){
      for (j=0; j<KBD_BUTTONS; j++){
	if (oldcodes[i][j] != newcodes[i][j]){
	  //Key Pressed (Down) = 1
	  if (newcodes[i][j]){
	    oldcodes[i][j]=1;
	    MLPP_PRINTK("Key pressed %d, %d\n ",i,j);
	    return (ph_key_map[i][j]);
	  }
	  //Key Released (Up) = 0
	  else{
	    oldcodes[i][j]=0;
	    MLPP_PRINTK("Key released %d, %d\n ",i,j);
	    return (ph_key_map[i][j] | KBD_KEYUP);
	  }
	}
      }
    }
  }
  else {
    switch (error_status) 
      {
      case PH_ADC_ERR_NOT_INITIALIZED:
      case PH_ADC_ERR_CH_NOTSELECTED: 
      case PH_ADC_ERR_WRONG_ADC_MODE:
	printk("ADC Error %x", error_status);
	break;
      }
  }    
  return (unsigned char) (KBD_NO_DATA);
}

static inline void ph_handle_keyboard_event(unsigned char scancode)
{
  MLPP_PRINTK("In ph_handle_keyboard_event\n ");

  if(scancode != (unsigned char)(KBD_NO_DATA)){
    MLPP_PRINTK("Scancode %x\n",scancode);
    handle_scancode(scancode, !(scancode & KBD_KEYUP));
    tasklet_schedule(&keyboard_tasklet);
  }
}

static unsigned char ph_handle_kbd_event(void)
{
   unsigned char scancode;
   MLPP_PRINTK("In ph_handle_kbd_event\n ");
   scancode = ph_kbd_read_input();
   ph_handle_keyboard_event(scancode);

   return 0;
}

/* Handle the automatic interrupts handled by the timer */
static void ph_keyboard_interrupt(unsigned long foo)
{
   spin_lock_irq(&kbd_controller_lock);
   ph_handle_kbd_event();
   spin_unlock_irq(&kbd_controller_lock);

   ph_kbd_timer.expires = 5 + jiffies;
   ph_kbd_timer.data = 0x00000000;
   ph_kbd_timer.function = (void(*)(unsigned long))&ph_keyboard_interrupt;

   add_timer(&ph_kbd_timer);
}

void ph_kbd_leds(unsigned char leds)
{
}

char ph_kbd_unexpected_up(unsigned char keycode)
{
  return 0;
}

int ph_kbd_getkeycode(unsigned int scancode)
{
  MLPP_PRINTK("In ph_kbd_getkeycode\n");
  if (scancode >= 0 && scancode < 128)
    return ph_keys[scancode];
  else {
    printk("Wrong scancode %x\n", scancode);
    return 0;
  }
}

int ph_kbd_setkeycode(unsigned int scancode, unsigned int keycode)
{
  MLPP_PRINTK("In ph_kbd_setkeycode\n");
  if (scancode >= 0 && scancode < 128)
    ph_keys[scancode] = keycode;
  else 
    printk("Wrong scancode %x\n", scancode);
  return 0;
}

int ph_kbd_translate(unsigned char scancode, unsigned char *keycode, char raw_mode)
{
  scancode &= 0x7f;
  *keycode = ph_keys[scancode];
  return 1;
}

void ph_kbd_init_hw(void)
{
  phErrorCode_t error_status = PH_OK;
  unsigned int chSelectBitmask;

  printk("Starting PNX0105 Keyboard Driver... \n");
 
  k_setkeycode	  = ph_kbd_setkeycode;
  k_getkeycode    = ph_kbd_getkeycode;
  k_translate     = ph_kbd_translate;
  k_unexpected_up = ph_kbd_unexpected_up;
  k_leds          = ph_kbd_leds;

  /*Initialise the ADC for `Continuous mode` scanning (Cont mode) */
  error_status = phAdc_Init(PH_ADC_CONTIN_MODE);      

  if(error_status != PH_OK){
    printk("PNX0105 Keyboard Driver Error in phAdc_Init... \n");
    goto err;
  }

  /* Set up channel0 and channel1 for continuous scan (Cont mode) */
  chSelectBitmask = 0;
  chSelectBitmask = PH_ADC_MSK_CHANNEL_0 | PH_ADC_MSK_CHANNEL_1;

  error_status = phAdc_ContinuousSetup(PH_ADCVREF_0,
				       PH_ADCRESOLUTION_10BITS,
				       chSelectBitmask);
      
  if(error_status != PH_OK){
    printk("PNX0105 Keyboard Driver Error in phAdc_ContinuousSetup... \n");
    goto err;
  }
  
  ph_kbd_timer.expires = 80 + jiffies;
  ph_kbd_timer.data = 0x00000000;
  ph_kbd_timer.function = (void(*)(unsigned long))&ph_keyboard_interrupt;

  add_timer(&ph_kbd_timer);

  printk("... PNX0105 Keyboard Driver started\r\n");

 err:
  if (error_status != PH_OK){
    switch(error_status)
      { 
      case PH_ADC_ERR_RESOURCE_OWNED: 
	break;
      case PH_ADC_ERR_NOT_INITIALIZED:
	break;
      case PH_ADC_ERR_WRONG_ADC_MODE: 
      case PH_ADC_ERR_ALREADY_SETUP:
	phAdc_DeInit();
	break;
      }
  }
  /* else : Do nothing */
}
