/*******************************************************************************************************

	mx21-keyp.c
	MX2ADS on-board keypad driver

	Author: MontaVista Software, Inc. <source@mvista.com>
	2004 (c) MontaVista Software, Inc. This file is licensed under
	the terms of the GNU General Public License version 2. This program
	is licensed "as is" without any warranty of any kind, whether express
	or implied.
	
*******************************************************************************************************/
#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <asm/uaccess.h>	/*  get_user,copy_to_user */
#include <linux/miscdevice.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/string.h>
#include <linux/init.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <linux/errno.h>
#include <linux/tqueue.h>
#include <linux/wait.h>

#include <asm/irq.h>
#include <asm/arch/hardware.h>
#include <asm/arch/irqs.h>
#include <asm/arch/gpio.h>
#include <linux/pm.h>

#include "mx21-keyp.h"

#define MODULE_NAME "kpp"

#define KPP_MX2_MASK			0x3f3f	/* it is 6*6 key matrix */
#define KPP_MX2_COL_NUM			6
#define KPP_MX2_ROW_NUM			6

#define kpp_row_enable(mask)	(KPP_KPCR|(mask))
#define kpp_col_enable(mask)	(KPP_KPCR|(mask<<8))

#define KPP_KPPEN_BIT_MASK		0x0400	/* kpp clock gating enable */
#define KPP_KRIE_BIT_MASK		0x0200	/* kpp release interrupt enable */
#define KPP_KDIE_BIT_MASK		0x0100	/* kpp depress interrupt enable */
#define KPP_KRSS_BIT_MASK		0x0008	/* kpp release synchronizer set */
#define KPP_KDSC_BIT_MASK		0x0004	/* kpp depress synchronizer clear */
#define KPP_KPKR_BIT_MASK		0x0002	/* kpp key release */
#define KPP_KPKD_BIT_MASK		0x0001	/* kpp key depress */

#define KPP_IRQ					21

static int g_kpp_major = 0;
static devfs_handle_t g_devfs_handle;

static u8 kpp_pressed = 0;

static struct timer_list kpp_timer;
static u8 kpp_timer_status;
wait_queue_head_t g_mx21_kpp_wait;
spinlock_t kpp_lock;
static u32 g_key_code[2];
static u32 old_key_code[2];
static u16 rptr, wptr;
key_code_t *g_mx21_kpp_buffer;
struct pm_dev *g_kpp_pm;
static int g_kpp_status;
#define KPP_OPEN_STATUS		0x0001
#define KPP_SUSPEND_STATUS	0x0002

#define NODATA() 		(rptr == wptr)
#define BUFFLENTH		8
#define BUFFSIZE		(BUFFLENTH*sizeof(key_code_t))
#define NEXTITEM(i)		{i=(i==BUFFLENTH-1)?0:i+1;}
#define GETNEXTIEM(i)	((i==BUFFLENTH-1)?0:i+1)
#define DATASIZE()		((wptr<rptr)?(BUFFLENTH-rptr):(wptr-rptr))
#define DEPRESSED		0xff
#define RELEASED		0x0
static int
init_buf(void)
{
	g_mx21_kpp_buffer = (key_code_t *) kmalloc(BUFFSIZE, GFP_KERNEL);
	if (!g_mx21_kpp_buffer) {
		printk(KERN_ERR
		       "no enough kernel memory for spi data buffer\n");
		return -1;
	}

	rptr = wptr = 0;
	return 1;
}

static void
add_key_code(u32 low, u32 high, u32 flag)
{

	if (GETNEXTIEM(wptr) == rptr)
		return;
	g_mx21_kpp_buffer[wptr].low = low;
	g_mx21_kpp_buffer[wptr].high = high;

	g_mx21_kpp_buffer[wptr].status = flag;

	NEXTITEM(wptr);		/*  goto next wptr */
}
static key_code_t *
get_data(void)
{
	key_code_t *data;

	if (NODATA())
		return NULL;

	data = &(g_mx21_kpp_buffer[rptr]);

	pr_debug
	    ("*** Read - low: 0x%04x, high: 0x%04x, rptr: 0x%04x, wptr: 0x%04x\n",
	     data->low, data->high, (int) rptr, (int) wptr);

	NEXTITEM(rptr);
	return data;
}

static void
mx2ads_mask_irq(unsigned int irq)
{
	AITC_INTENABLEL &= ~(1 << irq);
}

static void
mx2ads_unmask_irq(unsigned int irq)
{
	AITC_INTENABLEL |= (1 << irq);
}

/*  Configure KPP setting */
void
kpp_hw_configure()
{
	u16 row, col;
	row = KPP_MX2_MASK & 0x00ff;
	col = KPP_MX2_MASK & 0xff00;

	/*  GPIO portE3467 as alternate function, since the UART2 has some conflict to Keypad */

	mx2_register_gpios(PORT_E, (1 << 3) |
			   (1 << 4) | (1 << 6) | (1 << 7), PRIMARY | SECONDARY);

	/*  enable clk gate of kpp */
	CRM_PCCR1 |= PCCR1_KPP_EN;
	KPP_KPSR |= KPP_KPPEN_BIT_MASK;

	/*  enable key */
	KPP_KPCR |= row;
	/* clear kpp data register */
	KPP_KPDR &= ~(col | row);
	/* configure col as output opendrain */
	KPP_KPCR |= col;
	KPP_KDDR |= col;
	/* configure row as input */
	KPP_KDDR &= ~row;
	/* Clear KPKD status flag and synchronizer chain */
	KPP_KPSR |= KPP_KDSC_BIT_MASK;
	KPP_KPSR |= KPP_KPKD_BIT_MASK;
	/* Set KDIE control bit, clear the KRIE control bit */
	KPP_KPSR |= KPP_KDIE_BIT_MASK;
	KPP_KPSR &= ~KPP_KRIE_BIT_MASK;

}

static void
kpp_irq_enable()
{
	/* clear key depress event */
	KPP_KPSR |= KPP_KPKD_BIT_MASK;
	/* clear key release event */
	KPP_KPSR |= KPP_KPKR_BIT_MASK;
	/* enable irq */

	mx2ads_unmask_irq(KPP_IRQ);
}

/* for col0, will have eight bit mask, though bit6,7 is not used by any key in current keyboard */
static void
keycode_convert(u16 scancode, u32 * p_key_code)
{
	u16 tempcol, colnum;
	u16 temprow;
	u16 tmp;
	int i;
	u32 code[2];
	tmp = 1;
	tempcol = (scancode & KPP_MX2_MASK & 0xff00) >> 8;
	temprow = ~(scancode & KPP_MX2_MASK & 0x00ff);
	code[0] = code[1] = 0;
	pr_debug("scancode = 0x%x ,temprow = 0x%x \n", scancode, temprow);
	if (tempcol & 0xf) {	/* col0~3 */
		for (i = 0; i < 4; i++) {
			if (tempcol & (1 << i)) {
				colnum = i;
				break;
			}
		}
		for (i = 0; i < 6; i++) {
			if (temprow & (1 << i))
				code[0] |= 1 << (colnum * 8 + i);
		}
	}
	if (tempcol & 0xf0) {	/* col4~7 */
		for (i = 4; i < 8; i++) {
			if (tempcol & (1 << i)) {
				colnum = i - 4;
				break;
			}
		}
		for (i = 0; i < 6; i++) {
			if (temprow & (1 << i))
				code[1] |= 1 << (colnum * 8 + i);
		}
	}
	*p_key_code |= code[0];
	p_key_code++;
	*p_key_code |= code[1];

}
static void
kpp_scan_matrix()
{
	u16 row, col;
	u16 iCol, scanMask;
	u16 keypressed;

	old_key_code[0] = g_key_code[0];
	old_key_code[1] = g_key_code[1];
	g_key_code[0] = g_key_code[1] = 0;
	row = KPP_MX2_MASK & 0x00ff;
	col = KPP_MX2_MASK & 0xff00;
	iCol = 0;
	scanMask = 0;
	kpp_pressed = 0;
	for (iCol = 0; iCol < KPP_MX2_COL_NUM; iCol++) {
		/* write 1's to KPDR[15:8] */
		KPP_KPDR |= col;
		udelay(1);
		/* configure columns as totem-pole outputs */
		KPP_KPCR &= ~col;
		/* configure columns as open-drain */
		KPP_KPCR |= col;
		/* write a single column to 0,others to 1 */
		scanMask = (1 << (iCol + 8));
		KPP_KPDR &= ~scanMask;
		/* sample row inputs and save data. */
		keypressed = (KPP_KPDR & 0x00ff) | scanMask;
		if ((KPP_KPDR & row) != row) {
			kpp_pressed = 1;
			keycode_convert(keypressed, g_key_code);
		}
		/* restore the colmask */
		KPP_KPDR |= scanMask;
	}
	KPP_KPDR &= ~col;
	pr_debug("g_key_code[0] = 0x%x g_key_code[1]=0x%x \n", g_key_code[0],
		 g_key_code[1]);
	if ((g_key_code[0] != old_key_code[0])
	    || (g_key_code[1] != old_key_code[1])) {
		if (kpp_pressed == 1)
			add_key_code(g_key_code[0], g_key_code[1], DEPRESSED);
		else
			add_key_code(g_key_code[0], g_key_code[1], RELEASED);
		wake_up_interruptible(&g_mx21_kpp_wait);

	}
}

static void
kpp_timer_callback(unsigned long data)
{
	if (kpp_timer_status != 0)
		mod_timer(&kpp_timer, jiffies + HZ / 8000 + 2 * HZ / 100);
	kpp_scan_matrix();
}
static void
kpp_isr(int irq, void *dev_id, struct pt_regs *regs)
{
	/* clear Key depress event */
	KPP_KPSR |= KPP_KPKD_BIT_MASK;

	/* clear key release event */
	KPP_KPSR |= KPP_KPKR_BIT_MASK;

	/* if first depress key */
	if ((kpp_timer_status == 0) && (kpp_pressed == 0)) {
		/* disable key depress interrupt */
		KPP_KPSR &= ~KPP_KDIE_BIT_MASK;
		/* enable key release interrupt */
		KPP_KPSR |= KPP_KRIE_BIT_MASK;
		/* Start timer */
		spin_lock_irq(&kpp_lock);
		kpp_timer.expires = jiffies + HZ / 8000 + 2 * HZ / 100;
		add_timer(&kpp_timer);
		kpp_timer_status = 1;
		spin_unlock_irq(&kpp_lock);
		return;
	}
	/* if get key release interrupt */
	else if ((kpp_timer_status != 0) && (kpp_pressed == 0)) {
		/* release timer */
		spin_lock_irq(&kpp_lock);
		kpp_timer_status = 0;
		del_timer(&kpp_timer);
		spin_unlock_irq(&kpp_lock);
		/* disable key release interrupt */
		KPP_KPSR &= ~KPP_KRIE_BIT_MASK;
		/* enable key depress interrupt */
		KPP_KPSR |= KPP_KDIE_BIT_MASK;
		return;
	}
}

/* *****************************************************************************
 * Function Name: kpp_read
 *
 * Input: 		filp	: the file
 * 			buf	: data buffer
 * 			count	: number of chars to be readed
 * 			l	: offset of file
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: read device driver
 *
 *****************************************************************************/

ssize_t
kpp_read(struct file * filp, char *buf, size_t count, loff_t * l)
{
	key_code_t keycode;
	key_code_t *p_keycode;
	int nonBlocking;

	unsigned long flags;
	int ret = 0;

	nonBlocking = filp->f_flags & O_NONBLOCK;

	if (nonBlocking) {
		if (!NODATA()) {
			save_flags(flags);
			cli();
			p_keycode = get_data();
			restore_flags(flags);

			keycode.low = p_keycode->low;
			keycode.high = p_keycode->high;
			keycode.status = p_keycode->status;
			__copy_to_user(buf, &keycode, sizeof (key_code_t));
			pr_debug("copy back \n");
			ret = sizeof (key_code_t);
			return ret;
		} else
			return -EINVAL;
	} else {
		while (1) {
			if (!NODATA()) {
				save_flags(flags);
				cli();
				p_keycode = get_data();
				restore_flags(flags);

				keycode.low = p_keycode->low;
				keycode.high = p_keycode->high;
				keycode.status = p_keycode->status;
				__copy_to_user(buf, &keycode,
					       sizeof (key_code_t));
				pr_debug("copy back \n");

				ret = sizeof (key_code_t);
				return ret;
			} else {
				interruptible_sleep_on(&g_mx21_kpp_wait);
				if (signal_pending(current))
					return -ERESTARTSYS;
			}
		}
	}

}

/* *****************************************************************************
 * Function Name: kpp_release
 *
 * Input: 		inode	:
 * 				filp	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: release resource when close the inode
 *
 *****************************************************************************/
int
kpp_release(struct inode *inode, struct file *filp)
{
	printk("kpp_release: ----\n");

	/* disable the kpp interrupt */
	mx2ads_mask_irq(KPP_IRQ);
	g_kpp_status &= ~KPP_OPEN_STATUS;
	MOD_DEC_USE_COUNT;
	return 0;
}

/* *****************************************************************************
 * Function Name: kpp_open
 *
 * Input: 		inode	:
 * 			filp	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: allocate resource when open the inode
 *
 *****************************************************************************/
int
kpp_open(struct inode *inode, struct file *filp)
{
	printk("kpp_open: ----\n");
	MOD_INC_USE_COUNT;

	/* enable ADC interrupt */
	kpp_hw_configure();
	kpp_irq_enable();
	g_kpp_status = KPP_OPEN_STATUS;
	return 0;
}

/* *****************************************************************************
 * Function Name: kpp_fasync
 *
 * Input: 		fd	:
 * 			filp	:
 * 			mode	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: provide fasync functionality for select system call
 *
 *****************************************************************************/
static int
kpp_fasync(int fd, struct file *filp, int mode)
{
	return 0;
}

/* *****************************************************************************
 * Function Name: kpp_ioctl
 *
 * Input: 		inode	:
 * 			filp	:
 * 			cmd	: command for ioctl
 * 			arg	: parameter for command
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: ioctl for this device driver
 *
 *****************************************************************************/
int
kpp_ioctl(struct inode *inode,
	  struct file *filp, unsigned int cmd, unsigned long arg)
{
	int ret = -EIO;
	int minor;

	return ret;
}

struct file_operations g_kpp_fops = {
	.open = kpp_open,
	.release = kpp_release,
	.read = kpp_read,
	.poll = NULL,
	.ioctl = kpp_ioctl,
	.fasync = kpp_fasync,
};
int
kpp_pm_handler(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	switch (rqst) {
	case PM_RESUME:
		if ((g_kpp_status & KPP_SUSPEND_STATUS) != 0) {
			/* enable clk */
			CRM_PCCR1 |= PCCR1_KPP_EN;
			g_kpp_status &= ~KPP_SUSPEND_STATUS;
		}
		break;
	case PM_SUSPEND:
		if ((g_kpp_status & KPP_OPEN_STATUS) != 0) {
			/* disable clk */
			CRM_PCCR1 &= ~PCCR1_KPP_EN;
			g_kpp_status |= KPP_SUSPEND_STATUS;
		}
		break;
	default:
		break;
	}
	return 0;
}

/* ******************************************************
*kpp init function
*Parameters:None
*Return	
*	0	indicates SUCCESS
* 	-1	indicates FAILURE
*Description:
********************************************************/
static signed short __init
kpp_init(void)
{
	int tmp;
	g_mx21_kpp_buffer = 0;
	init_timer(&kpp_timer);
	kpp_timer.function = kpp_timer_callback;
	g_kpp_major = devfs_register_chrdev(0, MODULE_NAME, &g_kpp_fops);
	if (g_kpp_major < 0) {
		pr_debug("%s driver: Unable to register driver\n", MODULE_NAME);
		return -ENODEV;
	}

	g_devfs_handle = devfs_register(NULL, MODULE_NAME, DEVFS_FL_DEFAULT,
					g_kpp_major, 0,
					S_IFCHR | S_IRUSR | S_IWUSR,
					&g_kpp_fops, NULL);

	tmp = request_irq(KPP_IRQ,
			  (void *) kpp_isr,
			  SA_INTERRUPT | SA_SHIRQ, MODULE_NAME, MODULE_NAME);
	if (tmp) {
		printk("kpp_init:cannot init major= %d irq=%d\n", g_kpp_major,
		       KPP_IRQ);
		devfs_unregister_chrdev(g_kpp_major, MODULE_NAME);
		devfs_unregister(g_devfs_handle);
		return -1;
	}

	if (-1 == init_buf()) {
		printk("kpp_init: cannot init kpp buffer, exit\n");

		free_irq(KPP_IRQ, MODULE_NAME);

		devfs_unregister_chrdev(g_kpp_major, MODULE_NAME);
		devfs_unregister(g_devfs_handle);
		return -1;
	}

	init_waitqueue_head(&g_mx21_kpp_wait);
	g_kpp_pm = pm_register(PM_SYS_DEV, PM_SYS_VGA, kpp_pm_handler);
	g_kpp_status = 0;

	pr_debug("Kpp Driver 0.4.0\n");
	return 0;
}

static void __exit
kpp_cleanup(void)
{
	/* Do some cleanup work */
	disable_irq(KPP_IRQ);
	free_irq(KPP_IRQ, MODULE_NAME);
	kfree(g_mx21_kpp_buffer);
	g_mx21_kpp_buffer = NULL;

	CRM_PCCR1 &= ~PCCR1_KPP_EN;

	if (g_kpp_major > 0) {
		devfs_unregister_chrdev(g_kpp_major, MODULE_NAME);
		devfs_unregister(g_devfs_handle);
	}
	pm_unregister(g_kpp_pm);
}

module_init(kpp_init);
module_exit(kpp_cleanup);
MODULE_LICENSE("GPL");
