/*
 * Copyright 2003 Motorola, Inc. All Rights Reserved.
 * This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 *
 * The source file for 1wire driver.
 *
 * File Name: onewire.c
 *
 * Programers:
 *
 * Date of Creations:   20 Oct,2003
 *
 * Synopsis:
 *
 * Descirption:
 * 	Driver for the DB-MX21 OneWire
 *
 * Modification History:
 *
 */
#include <linux/module.h>
#include <linux/version.h>

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <asm/uaccess.h>

#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/string.h>
#include <linux/init.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <linux/errno.h>
#include <linux/tqueue.h>
#include <linux/wait.h>

#include <asm/irq.h>
#include <asm/arch/hardware.h>
#include <asm/arch/gpio.h>
#include <linux/pm.h>

#include "mx21-1wire.h"

static int Owire_open(struct inode *inode, struct file *filp);
static int Owire_release(struct inode *inode, struct file *filp);
static ssize_t Owire_read(struct file *filp,
			  char *buf, size_t count, loff_t * l);
static ssize_t Owire_write(struct file *file,
			   const char *buffer, size_t count, loff_t * ppos);
static int Owire_ioctl(struct inode *inode,
		       struct file *filp, unsigned int cmd, unsigned long arg);
static unsigned int Owire_poll(struct file *filp,
			       struct poll_table_struct *wait);

struct file_operations g_Owire_fops = {
	.open = Owire_open,
	.release = Owire_release,
	.write = Owire_write,
	.read = Owire_read,
	.poll = Owire_poll,
	.ioctl = Owire_ioctl,
	.fasync = NULL,
};

static int
Owire_open(struct inode *inode, struct file *filp)
{

	/* set AIPI1_PSR0 bit[9] and clear AIPI1_PSR1bit[9] to match 1-wire bus width 16bits */

	AIPI_PSR0(1) |= 0x00200;
	AIPI_PSR1(1) &= ~0x00200;

	TRACE("AIPI1_PSR0=%x, AIPI1_PSR1=%x \n", AIPI_PSR0(1), AIPI_PSR1(1));
	/* enable clk */
	CRM_PCCR1 |= 0x80000000;
	/* GPIO config */

	mx2_register_gpios(PORT_E, (1 << 16) , SECONDARY | NOINTERRUPT);

	/* reset the 1-wire module */
	OWIRE_RESET |= 1;
	OWIRE_RESET = 0;

	/* set the time devider to get the 1M clk from the main clock */
	OWIRE_TIME_DIV = 0x2B;

	OWIRE_CTRL |= 0x80;	/* control register bit 7 = 1; */
	/* check if PST is set */
	mdelay(1);
	TRACE("OWIRE_CTRL = %x \n", OWIRE_CTRL);
	/* check if RPP self cleared */
	mdelay(1);
	TRACE("OWIRE_CTRL = %x \n", OWIRE_CTRL);
	g_Owire_status = OWIRE_OPEN_STATUS;

	CRM_PCCR1 &= ~0x80000000;

	MOD_INC_USE_COUNT;
	return 0;
}

static int
Owire_release(struct inode *inode, struct file *filp)
{
	g_Owire_status &= ~OWIRE_OPEN_STATUS;
	/* disable clk */
	CRM_PCCR1 &= ~0x80000000;
	MOD_DEC_USE_COUNT;
	return 0;
}

static ssize_t
Owire_read(struct file *filp, char *buf, size_t count, loff_t * l)
{
	int i, j;
	char tmp, value;
	char *pBuf;
	pBuf = buf;
	TRACE("count = %d \n", count);
	/* enable clk */
	CRM_PCCR1 |= 0x80000000;
	for (i = 0; i < count; i++) {
		value = 0;
		for (j = 0; j < 8; j++) {
			/* control register bit 4 = 1 */
			OWIRE_CTRL |= 0x10;
			udelay(117);

			tmp = (OWIRE_CTRL & 0x08) ? 1 : 0;
			value |= tmp << j;
		}
		TRACE("value %d = %x \n", i, value);
		__copy_to_user(pBuf, &value, 1);
		pBuf++;

	}
	TRACE("read finished \n");
	CRM_PCCR1 &= ~0x80000000;
	return count;
}

static ssize_t
Owire_write(struct file *file, const char *buffer, size_t count, loff_t * ppos)
{
	int i, j;
	char value, tmp;
	/* enable clk */
	CRM_PCCR1 |= 0x80000000;

	for (i = 0; i < count; i++) {
		__copy_from_user(&value, buffer, 1);
		TRACE("value = %x \n", value);
		for (j = 0; j < 8; j++) {
			tmp = value & 0x01;
			if (tmp == 0) {	/* control register bit 5 = 1 */
				OWIRE_CTRL |= 0x20;
				udelay(117);
			} else if (tmp == 1) {	/* control register bit 4 = 1 */
				OWIRE_CTRL |= 0x10;
				udelay(117);
			}
			value >>= 1;
		}
		buffer++;
	}
	CRM_PCCR1 &= ~0x80000000;
	TRACE("write finished \n");
	return count;
}

/*
 * Function Name: check_device
 *
 * Input: 		inode	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: verify if the inode is a correct inode
 *
 */
static int
check_device(struct inode *pInode)
{
	int minor;
	kdev_t dev = pInode->i_rdev;

	if (MAJOR(dev) != g_Owire_major) {
		printk("Owire: check_device bad major = %d\n", MAJOR(dev));
		return -1;
	}
	minor = MINOR(dev);

	if (minor < 1)
		return minor;
	else {
		printk("Owire: check_device bad minor = %d\n", minor);
		return -1;
	}
}

/*
 * Function Name: Owire_ioctl
 *
 * Input: 	inode	:
 * 			filp	:
 * 			cmd	: command for ioctl
 * 			arg	: parameter for command
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: ioctl for this device driver
 *
 * 	
 */
static int
Owire_ioctl(struct inode *inode,
	    struct file *filp, unsigned int cmd, unsigned long arg)
{
	int ret = -EIO;
	int minor;

	minor = check_device(inode);
	if (minor == -1) {
		printk("OneWire_ioctl:bad minor\n");
		return -ENODEV;
	}

	printk("OneWire_ioctl:minor=%08x cmd=%d\n", minor, cmd);

	return ret;
}

/*
 * Function Name: Owire_poll
 *
 * Input: 		filp	:
 * 			wait	:
 * Value Returned:	int	: Return status.If no error, return 0.
 *
 * Description: support poll and select
 *
 */
static unsigned int
Owire_poll(struct file *filp, struct poll_table_struct *wait)
{
	return 0;
}

static int
Owire_pm_handler(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	switch (rqst) {
	case PM_RESUME:
		if ((g_Owire_status & OWIRE_SUSPEND_STATUS) != 0) {
			/* enable clk */
			CRM_PCCR1 |= 0x80000000;
			g_Owire_status &= ~OWIRE_SUSPEND_STATUS;
		}
		break;
	case PM_SUSPEND:
		if ((g_Owire_status & OWIRE_OPEN_STATUS) != 0) {
			/* disable clk */
			CRM_PCCR1 &= ~0x80000000;
			g_Owire_status |= OWIRE_SUSPEND_STATUS;

		}
		break;
	default:
		break;
	}
	return 0;
}

Owire_init(void)
{
	/* register our character device */
	g_Owire_major = devfs_register_chrdev(0, MODULE_NAME, &g_Owire_fops);
	if (g_Owire_major < 0) {
		TRACE("%s driver: Unable to register driver\n", MODULE_NAME);
		return -ENODEV;
	}

	g_devfs_handle = devfs_register(NULL, MODULE_NAME, DEVFS_FL_DEFAULT,
					g_Owire_major, 0,
					S_IFCHR | S_IRUSR | S_IWUSR,
					&g_Owire_fops, NULL);

	/* Modified by Bill, May 9, 2004 */
	g_Owire_pm = pm_register(PM_SYS_DEV, PM_SYS_UNKNOWN, Owire_pm_handler);
	g_Owire_status = 0;

	printk("Owire Driver 0.1.0\n");

	return 0;
}

void __exit
Owire_cleanup(void)
{
	devfs_unregister_chrdev(g_Owire_major, MODULE_NAME);
	devfs_unregister(g_devfs_handle);
	pm_unregister(g_Owire_pm);
}

module_init(Owire_init);
module_exit(Owire_cleanup);
