/******************************************************************************
	mx2-slcdc.c

	This is the basic release for SLCDC driver,
	which is to support EPSON L2F50032T00 16bit seria mode5-6-5
	This driver acts as a standard character device,
	which can be dinamically loaded.

	Copyright (C) 2003 Motorola Semiconductors HK Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
	
	First version 12-15-2003, Karen Kang

********************************************************************************/
#undef SLCDC_MUX_SDHC2
#define SLCDC_MUX_LCDC

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/wrapper.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <linux/miscdevice.h>
#include <linux/string.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/string.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/tqueue.h>
#include <linux/wait.h>
#include <linux/pm.h>

#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/uaccess.h>	/* get_user,copy_to_user */
#include <asm/irq.h>
#include <asm/arch/gpio.h>
#include <asm/arch/hardware.h>
#include <asm/arch/irqs.h>
#include <asm/page.h>
#include <asm/pgtable.h>

#define MODULE_NAME "slcdc"

#define SLCDC_IMG_8L	0x00020000

/* definition for SLCDC cmd */
/* same as Epson 10043 & 50052 */

#define SLCDC_CMD_DISON		0xaf00	/* 1. display on */
#define SLCDC_CMD_DISOFF	0xae00	/* 2. display off */
#define SLCDC_CMD_DISNOR	0xa600	/* 3. normal display */
#define SLCDC_CMD_DISINV	0xa700	/* 4. inverse display */
#define SLCDC_CMD_DISCTL	0xca00	/* 5. display control */
#define SLCDC_CMD_SLPIN		0x9500	/* 10.sleep in */
#define SLCDC_CMD_SLPOUT	0x9400	/* 11.sleep out */
#define SLCDC_CMD_SD_PSET	0x7500	/* 12.page address set */
#define SLCDC_CMD_SD_CSET	0x1500	/* 14.column address set */

#define SLCDC_CMD_DATCTL	0xbc	/* 16.data scan direction, etc. */
//#define SLCDC_CMD_DATCTL              0xbc00  /* 16.data scan direction, etc. */
#define SLCDC_CMD_RAMWR		0x5c00	/* 17.writing to memory */
#define SLCDC_CMD_PTLIN		0xa800	/* 19.partial display in */
#define SLCDC_CMD_PTLOUT	0xa900	/* 20.partial display out */

/* value different from 10043 but same as 50052 */
#define SLCDC_CMD_VOLCTR	0xc600	/* 25.Electronic volume control */

/* commands not found in 10043 but in 50052 */
#define SLCDC_CMD_GCP64		0xcb00	/* 6. 64 grayscale pulse positon set */
#define SLCDC_CMD_GCP16		0xcc00	/* 7. 16 grayscale pulse positon set */
#define SLCDC_CMD_GSSET		0xcd00	/* 8. grayscale set */
#define SLCDC_CMD_RAMRD		0x5d00	/* 18.memory read */
#define SLCDC_CMD_ASCSET	0xaa00	/* 21.area scroll set */
#define SLCDC_CMD_SCSTART	0xab00	/* 22.scroll start set */
#define SLCDC_CMD_EPCTIN	0x6100	/* 26.Power IC control for EVR */
#define SLCDC_CMD_EPCTOUT	0x6200	/* 27.Power IC control for EVR */

#define SLCDC_IRQ	60
#define SLCDC_CMD_MEM_SIZE		4
#define SLCDC_WIDTH				240
#define SLCDC_HIGH				320
#define SLCDC_BPP				16
#define SLCDC_PIXEL_MEM_SIZE	(SLCDC_WIDTH*SLCDC_HIGH*SLCDC_BPP)/8
#define _SLCDC_DATA_SIZE_	(SLCDC_PIXEL_MEM_SIZE + 32)
#define SLCDC_DATA_MEM_SIZE	(PAGE_ALIGN(_SLCDC_DATA_SIZE_ + PAGE_SIZE * 2))

#define SLCDC_TRANSFER_BUSY		0x4
#define SLCDC_TRANSFER_ERROR	0x10

u16 *g_slcdc_dbuffer_address;	  /* used for SLCDC data buffer */
u16 *g_slcdc_dbuffer_phyaddress;  /* physical address for SLCDC data buffer */
u16 *g_slcdc_cbuffer_address;	  /* used for SLCDC command buffer */
u16 *g_slcdc_cbuffer_phyaddress;  /* physical address for SLCDC command buffer */
static wait_queue_head_t slcdc_wait;

static int g_slcdc_major = 0;
static devfs_handle_t g_devfs_handle;
static wait_queue_head_t slcdc_wait;

struct pm_dev *g_slcdc_pm;
static int g_slcdc_status;
#define SLCDC_OPEN_STATUS		0x0001
#define SLCDC_SUSPEND_STATUS	0x0002

int slcdc_open(struct inode *inode, struct file *filp);
int slcdc_release(struct inode *inode, struct file *filp);
static int slcdc_ioctl(struct inode *inode, struct file *filp, u_int cmd,
		       u_long arg);
static int slcdc_mmap(struct file *filp, struct vm_area_struct *vma);

typedef struct {
	u16 *screen_start_address;
	u16 *v_screen_start_address;
} slcdc_par_t;
slcdc_par_t slcdc_par;
struct file_operations g_slcdc_fops = {
	open:slcdc_open,
	release:slcdc_release,
	ioctl:slcdc_ioctl,
	mmap:slcdc_mmap,
};

#define USING_INTERRUPT_SLCDC

void
slcdc_gpio_serial(void)
{

#ifdef SLCDC_MUX_LCDC
/* configure slcdc gpio setting using pa24~27 (AIN), muxed with LCD panel */
	mx2_register_gpios(PORT_A, 1 << 24 | 1 << 25 | 1 << 26 | 1 << 27,
			   GPIO | OUTPUT);
#endif

#ifdef SLCDC_MUX_SDHC2
/* configure slcdc gpio setting using pb6~9(AIN), muxed with SDHC2 */
	mx2_register_gpios(PORT_B, 1 << 6 | 1 << 7 | 1 << 8 | 1 << 9,
			   GPIO | OUTPUT);
#endif
}

void
slcdc_reset(int level)
{
	if (level == 0) {
#ifdef SLCDC_MUX_LCDC
		/* OE_ACD as a reset pin */
		mx2_register_gpios(PORT_A, 1 << 31, GPIO | OUTPUT | OCR_DATA);

		/* set reset pin to low */
		mx2_gpio_set_bit(PORT_A, 31, 0);
#endif

#ifdef SLCDC_MUX_SDHC2
		/* SD2_D1 as reset pin */
		mx2_register_gpios(PORT_B, 1 << 5, GPIO | OUTPUT | OCR_DATA);

		/* set reset pin to low */
		mx2_gpio_set_bit(PORT_B, 5, 0);
#endif
	} else {
#ifdef SLCDC_MUX_LCDC
		/* set reset pin to high */
		mx2_gpio_set_bit(PORT_A, 31, 1);
#endif
#ifdef SLCDC_MUX_SDHC2
		/* set reset pin to high */
		mx2_gpio_set_bit(PORT_B, 5, 1);
#endif

	}
}

void
slcdc_init_dev(void)
{

	MAX_SLV_SGPCR(3) |= 0x00040000;
	SYS_PSCR |= 0x00000004;	/* set LCD/SLCD bus master high priority */
	/* enable the slcd clk */
	CRM_PCCR0 |= PCCR0_SLCD_EN | PCCR0_HCLK_SLCD_EN;
}

void
slcdc_init_reg(void)
{
	SLCDC_DBADDR = 0;
	SLCDC_DBUF_SIZE = 0;
	SLCDC_CBADDR = 0;
	SLCDC_CBUF_SIZE = 0;
	SLCDC_CBUF_SSIZE = 0;
	SLCDC_FIFO_CONFIG = 0;
	SLCDC_LCD_CONFIG = 0;
	SLCDC_LCD_TXCONFIG = 0;
	SLCDC_LCD_CTRL_STAT = 0;
	SLCDC_LCD_CLKCONFIG = 0;
}

void
slcdc_config(int datlen)
{
	u32 xfrmode, sckpol, worddefcom, imgend = 0, worddefwrite =
	    0, worddefdat = 0;
	if (datlen == 8) {
		imgend = 0x2;	/* 8-bit little endian; */
		worddefdat = 0;	/* 8-bit data */
		worddefwrite = 10;
	} else if (datlen == 16) {
		imgend = 0x1;	/* 16-bit little endian; */
		worddefdat = 1;	/* 16-bit data */
		worddefwrite = 1;
	}
	worddefcom = 1;
	xfrmode = 0;		/* serial mode */
	sckpol = 1;		/* falling edge */

	/* config to be little endian serial 16bit */
	SLCDC_LCD_TXCONFIG =
	    (imgend << 16) | (worddefdat << 4) | (worddefcom << 3) | (xfrmode <<
								      2) |
	    (sckpol);

	/* config dma setting */
	SLCDC_FIFO_CONFIG = 5;	/* burst length is 4 32-bit words */
	/* config buffer address setting */
	SLCDC_DBADDR = (u32) (slcdc_par.screen_start_address);
	SLCDC_CBADDR = (u32) g_slcdc_cbuffer_phyaddress;
	/* config clk setting */
	SLCDC_LCD_CLKCONFIG = 0x3;
	/* set GO 0 */
	SLCDC_LCD_CTRL_STAT = 0;

	udelay(5);

}

/* for command transfer, it is very short, will not use interrupt */
int
slcdc_send_cmd(u32 length)
{
	u32 status;
	/* disable interrupt */
	SLCDC_LCD_CTRL_STAT &= 0xffffff7f;
	/* set length */
	SLCDC_DBUF_SIZE = length;
	/* set automode 00 for command */
	SLCDC_LCD_CTRL_STAT &= 0x000001ff;
	/* set GO */
	SLCDC_LCD_CTRL_STAT |= 0x1;
	/* polling for data transfer finish */
	status = SLCDC_LCD_CTRL_STAT;
	while ((!(status & SLCDC_TRANSFER_ERROR))
	       && (status & SLCDC_TRANSFER_BUSY)) {
		status = SLCDC_LCD_CTRL_STAT;
	}

	if (status & SLCDC_TRANSFER_ERROR) {
		pr_debug("send cmd error status=0x%x \n", status);
		return 1;
	} else {
		SLCDC_LCD_CTRL_STAT |= 0x40;
		SLCDC_LCD_CTRL_STAT |= 0x20;
		return 0;
	}
}

void
slcdc_send_data(u32 length)
{
	/* enable interrupt */
#ifdef USING_INTERRUPT_SLCDC
	SLCDC_LCD_CTRL_STAT |= ~0xffffff7f;
#endif
	/* set length */
	SLCDC_DBUF_SIZE = length;
	/* set automode 01 for data */
	SLCDC_LCD_CTRL_STAT |= 0x00000800;
	/* set GO */
	SLCDC_LCD_CTRL_STAT |= 0x1;
#ifdef USING_INTERRUPT_SLCDC
	interruptible_sleep_on(&slcdc_wait);
#else
	while ((SLCDC_LCD_CTRL_STAT & 0x00000004) != 0) ;

	SLCDC_LCD_CTRL_STAT |= 0x40;
	SLCDC_LCD_CTRL_STAT |= 0x20;
#endif
	return;

}

static void
slcdc_isr(int irq, void *dev_id, struct pt_regs *regs)
{
	u32 status;
	status = SLCDC_LCD_CTRL_STAT;
	/* clear interrupt */
	SLCDC_LCD_CTRL_STAT |= 0x40;
	wake_up_interruptible(&slcdc_wait);
	return;
}

int
slcdc_init_buffer(void)
{
	u_int required_pages;
	u_int extra_pages;
	u_int order;
	struct page *page;
	u16 *allocated_region;

	pr_debug("slcdc data buffer size = %x \n",
		 (unsigned int) SLCDC_DATA_MEM_SIZE);

	if (g_slcdc_dbuffer_address != NULL)
		return -EINVAL;
	if (g_slcdc_cbuffer_address != NULL)
		return -EINVAL;
	/*find order required to allocate enough memory for it */
	required_pages = SLCDC_DATA_MEM_SIZE >> PAGE_SHIFT;
	for (order = 0; required_pages >> order; order++) {;
	}
	extra_pages = (1 << order) - required_pages;
	pr_debug("PAGE_SHIFT=0x%x required_pages=0x%x\n", PAGE_SHIFT,
		 required_pages);
	pr_debug("extra_page=0x%x \n", extra_pages);

	if ((allocated_region =
	     (u16 *) __get_free_pages(GFP_KERNEL | GFP_DMA, order)) == NULL) {
		printk(KERN_ERR "can not allocated memory\n");
		return -ENOMEM;
	}

	g_slcdc_dbuffer_address =
	    (u16 *) ((u_char *) allocated_region + (extra_pages << PAGE_SHIFT));
	g_slcdc_dbuffer_phyaddress =
	    (u16 *) ((u_char *)
		     __virt_to_phys((u_long) g_slcdc_dbuffer_address));
	pr_debug("g_slcdc_dbuffer_address=0x%x \n",
		 (unsigned long) g_slcdc_dbuffer_address);

	/* Free all pages that we don't need but were given to us because */
	/* __get_free_pages() works on powers of 2. */
	for (; extra_pages; extra_pages--)
		free_page((u_int) allocated_region +
			  ((extra_pages - 1) << PAGE_SHIFT));

	/* Set reserved flag for fb memory to allow it to be remapped into */
	/* user space by the common fbmem driver using remap_page_range(). */
	for (page = virt_to_page(g_slcdc_dbuffer_address);
	     page < virt_to_page(g_slcdc_dbuffer_address + SLCDC_DATA_MEM_SIZE);
	     page++) {
		mem_map_reserve(page);
	}
	slcdc_par.screen_start_address =
	    (u16 *) ((u_long) g_slcdc_dbuffer_phyaddress);
	slcdc_par.v_screen_start_address =
	    (u16 *) ((u_long) g_slcdc_dbuffer_address);
	pr_debug("startaddr= 0x%x, phyaddr=0x%x , size=0x%x \n",
		 slcdc_par.screen_start_address,
		 slcdc_par.v_screen_start_address, SLCDC_DATA_MEM_SIZE);
	memset(slcdc_par.v_screen_start_address, 0, SLCDC_DATA_MEM_SIZE);

	g_slcdc_cbuffer_address =
	    consistent_alloc(GFP_KERNEL | GFP_DMA, (SLCDC_CMD_MEM_SIZE + 4),
			     (dma_addr_t *) & g_slcdc_cbuffer_phyaddress);
	g_slcdc_cbuffer_address =
	    (u16 *) ((u_long) g_slcdc_cbuffer_address & 0xfffffff4);
	g_slcdc_cbuffer_phyaddress =
	    (u16 *) ((u_long) g_slcdc_cbuffer_phyaddress & 0xfffffff4);

	pr_debug("slcdc data buffer address = %x cmd buffer address= %x \n",
		 (unsigned int) slcdc_par.screen_start_address,
		 g_slcdc_cbuffer_address);

	return 0;
}

int
slcdc_free_buffer(void)
{
	struct page *page;
	u_int required_pages;
	required_pages = SLCDC_DATA_MEM_SIZE >> PAGE_SHIFT;
	/* free command buffer */
	if (g_slcdc_cbuffer_address != NULL)
		consistent_free(g_slcdc_cbuffer_address, SLCDC_CMD_MEM_SIZE,
				(dma_addr_t) g_slcdc_cbuffer_phyaddress);
	else
		return -EINVAL;
	/* free data buffer       */
	for (page = virt_to_page(g_slcdc_dbuffer_address);
	     page < virt_to_page(g_slcdc_dbuffer_address + SLCDC_DATA_MEM_SIZE);
	     page++) {
		mem_map_unreserve(page);
	}
	for (; required_pages; required_pages--)
		free_page((u_int) g_slcdc_dbuffer_address +
			  ((required_pages - 1) << PAGE_SHIFT));

	return 0;
}

static int
slcdc_mmap(struct file *filp, struct vm_area_struct *vma)
{
	unsigned long page, pos;
	unsigned long start = (unsigned long) vma->vm_start;
	unsigned long size = (unsigned long) (vma->vm_end - vma->vm_start);

	if (size > SLCDC_DATA_MEM_SIZE)
		return -EINVAL;

	pos = (unsigned long) slcdc_par.v_screen_start_address;

	while (size > 0) {
		page = virt_to_phys((void *) pos);

		vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
		/*  This is an IO map - tell maydump to skip this VMA */
		vma->vm_flags |= VM_IO;

		if (remap_page_range(start, page, PAGE_SIZE, PAGE_SHARED))
			return -EAGAIN;
		start += PAGE_SIZE;
		pos += PAGE_SIZE;
		size -= PAGE_SIZE;
	}

	return 0;
}

void
slcdc_display_on(void)
{
	u16 *databuffer = NULL;

	SLCDC_DBADDR = (u32) g_slcdc_cbuffer_phyaddress;

	/* put cmd into cmd buffer */
	databuffer = g_slcdc_cbuffer_address;
	/* put cmd into cmd buffer */
	*databuffer = SLCDC_CMD_DISON;
	/* send cmd */
	slcdc_send_cmd(1);
	/* delay */
	udelay(50);
	slcdc_send_cmd(1);
	udelay(50);
}

void
slcdc_display_off(void)
{
	u16 *databuffer = NULL;
	/* Cause LCD module to enter sleep mode.Bu sure to input display OFF
	   command to turn off the display before inputting the SLPIN command
	   Keep the logic power supply turned on for 40ms
	   after the LCD module enters the sleep mode.
	   put cmd into cmd buffer
	 */
	databuffer = slcdc_par.v_screen_start_address;
	/* send cmd SLPIN */
	*databuffer = SLCDC_CMD_SLPIN;
	slcdc_send_cmd(1);
	/* put reset high */
	mx2_gpio_set_bit(PORT_B, 5, 0);
	/* delay for 50ms */
	udelay(50);
	/* put cmd into cmd buffer */
	*databuffer = SLCDC_CMD_DISOFF;
	/* send cmd DISOFF */
	slcdc_send_cmd(1);
	pr_debug("disoff \n");
	udelay(50);
}

void
slcdc_display_normal(void)
{
	u16 *databuffer = NULL;
	databuffer = slcdc_par.v_screen_start_address;
	*databuffer = SLCDC_CMD_DISNOR;
	slcdc_send_cmd(1);
	udelay(1);

}

void
slcdc_sleep_out(void)
{
	u16 *databuffer = NULL;
	databuffer = slcdc_par.v_screen_start_address;
	*databuffer = SLCDC_CMD_SLPOUT;
	slcdc_send_cmd(1);
	udelay(50);
}

void
slcdc_display_ctl(void)
{
	int i;
	u16 disctl[11] = { 0x1c00, 0x0200, 0x8200, 0x0000,
		0x1e00, 0xe000, 0x0000, 0xdc00,
		0x0000, 0x0200, 0x0000
	};
	u16 *databuffer = NULL;
	/* It make various display timing settings. */
	databuffer = slcdc_par.v_screen_start_address;
	*databuffer = SLCDC_CMD_DISCTL;
	slcdc_send_cmd(1);
	udelay(2);
	/* put parameter into data buffer */
	for (i = 0; i < 11; i++) {
		*databuffer = disctl[i];
		databuffer++;
	}
	/* send data */
	slcdc_send_data(11);

}

void
slcdc_page_set(void)
{
/* It specify a page address area in order to access the display data RAM
from the MPU. Be sure to set both starting and ending pages. */
	int i;
	u16 *databuffer = NULL;
	u16 pset[4] = { 0x0000, 0x0000, 0xb100, 0x0000 };
	databuffer = (u16 *) (slcdc_par.v_screen_start_address);
	*databuffer = SLCDC_CMD_SD_PSET;
	slcdc_send_cmd(1);
	udelay(20);

	for (i = 0; i < 4; i++) {
		*databuffer = pset[i];
		databuffer++;
	}
	slcdc_send_data(4);

	udelay(20);
}

void
slcdc_col_set(void)
{
/* It specify starting and ending collumns. */
	int i;
	u16 cset[4] = { 0x0200, 0x0000, 0xb100, 0x0000 };
	u16 *databuffer = NULL;
	databuffer = (u16 *) (slcdc_par.v_screen_start_address);
	*databuffer = SLCDC_CMD_SD_CSET;
	slcdc_send_cmd(1);
	udelay(20);
	for (i = 0; i < 4; i++) {
		*databuffer = cset[i];
		databuffer++;
	}
	slcdc_send_data(4);
	udelay(20);

}

void
slcdc_data_ctl(void)
{
	u8 *databuffer = NULL;
	databuffer = (u8 *) (slcdc_par.v_screen_start_address);
	*databuffer = SLCDC_CMD_DATCTL;
	slcdc_send_cmd(1);
	udelay(20);
	*databuffer = 0x28;
	slcdc_send_data(1);
	udelay(20);
}

void
slcdc_volctl(void)
{
	u16 *databuffer = NULL;
	databuffer = slcdc_par.v_screen_start_address;
	*databuffer = SLCDC_CMD_VOLCTR;
	slcdc_send_cmd(1);
	udelay(20);
	*databuffer = 0x9f00;
	slcdc_send_data(1);
	udelay(20);
}

void
slcdc_ascset(void)
{
	int i;
	u16 lcd_para_ASCSET[7] = { 0x0000, 0x0000, 0x1f00, 0x0100,
		0x1f00, 0x0100, 0x0300
	};
	u16 *databuffer = NULL;
	databuffer = (u16 *) (slcdc_par.v_screen_start_address);
	*databuffer = SLCDC_CMD_ASCSET;
	slcdc_send_cmd(1);
	udelay(2);
	for (i = 0; i < 7; i++) {
		*databuffer = lcd_para_ASCSET[i];
		databuffer++;
	}
	slcdc_send_data(7);
	udelay(20);
}

void
slcdc_scstart(void)
{
	int i;
	u16 lcd_para_SCSTART[2] = { 0x00, 0x00 };
	u16 *databuffer = NULL;
	databuffer = (u16 *) (slcdc_par.v_screen_start_address);
	*databuffer = SLCDC_CMD_SCSTART;
	slcdc_send_cmd(1);
	udelay(10);
	for (i = 0; i < 2; i++) {
		*databuffer = lcd_para_SCSTART[i];
		databuffer++;
	}
	slcdc_send_data(2);
	udelay(20);

}

void
slcdc_config_panel(void)
{
	int i;

	slcdc_config(8);
	udelay(50);
	slcdc_reset(1);		/* pull reset signal high */

	/* set data format */
	udelay(20);
	slcdc_data_ctl();
	udelay(1);
	slcdc_config(16);

	slcdc_sleep_out();
	udelay(20);

	for (i = 0; i < 8; i++) {
		slcdc_volctl();
		udelay(2);
	}
	udelay(50);
	udelay(50);
	udelay(50);

	slcdc_display_on();
	udelay(50);

	/* set col address */
	slcdc_col_set();
	/* set page address */
	slcdc_page_set();
	/* set area in screen to be used for scrolling */
	slcdc_ascset();
	udelay(20);
	/* set top scroll page within the scroll area */
	slcdc_scstart();
	udelay(50);
}

static int
slcdc_ioctl(struct inode *inode, struct file *filp, u_int cmd, u_long arg)
{
	u16 *databuffer = NULL;
	switch (cmd) {
	case SLCDC_CMD_DISON:
		slcdc_display_on();
		break;

	case SLCDC_CMD_DISOFF:
		slcdc_display_off();
		break;

	case SLCDC_CMD_DISNOR:
		slcdc_display_normal();
		break;

	case SLCDC_CMD_DISINV:
		databuffer = slcdc_par.v_screen_start_address;
		*databuffer = SLCDC_CMD_DISINV;
		slcdc_send_cmd(1);
		udelay(1);
		break;

	case SLCDC_CMD_DATCTL:
		break;

	case SLCDC_CMD_RAMWR:
		/* Causes the MPU to be a data entry mode,
		  allowing it to serite data in the display memory.
		  Inputting any other cmds other than NOP cancels
		  the data entry mode.
		 */
		SLCDC_DBADDR = (u32) g_slcdc_cbuffer_phyaddress;

		databuffer = g_slcdc_cbuffer_address;
		*databuffer = SLCDC_CMD_RAMWR;
		slcdc_send_cmd(1);
		udelay(2);

		/* this is to display one data per time, it is ok. */
		udelay(50);
		udelay(50);
		SLCDC_DBADDR = (u32) slcdc_par.screen_start_address;

		udelay(50);
		udelay(50);
		slcdc_send_data(38720);
		udelay(50);
		udelay(50);

		break;

	case SLCDC_CMD_RAMRD:
		break;

	case SLCDC_CMD_PTLIN:
		/* This command is used to display a partial screen
		   for power saving. */
		break;

	case SLCDC_CMD_PTLOUT:
		/* This command is used to exit the partila diaplay mode. */
		break;

	case SLCDC_CMD_GCP64:
		/* make 63 pulse position settings of GCP for 64 gray scales. */
		break;

	case SLCDC_CMD_GCP16:
		break;

	case SLCDC_CMD_GSSET:
		break;

	case SLCDC_CMD_ASCSET:
		/* make partial screen scroll settings. */
		break;

	case SLCDC_CMD_SCSTART:
		/* set a scroll starting page in the scrolling area.
		   Be sure to send this cmd after ASCSET . */
		break;
	default:
		break;
	}
	return 0;
}

int
slcdc_release(struct inode *inode, struct file *filp)
{
	pr_debug("slcdc_release: ----\n");
	MOD_DEC_USE_COUNT;
	return 0;
}

int
slcdc_open(struct inode *inode, struct file *filp)
{
	pr_debug("slcdc_open: ----\n");
	MOD_INC_USE_COUNT;
	slcdc_gpio_serial();

	slcdc_init_dev();

	slcdc_reset(0);		/* pull reset low */

	slcdc_init_reg();

	init_waitqueue_head(&slcdc_wait);

	/* send some command for panel configuration */
	slcdc_config_panel();
	pr_debug("TRANS_CONFIG_REG=%x \n", SLCDC_LCD_TXCONFIG);

	return 0;
}

int
slcdc_pm_handler(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	switch (rqst) {
	case PM_RESUME:
		if ((g_slcdc_status & SLCDC_SUSPEND_STATUS) != 0) {
			slcdc_sleep_out();
			udelay(20);
			udelay(50);
			slcdc_display_on();
			g_slcdc_status &= ~SLCDC_SUSPEND_STATUS;
		}
		break;
	case PM_SUSPEND:
		if ((g_slcdc_status & SLCDC_OPEN_STATUS) != 0) {
			slcdc_display_off();
			g_slcdc_status |= SLCDC_SUSPEND_STATUS;
		}
		break;
	default:
		break;
	}
	return 0;
}

signed short __init
slcdc_init(void)
{
	int tmp;

	g_slcdc_major = devfs_register_chrdev(0, MODULE_NAME, &g_slcdc_fops);
	if (g_slcdc_major < 0) {
		pr_debug("%s driver: Unable to register driver\n", MODULE_NAME);
		return -ENODEV;
	}

	g_devfs_handle = devfs_register(NULL, MODULE_NAME, DEVFS_FL_DEFAULT,
					g_slcdc_major, 0,
					S_IFCHR | S_IRUSR | S_IWUSR,
					&g_slcdc_fops, NULL);

	tmp = request_irq(SLCDC_IRQ,
			  (void *) slcdc_isr,
			  SA_INTERRUPT | SA_SHIRQ, MODULE_NAME, MODULE_NAME);
	if (tmp) {
		printk("slcdc_init:cannot init major= %d irq=%d\n",
		       g_slcdc_major, SLCDC_IRQ);
		devfs_unregister_chrdev(g_slcdc_major, MODULE_NAME);
		devfs_unregister(g_devfs_handle);
		return -1;
	}
	g_slcdc_dbuffer_address = NULL;
	g_slcdc_dbuffer_phyaddress = NULL;
	g_slcdc_cbuffer_address = NULL;
	g_slcdc_cbuffer_phyaddress = NULL;
	slcdc_init_buffer();

	g_slcdc_pm = pm_register(PM_SYS_DEV, PM_SYS_VGA, slcdc_pm_handler);
	g_slcdc_status = 0;

	return 0;
}

void __exit
slcdc_cleanup(void)
{
	/*Do some cleanup work */
	free_irq(SLCDC_IRQ, MODULE_NAME);
	slcdc_free_buffer();

	if (g_slcdc_major > 0) {
		devfs_unregister_chrdev(g_slcdc_major, MODULE_NAME);
		devfs_unregister(g_devfs_handle);
	}
	pm_unregister(g_slcdc_pm);

}

int
init_module(void)
{
	return slcdc_init();
}

void
cleanup_module(void)
{
	slcdc_cleanup();
}

MODULE_LICENSE("GPL");
