/*
 *  MX1 Pulse Width Modulator
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~
 *  Copyright (C) 2002 Motorola Semiconductors HK Ltd
 *  Copyright (C) 2003 MontaVista Software Inc. <source@mvista.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <asm/uaccess.h>	/* get_user,copy_to_user */

#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/string.h>
#include <linux/init.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <linux/errno.h>
#include <linux/tqueue.h>
#include <linux/wait.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/pm.h>

#include <asm/irq.h>
#include <asm/arch/hardware.h>
#include <asm/arch/irqs.h>

#include <asm/arch/mx1ads-gpio.h>

#include "mx1ads-pwm.h"

#define rg_PWMC (*(volatile u32*)(IO_ADDRESS((PWM_BASE+PWMC))))
#define rg_PWMP (*(volatile u32*)(IO_ADDRESS((PWM_BASE+PWMP))))
#define rg_PWMS (*(volatile u32*)(IO_ADDRESS((PWM_BASE+PWMS))))

static pwm_timer_t pwm_timer;
static timer_blk_t timer_blk;

/* simple functions used to read/write registers on MX1ads */
static __inline__ u32
MX1_RR(u32 r)
{
	return (*(volatile u32 *) (IO_ADDRESS((PWM_BASE + r))));
}

static __inline__ void
MX1_WR(u32 r, u32 what)
{
	(*(volatile u32 *) (IO_ADDRESS((PWM_BASE + r)))) = what;
}

static __inline__ void
MX1_setbit(u32 r, int Nbit)
{
	u32 temp;

	temp = MX1_RR(r);
	temp |= (0x00000001 << Nbit);
	MX1_WR(r, temp);
}
static __inline__ void
MX1_clrbit(u32 r, int Nbit)
{
	u32 temp;

	temp = MX1_RR(r);
	temp &= ~(0x00000001 << Nbit);
	MX1_WR(r, temp);
}

/* variables */
int irq = 0;
int pwm_write_ok;
int pwm_mode;
char *gpWriteBuf;
U8 *gpWriteBufPtr8;
U16 *gpWriteBufPtr16;
int pwm_write_cnt;
int pwm_data_len;
U16 pwm_sample_value;

int pwm_debug = 0;

#define pwm_printk( dbg, params ) if ( (dbg) <= pwm_debug ) printk params

MODULE_PARM(irq, "i");
MODULE_PARM(pwm_debug, "i");
MODULE_LICENSE("GPL");

DECLARE_WAIT_QUEUE_HEAD(pwm_wait);

int gMajor = 0;

struct file_operations ts_fops = {
	.open = pwm_open,
	.release = pwm_release,
	.read = pwm_read,
	.write = pwm_write,
	.ioctl = pwm_ioctl,
	.fasync = pwm_fasync,
};

static devfs_handle_t devfs_handle;

/*
 * Function:
 *      pwm_delay
 * Synopsis:
 *      do the delay for period of clocks
 */
void
pwm_delay(int clocks)
{
	while (--clocks > 0) {
		int ctr;

		for (ctr = 0; ctr < 100000; ctr++)
			continue;
	}
}

/*
 * Function:
 * 	pwm_soft_reset
 * Synopsis:
 * 	reset the PWM
 */
void
pwm_soft_reset(void)
{
	MX1_setbit(PWMC, 4);
	MX1_setbit(PWMC, 16);
	MX1_clrbit(PWMC, 4);
	pwm_delay(5);
}

/*
 * Function:
 *    pwm_release
 * Synopsis:
 *    release the filp
 */
static int
pwm_release(struct inode *inode, struct file *filp)
{
	pwm_printk(2, ("%s entered...\n", __FUNCTION__));
	interruptible_sleep_on(&pwm_wait);
	pwm_printk(2, ("%s: awaken!\n", __FUNCTION__));
	MOD_DEC_USE_COUNT;
	pwm_printk(2, ("%s, done\n", __FUNCTION__));
	return 0;
}

/*
 * Function:
 * 	pwm_open
 * Synopsis:
 *	open the device
 */
static int
pwm_open(struct inode *inode, struct file *filp)
{

	MOD_INC_USE_COUNT;
	pwm_init_hw();		/* Init the PWM hardware */
	pwm_write_ok = 0;
	pwm_printk(2, ("%s, done\n", __FUNCTION__));
	return 0;
}

/*
 * Function:
 *	pwm_fasync
 */
static int
pwm_fasync(int fd, struct file *filp, int mode)
{
	pwm_printk(2, ("%s (fd = %d, fp=%p, mode=%d) entered...\n",
		       __FUNCTION__, fd, filp, mode));
	return 0;
}

/*
 * Function:
 *	pwm_ioctl
 * Synopsis:
 *	handle IOCTL addressed to PWM device
 * Parameters:
 *      arg may be one of:
 *         PWM_IOC_SMODE     set PWM mode: tone/play
 *         PWM_IOC_SFREQ     set frequency of tone to play
 *         PWM_IOC_SDATALEN
 *         PWM_IOC_SSAMPLE
 *         PWM_IOC_SPERIOD
 *         PWM_IOC_STOP
 *         PWM_IOC_SWAPDATA
 */
static int
pwm_ioctl(struct inode *inode,
	  struct file *filp, unsigned int cmd, unsigned long arg)
{
	char *str = NULL;
	int ret = 0;

	pwm_printk(2, ("%s (inode=>%p, fp=%p, cmd=%d, arg=%lx) entered...\n",
		       __FUNCTION__, inode, filp, cmd, arg));
	switch (cmd) {
	case PWM_IOC_SMODE:
		if ((pwm_mode = arg) == PWM_TONE_MODE) {
			/* create periodic timer when tone mode */
			pwm_printk(2, ("%s: set tone mode\n", __FUNCTION__));
			pwm_timer.timer_blk_ptr = &timer_blk;
			pwm_timer.timer_func = pwm_int_func;
			pwm_create_timer(&pwm_timer);
		} else {
			pwm_printk(2, ("%s: play mode.\n", __FUNCTION__));
		}
		break;

	case PWM_IOC_SFREQ:
		MX1_clrbit(PWMC, 4);	/* disable PWM */
		rg_PWMC &= 0xFFFF00F0;	/* sysclk; */

		/* check sampling rate */
		if (PWM_SAMPLING_8KHZ == arg) {
			/* 96M / 23/2/256 = 8.152kHz */
			rg_PWMC |= (0 << 8);
			str = "8Khz";
		} else if (arg == PWM_SAMPLING_16KHZ) {
			rg_PWMC |= (0 << 8);
			str = "16Khz";
		} else {
			str = NULL;
		}

		rg_PWMC &= ~0x3;	/* clksel = 0 (/2) */

		if (str) {
			pwm_printk(2,
				   ("%s: sample rate is %s\n", __FUNCTION__,
				    str));
		}
		break;

	case PWM_IOC_SDATALEN:
		if ((pwm_data_len = arg) == PWM_DATA_8BIT) {
			rg_PWMP = 0xfe;	/* Period, 8bit */
			str = "8bit";
		} else		/* if(arg == PWM_DATA_16BIT) */
		{
			rg_PWMP = 0xfffe;	/* Period, 16bit */
			str = "16bit";
		}

		pwm_printk(2, ("%s: data length = %s\n", __FUNCTION__, str));
		break;

	case PWM_IOC_SSAMPLE:
		pwm_sample_value = arg;
		break;

	case PWM_IOC_SPERIOD:
		if (PWM_TONE_MODE == pwm_mode) {
			pwm_printk(2,
				   ("%s: PWM period = %ld\n", __FUNCTION__,
				    arg));
			pwm_timer.period = arg / 12;
		}
		break;

	case PWM_IOC_STOP:
		pwm_stop();
		break;

	case PWM_IOC_SWAPDATA:
		if (arg & PWM_SWAP_HCTRL)	/*Halfword FIFO data swapping */
		{
			rg_PWMC |= PWM_SWAP_HCTRL;
		} else {
			rg_PWMC &= ~PWM_SWAP_HCTRL;
		}

		if (arg & PWM_SWAP_BCTRL)	/*Byte FIFO data swapping */
		{
			rg_PWMC |= PWM_SWAP_BCTRL;
		} else {
			rg_PWMC &= ~PWM_SWAP_BCTRL;
		}
		break;

	default:
		ret = -EINVAL;
		break;
	}
	return ret;
}

/*
 * Function:
 *     pwm_read
 * Synopsis:
 *     handle the call of read
 */
ssize_t
pwm_read(struct file * filp, char *buf, size_t count, loff_t * l)
{
	pwm_printk(2,
		   ("%s ( fp=%p, buf = %p, count = %ld, l = %p) entered...\n",
		    __FUNCTION__, filp, buf, (long) count, l));
	return (pwm_write_cnt == 0) ? 1 : 0;
}

/*
 * Function:
 * 	pwm_interrupt
 * Synopsis:
 *	IRQ handler
 */
void
pwm_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	U16 tmp16;
	U32 status;

	/* clear status */
	status = rg_PWMC;

	/* check end */
	if (pwm_write_cnt < 0) {
		pwm_stop();
		return;
	}

	if (PWM_TONE_MODE != pwm_mode) {
		if (PWM_DATA_8BIT == pwm_data_len) {
			rg_PWMS = (U32) (*gpWriteBufPtr8++);
			pwm_printk(2, ("%s: (%d) PWMS = 0x%08x\n",
				       __FUNCTION__, pwm_write_cnt,
				       (U32) rg_PWMS));
			pwm_write_cnt--;
		} else if (pwm_data_len == PWM_DATA_16BIT) {
			tmp16 = *gpWriteBufPtr8++;
			tmp16 |= (*gpWriteBufPtr8++ << 8);
			rg_PWMS = (U32) tmp16;
			pwm_printk(2, ("%s: (%d)PWMS = 0x%8x\n",
				       __FUNCTION__, pwm_write_cnt,
				       (U32) rg_PWMS));
			pwm_write_cnt -= 2;
		}
	}
}

/*
 * Function:
 *	pwm_write
 * Synopsis:
 *	write to the device. Please write the sequence of tones to play
 */
static ssize_t
pwm_write(struct file *filp, const char *buf, size_t count, loff_t * f_pos)
{
	pwm_printk(2, ("%s (fp=%p, buf = %p, count = %ld ) entered...\n",
		       __FUNCTION__, filp, buf, (long) count));

	pwm_write_ok = 0;

	if (NULL == (gpWriteBuf = kmalloc(count, GFP_KERNEL))) {
		return -ENOMEM;
	}
	gpWriteBufPtr8 = gpWriteBuf;

	if (copy_from_user(gpWriteBuf, buf, count)) {
		kfree(gpWriteBuf);
		return -EFAULT;
	}
	/*write data to PWM */
	pwm_write_cnt = count;
	pwm_printk(2, ("%s: count = %d\n", __FUNCTION__, pwm_write_cnt));

	rg_PWMC |= 0x00000010;	/* enable the PWM */
	rg_PWMC &= ~0x00000C0;	/* set REPEAT to zero */
	if (PWM_TONE_MODE != pwm_mode) {	/* pwm play mode */
		rg_PWMC |= 0x00000040;	/* enable interrupts */
	} else {
		rg_PWMS = pwm_sample_value;
		rg_PWMC |= 0x7f00;	/* set prescaler to 128 */
		gpWriteBufPtr16 = (U16 *) gpWriteBuf;

		pwm_write_cnt >>= 1;	/*input size is 8bit size, need /2 to get 16bit size */

		pwm_start_timer(&pwm_timer);
	}

	pwm_printk(2, ("%s: PWMC = %08X\n", __FUNCTION__, (U32) rg_PWMC));

	return 0 /* ESUCCES */ ;
}

/*
 * Function:
 *	pwm_init_hw
 * Synopsis:
 *	initialize the hardware. Currently it is called from
 *	open()
 */
static int
pwm_init_hw()
{
	return mx1_register_gpio(PORT_A, 2, PRIMARY | NOINTERRUPT);
}

static int
handle_pm_event(struct pm_dev *dev, pm_request_t rq, void *data)
{
	switch (rq) {
	case PM_SUSPEND:
		pwm_printk(2, ("%s: suspend\n", __FUNCTION__));
/*      pwm_stop(); */
		MX1_clrbit(PWMC, 4);
		break;
	case PM_RESUME:
		pwm_printk(2, ("%s: resume\n", __FUNCTION__));
		MX1_setbit(PWMC, 4);
		if (PWM_TONE_MODE == pwm_mode ) {
			pwm_stop_timer( &pwm_timer );
			rg_PWMS = pwm_sample_value;
			rg_PWMC |= 0x7f00;	// set prescaler to 128
			pwm_int_func(0);
		}
		break;
	default:
		break;
	}
	return 0;
}

int __init
pwm_init(void)
{
	int result;

	printk("PWM driver\n");
	pwm_printk(2, ("\n%s entered\n", __FUNCTION__));
	pwm_soft_reset();
	pwm_printk(2, ("\n%s pwm_soft_reset", __FUNCTION__));
	if ((result = devfs_register_chrdev(0, MOD_NAME, &ts_fops)) < 0) {
		pwm_printk(1,
			   ("%s: cannot register_chrdev with %d\n",
			    __FUNCTION__, result));
		return result;
	}
	devfs_handle = devfs_register(NULL, "pwm", DEVFS_FL_DEFAULT,
				      result, 0,
				      S_IFCHR | S_IRUSR | S_IWUSR,
				      &ts_fops, NULL);




	pwm_printk(2, ("\n%s registered %d", __FUNCTION__, result));
	if (gMajor == 0) {
		gMajor = result;
		printk(", major %d\n", gMajor);
	}

	pwm_printk(2, ("%s: request_irq\n", __FUNCTION__));
	if ((result = request_irq(AITC_PWM_INT,
				  pwm_interrupt,
				  SA_INTERRUPT,
				  DEV_IRQ_NAME, DEV_IRQ_ID)) < 0) {
		devfs_unregister_chrdev(gMajor, MOD_NAME);
		pwm_printk(1,
			   ("%s: request_irq failed with %d\n", __FUNCTION__,
			    result));
		return result;
	}

	init_waitqueue_head(&pwm_wait);

	/* register power manager */
	pm_register(PM_SYS_DEV, PM_SYS_UNKNOWN, handle_pm_event);

	/* init hardware */
	pwm_init_hw();
	return 0;
}

void __exit
pwm_cleanup(void)
{
	/* unregister GPIO */
	pm_unregister_all(handle_pm_event);
	mx1_unregister_gpio(PORT_A, 2);

	free_irq(AITC_PWM_INT, DEV_IRQ_ID);
	devfs_unregister_chrdev(gMajor, MOD_NAME);
	disable_irq(AITC_PWM_INT);
}

/* timer */
int
pwm_create_timer(pwm_timer_t * timer)
{
	init_timer(timer->timer_blk_ptr);
	timer->timer_blk_ptr->function = timer->timer_func;

	return (0);
}

int
pwm_start_timer(pwm_timer_t * timer)
{
	timer->timer_blk_ptr->expires = jiffies + timer->period;
	timer->stop_flag = 0;

	add_timer((struct timer_list *) timer->timer_blk_ptr);

	return (0);
}

int
pwm_stop_timer(pwm_timer_t * timer)
{
	timer->stop_flag = 1;
	del_timer_sync((struct timer_list *) timer->timer_blk_ptr);
	return (0);
}

void
pwm_int_func(unsigned long unused)
{
	U32 period;

	pwm_printk(2, ("%s entered\n", __FUNCTION__));

	if (pwm_write_cnt <= 0) {
		pwm_stop();
		return;
	}
	pwm_printk(2, ("%s: PWMS = 0x%08x\n", __FUNCTION__, (U32) rg_PWMS));

	/* 96M/128/2/period */
	period = *gpWriteBufPtr16++;

	if (period > 0) {
		rg_PWMP = 96000000 / 128 / 12 / 2 / period;	/*96000000/128/2*11 */
	}
	pwm_printk(2, ("%s: PWMP = 0x%08x\n", __FUNCTION__, (U32) rg_PWMP));

	pwm_start_timer(&pwm_timer);
	pwm_write_cnt--;
}

void
pwm_stop()
{
	pwm_printk(2, ("%s entered\n", __FUNCTION__));
	if (gpWriteBuf) {
		kfree(gpWriteBuf);
		gpWriteBuf = NULL;
	}

	MX1_clrbit(PWMC, 6);
	MX1_clrbit(PWMC, 4);

	pwm_write_ok = 1;
	pwm_write_cnt = 0;
	pwm_data_len = 0;
	pwm_sample_value = 0;

	pwm_printk(2, ("%s: mode = %d\n", __FUNCTION__, pwm_mode));
	if (PWM_TONE_MODE == pwm_mode) {
		pwm_stop_timer(&pwm_timer);
	}

	pwm_printk(2, ("%s: wake_up_interruptible\n", __FUNCTION__));
	/* can release PWM now */
	wake_up_interruptible(&pwm_wait);

	pwm_printk(2, ("%s: data completed, PWMC = %08X\n",
		       __FUNCTION__, (U32) rg_PWMC));
}

module_init(pwm_init);
module_exit(pwm_cleanup);
