/*	cs5535_wdt.c	
 *	AMD Geode GX2 CS5535 WDT driver for Linux 2.4.x
 *      May also work on other similar models
 *
 *	Author: MontaVista Software, Inc.
 *         source@mvista.com
 *       
 *	2002-2004 (c)MontaVista Software, Inc. This file is licensed under
 * 	the terms of the GNU General Public License version 2. This program
 * 	is licensed "as is" without any warranty of any kind, whether express
 * 	or implied.
 *
 *	Based on advantechwdt.c which is based on wdt.c.
 *	Original copyright messages:
 *
 *	(c) Copyright 1996-1997 Alan Cox <alan@redhat.com>, All Rights Reserved.
 *				http://www.redhat.com
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *	
 *	Neither Alan Cox nor CymruNet Ltd. admit liability nor provide 
 *	warranty for any of this software. This material is provided 
 *	"AS-IS" and at no charge.	
 *
 *	(c) Copyright 1995    Alan Cox <alan@lxorguk.ukuu.org.uk>
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>
#include <linux/spinlock.h>

#include "cs5535_wdt.h"

static unsigned long wdt_is_open;
static spinlock_t wdt_lock;
static int expect_close = 0;

#ifdef CONFIG_WATCHDOG_NOWAYOUT
static int nowayout = 1;
#else
static int nowayout = 0;
#endif

#define WD_TIMO	300		/* 1 minute */
#define WD_FREQ	125		/* 125 ticks/sec */

static int wd_margin = WD_TIMO;
static unsigned int wd_freq = WD_FREQ;
static int max_wd_margin;
static int running;

MODULE_PARM(nowayout, "i");
MODULE_PARM_DESC(nowayout, "Watchdog cannot be stopped once started \
(default=CONFIG_WATCHDOG_NOWAYOUT)");

/* LBAR IO base adress */
static unsigned long cs5535_gpt_base;

static inline void
cs5535_gpt_outw(unsigned int offset, unsigned int what)
{
	outw(what, cs5535_gpt_base + offset);
}

static inline unsigned int
cs5535_gpt_inw(unsigned int offset)
{
	return inw(cs5535_gpt_base + offset);
}

static void
cs5535_wdt_ping(void)
{
	/* trigger watchdog */
	spin_lock(&wdt_lock);
	cs5535_gpt_outw(MFGPT0_CNT, 0);
	spin_unlock(&wdt_lock);
}

static int
freq2setup(int freq)
{
	int amnt = sizeof (cs5535_freq) / sizeof (struct frequencies);
	int i;
	for (i = 0; i < amnt; i++) {
		if (cs5535_freq[i].freq == freq)
			return cs5535_freq[i].setup_flags;
	}
	return -1;
}

static void
cs5535_wdt_start(void)
{
	unsigned long val, dummy;
	/*Compute the comparator value */
	int latch = wd_freq * wd_margin;
	if (!running) {
		pr_info("WDT: starting up, timeout= %d sec\n", wd_margin);
		/* Enable board reset on timeout */
		rdmsr(MFGPT_NR, val, dummy);
		val |= MFGPT0_C2_RSTEN;
		wrmsr(MFGPT_NR, val, dummy);

		/*We will use comparator 2 */
		cs5535_gpt_outw(0x02, latch);
		/*Enable the counter... */
		cs5535_gpt_outw(MFGPT0_SETUP,
				CS5535_GPT_EN | CS5535_GPT_EXT_EN |
				CS5535_MFGPT_CMP2MODE | freq2setup(wd_freq));
		running = 1;
	}
}

static void
cs5535_wdt_stop(void)
{
	unsigned int val, dummy = 0;
	val = cs5535_gpt_inw(MFGPT0_SETUP);

	/* stop watchdog */
	cs5535_gpt_outw(MFGPT0_SETUP, val & ~CS5535_GPT_EN);

	/* Disable board reset on timeout */
	val = 0;
	wrmsr(MFGPT_NR, val, dummy);

	running = 0;
}

static ssize_t
cs5535_wdt_read(struct file *file, char *buf, size_t count, loff_t * ptr)
{
	static int once;
	char cp[256];
	int i;
	unsigned int val = cs5535_gpt_inw(MFGPT0_CNT);

	/*  Can't seek (pread) on this device  */
	if (ptr != &file->f_pos)
		return -ESPIPE;
	if (!once) {
		once = 1;
		i = sprintf(cp, "%d\n", val);
		if (copy_to_user(buf, cp, i + 1))
			return -EFAULT;
		return i;
	} else {
		once = 0;
		return 0;
	}

}

static ssize_t
cs5535_wdt_write(struct file *file, const char *buf,
		 size_t count, loff_t * ppos)
{
	char buff[256];

	/*  Can't seek (pwrite) on this device  */
	if (ppos != &file->f_pos)
		return -ESPIPE;
	if (count > 256) {
		cs5535_wdt_ping();
		return -EINVAL;
	}
	if (count) {
		if (copy_from_user(buff, buf, count))
			return -EFAULT;
		if (strstr(buff, MAGIC) && !nowayout) {
			pr_info("%s: WDT got magic sequence!\n", __FILE__);
			expect_close = 1;
			return 1;
		}
		cs5535_wdt_ping();
		return 1;
	}
	return 0;
}

static int
cs5535_wdt_ioctl(struct inode *inode, struct file *file,
		 unsigned int cmd, unsigned long arg)
{
	int new_margin;
	static struct watchdog_info ident = {
		WDIOF_KEEPALIVEPING | WDIOF_SETTIMEOUT | WDIOF_MAGICCLOSE,
		1, "GX2 CS5535 WDT"
	};
	int one = 1;

	switch (cmd) {
	case WDIOC_GETSUPPORT:
		if (copy_to_user
		    ((struct watchdog_info *) arg, &ident, sizeof (ident)))
			return -EFAULT;
		break;

	case WDIOC_GETSTATUS:
		if (copy_to_user((int *) arg, &one, sizeof (int)))
			return -EFAULT;
		break;

	case WDIOC_KEEPALIVE:
		cs5535_wdt_ping();
		break;

	case WDIOC_SETTIMEOUT:
		if (get_user(new_margin, (int *) arg))
			return -EFAULT;
		if (new_margin > max_wd_margin)
			return -EINVAL;
		wd_margin = new_margin;
		cs5535_wdt_stop();
		cs5535_wdt_start();
		/* Fall */
	case WDIOC_GETTIMEOUT:
		return put_user(wd_margin, (int *) arg);

	default:
		return -ENOTTY;
	}
	return 0;
}

static int
cs5535_wdt_open(struct inode *inode, struct file *file)
{
	if (test_and_set_bit(0, &wdt_is_open))
		return -EBUSY;
	cs5535_wdt_start();

	return 0;
}

static int
cs5535_wdt_close(struct inode *inode, struct file *file)
{
	clear_bit(0, &wdt_is_open);
	if (expect_close) {
		pr_info("WDT: Stopping watchdog timer.\n");
		cs5535_wdt_stop();
		expect_close = 0;
	}
	return 0;
}

/*
 *	Notifier for system down
 */

static int
cs5535_wdt_notify_sys(struct notifier_block *this,
		      unsigned long code, void *unused)
{
	if (code == SYS_DOWN || code == SYS_HALT) {
		/* Turn the WDT off */
		cs5535_wdt_stop();
	}
	return NOTIFY_DONE;
}

/*
 *	Kernel Interfaces
 */

static struct file_operations wdt_fops = {
	.owner = THIS_MODULE,
	.write = cs5535_wdt_write,
	.read = cs5535_wdt_read,
	.ioctl = cs5535_wdt_ioctl,
	.open = cs5535_wdt_open,
	.release = cs5535_wdt_close,
};

static struct miscdevice wdt_miscdev = {
	.minor = WATCHDOG_MINOR,
	.name  = "watchdog",
	.fops  = &wdt_fops
};

/*
 *	The WDT needs to learn about soft shutdowns in order to
 *	turn the timebomb registers off. 
 */

static struct notifier_block wdt_notifier = {
	.notifier_call = cs5535_wdt_notify_sys
};

static int __init
cs5535_wdt_init(void)
{
	unsigned long dummy;
	rdmsr(MSR_LBAR_MFGPT, cs5535_gpt_base, dummy);
	if (misc_register(&wdt_miscdev) < 0)
		goto error;
	max_wd_margin = (int) (65530 / wd_freq);
	pr_info("%s: WDT driver for AMD Geode GX2.GPT IO at 0x%lx\n",
		__FILE__, cs5535_gpt_base);
	pr_info("%s: Frequency %d, timeout %d sec (up to %d)\n",
		__FILE__, wd_freq, wd_margin, max_wd_margin);

	/*Enable the counter... */
	cs5535_wdt_start();

	return 0;
      error:
	return -ENODEV;
}

static void __exit
cs5535_wdt_exit(void)
{
	misc_deregister(&wdt_miscdev);
	cs5535_wdt_stop();
	unregister_reboot_notifier(&wdt_notifier);
}

module_init(cs5535_wdt_init);
module_exit(cs5535_wdt_exit);

MODULE_AUTHOR("MontaVista Software Inc. <source@mvista.com>");
MODULE_LICENSE("GPL");

/* end of cs5535_wdt.c */
