#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/irq.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <asm/tx4925/basil_s1/basil_s1.h>
#include <asm/uaccess.h>



/********************************************************/
/*	Macro Definition 				*/
/********************************************************/
/* Definition of Device MajorNo */
#define TACTSW_CHAR_MAJOR		233

#define TACTSW_UNPUSHED			0
#define TACTSW_PUSHED			1

#define TACTSW_IRQ			TX4925_IRQ_IRC_INT7
#define TACTSW_VERSION			"1.00"

#define MSEC2TICK(ms)			(ms * HZ / 1000)
#define TICK2MSEC(tick)			(tick * 1000 / HZ)


/* get Tact-SW status Macro */
#define BASIL_S1_TACTSW			((u32)(1 << 29))
#define TACTSW				TX4925_RD(TX4925_MKA(TX4925_PIO_PIODI))

#define INT_REGISTER_IN_OPEN	/* register INT-handler in open */


#undef TACT_DEBUG

#if defined(TACT_DEBUG)
#define DBG_PRINT(arg...)	printk(__FUNCTION__ arg)
#define DBG_PRINT_START		printk(__FUNCTION__": start\n")
#else
#define DBG_PRINT(arg...)
#define DBG_PRINT_START
#endif


/********************************************************/
/*	Function & Structure Definition			*/
/********************************************************/
static int basil_tactsw_open(struct inode * inode, struct file * file);
static int basil_tactsw_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg);
static int basil_tactsw_close(struct inode * inode, struct file * file);
static unsigned int basil_tactsw_poll(struct file *file, poll_table *wait);

struct tactswdev {
	wait_queue_head_t wait;
	spinlock_t lock;
	unsigned char ready;			/* polling status */
	unsigned int state;			/* current status */
	unsigned int sw_state;			/* recent sw status */
	unsigned long push_start_time;		/* push start time */
	unsigned long push_total_time;		/* push detect time */
	struct timer_list timer;		/* unpush expired timer */
	unsigned long poll_time;		/* polling interval */
};



/********************************************************/
/*	Global Variables 				*/
/********************************************************/
static struct tactswdev tactsw_dev;


/********************************************************/
/*	Get TactSw State				*/
/********************************************************/
static inline void basil_tactsw_state(void)
{
	unsigned int old_state = tactsw_dev.sw_state;

	spin_lock_irq(&tactsw_dev.lock);
	tactsw_dev.sw_state = ((TACTSW & BASIL_S1_TACTSW) ? TACTSW_UNPUSHED: TACTSW_PUSHED);
#if 0
	DBG_PRINT(":old(%d) new(%d) mask(0x%x) org(0x%x)\n",
		old_state, tactsw_dev.state, BASIL_S1_TACTSW, TACTSW);
#endif
	if (tactsw_dev.state == TACTSW_PUSHED) {
		if ((old_state == TACTSW_UNPUSHED) &&
			(tactsw_dev.sw_state == TACTSW_UNPUSHED)) {
			/* push state -> un-push state */
			DBG_PRINT(": Changed to unpushed state");
			/* set pushed total time */
			tactsw_dev.push_total_time = jiffies - tactsw_dev.push_start_time;
			DBG_PRINT(" span(%ld) cur(%ld), start(%ld)\n", tactsw_dev.push_total_time, jiffies, tactsw_dev.push_start_time);
			tactsw_dev.ready = 1;
			tactsw_dev.state = TACTSW_UNPUSHED;

			/* wake up the wait for select() */
			wake_up_interruptible(&tactsw_dev.wait);
			/* enable irq */
			enable_irq(TACTSW_IRQ);
		} else {
			/* modify expiration check timer */
			mod_timer(&tactsw_dev.timer, jiffies + tactsw_dev.poll_time);
		}
	} else {
		if (tactsw_dev.sw_state == TACTSW_UNPUSHED) {
			/* un-push state -> un-push state */
			DBG_PRINT(": Unchanged to pushed state\n");
			/* state is un-push state */
			enable_irq(TACTSW_IRQ);
		} else {
			if (old_state == TACTSW_PUSHED) {
				/* un-push state -> push state */
				DBG_PRINT(": Changed to pushed state\n");
				tactsw_dev.state = TACTSW_PUSHED;
			}
			/* modify expiration check timer */
			mod_timer(&tactsw_dev.timer, jiffies + tactsw_dev.poll_time);
		}
	}
	spin_unlock_irq(&tactsw_lock);
}

/********************************************************/
/*	TactSw Polling Timer handler			*/
/********************************************************/
static void basil_tactsw_timer(unsigned long data)
{
	DBG_PRINT_START;
	basil_tactsw_state();
}

/********************************************************/
/*	TactSw INT					*/
/********************************************************/
static void basil_tactsw_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	/* disable interrupt */
	disable_irq(irq);

	/* start polling timer */
	tactsw_dev.timer.expires = tactsw_dev.poll_time;
	add_timer(&tactsw_dev.timer);

	/* set push start time */
	tactsw_dev.push_start_time = jiffies;

	/* initialize parameters */
	tactsw_dev.state = TACTSW_UNPUSHED;
	tactsw_dev.sw_state = TACTSW_UNPUSHED;
	tactsw_dev.push_total_time = 0;
}


/********************************************************/
/*	TactSw PROC I/F					*/
/********************************************************/
static int basil_tactsw_read_proc(char *buf, char **start, off_t offset,
                         int count, int *eof, void *data)
{
        int len;
	char* p = buf;
	char* push[] = {"unpushed", "pushed"};

	p += sprintf(p, "--- Tact-SW Status ---\n"
			" Current State    = %d(%s)\n"
			" Current SW State = %d(%s)\n",
			tactsw_dev.state, push[tactsw_dev.state],
			tactsw_dev.sw_state, push[tactsw_dev.sw_state]);
	p += sprintf(p, " Polling interval = %ld\n",
			tactsw_dev.poll_time);
	p += sprintf(p, " Current time     = %ld (in jiffies)\n",
			jiffies);
	p += sprintf(p, " Start time       = %ld (in jiffies)\n",
			tactsw_dev.push_start_time);
	p += sprintf(p, " Total time       = %ld (in jiffies)\n",
			tactsw_dev.push_total_time);

	len = p - buf;
        *start = buf + offset;
        len -= offset;
        if (len > count) {
		*eof = 0;
		len = count;
	} else {
		*eof = 1;
	}

        if (len < 0) {
		len = 0;
	}
        return len;
}

/********************************************************/
/*	TACTSW Control Drivers				*/
/*							*/
/********************************************************/
struct file_operations basil_tactsw_fopes = {
	owner:		THIS_MODULE,
	open:		basil_tactsw_open,
	release:	basil_tactsw_close,
	ioctl:		basil_tactsw_ioctl,
	poll:		basil_tactsw_poll,
};

/********************************************************/
/*	TactSw Open					*/
/********************************************************/
static int basil_tactsw_open(struct inode * inode, struct file * file)
{
	DBG_PRINT_START;

#if defined(INT_REGISTER_IN_OPEN)
	if (request_irq(TACTSW_IRQ, basil_tactsw_interrupt, SA_INTERRUPT,
			 "TactSW", &tactsw_dev)) {
		printk(KERN_ERR "TactSW: Can't register IRQ %d\n", TACTSW_IRQ);
		return -EIO;
	}
#endif
	MOD_INC_USE_COUNT;

	tactsw_dev.poll_time = MSEC2TICK(100);	/* 100msec */
	tactsw_dev.ready = 0;

	return 0;
}

/********************************************************/
/*	TactSw Close					*/
/********************************************************/
static int basil_tactsw_close(struct inode * inode, struct file * file)
{
	DBG_PRINT_START;
#if defined(INT_REGISTER_IN_OPEN)
	free_irq(TACTSW_IRQ, &tactsw_dev);
#endif
	MOD_DEC_USE_COUNT;
	return 0;
}

/********************************************************/
/*	Normal TactSw Ioctl				*/
/********************************************************/
static int basil_tactsw_ioctl(struct inode *inode, struct file *filp,
				unsigned int cmd, unsigned long arg)
{
	unsigned long prm;

	DBG_PRINT_START;

	DBG_PRINT(": CMD = 0x%X\n", cmd);
	switch( cmd ){
	case TIOCSTCTRES:
		/* Set TactSW polling interval : 0x54d0 */
		/* arg is milli-seconds */
		tactsw_dev.poll_time = MSEC2TICK(arg);
		break;
	case TIOCGTCTTM:
		/* Get TactSW pushed total time : 0x54d1 */
		/* arg is milli-seconds */
		prm = TICK2MSEC(tactsw_dev.push_total_time);
		put_user(prm, ((unsigned long *)arg));
		break;
	default:
		return -EOPNOTSUPP;
	}
	return 0;
}

/********************************************************/
/*	TactSw Poll					*/
/********************************************************/
static unsigned int basil_tactsw_poll(struct file *file, poll_table *wait)
{
	unsigned int l;

//	DBG_PRINT_START;

	poll_wait(file, &tactsw_dev.wait, wait);

	spin_lock_irq(&tactsw_dev.lock);
	l = tactsw_dev.ready;
	tactsw_dev.ready = 0;
	spin_unlock_irq(&tactsw_dev.lock);

	if (l != 0) {
		return POLLIN | POLLRDNORM;
	}
	return 0;
}

/********************************************************/
/*	TactSw Init Module				*/
/********************************************************/
static int __init basil_tactsw_init_module(void)
{
	DBG_PRINT_START;

	if (register_chrdev(TACTSW_CHAR_MAJOR, "tactsw", &basil_tactsw_fopes)) {
		printk(KERN_NOTICE "Can't allocate major number %d for Tact-SW Driver.\n",
		       TACTSW_CHAR_MAJOR);
		return -EAGAIN;
	}
	create_proc_read_entry ("driver/tactsw", 0, 0, basil_tactsw_read_proc, NULL);
#if !defined(INT_REGISTER_IN_OPEN)
	if (request_irq(TACTSW_IRQ, basil_tactsw_interrupt, SA_INTERRUPT,
			 "TactSW", &tactsw_dev)) {
		printk(KERN_ERR "TactSW: Can't register IRQ %d\n", TACTSW_IRQ);
		return -EIO;
	}
#endif
	init_waitqueue_head(&tactsw_dev.wait);
	init_timer(&tactsw_dev.timer);
	tactsw_dev.timer.function = basil_tactsw_timer;
	tactsw_dev.lock = SPIN_LOCK_UNLOCKED;

	printk("Basil-S1 Tact-SW Driver v" TACTSW_VERSION ".\n");
	return 0;
}


/********************************************************/
/*	TactSw Cleanup Module				*/
/********************************************************/
void __exit basil_tactsw_cleanup_module(void)
{
	DBG_PRINT_START;

#if !defined(INT_REGISTER_IN_OPEN)
	free_irq(TACTSW_IRQ, &tactsw_dev);
#endif
	remove_proc_entry("driver/tactsw", NULL);
	unregister_chrdev(TACTSW_CHAR_MAJOR, "tactsw");
}

module_init(basil_tactsw_init_module);
module_exit(basil_tactsw_cleanup_module);

