/*******************************************************************************
 * (C) Copyright Koninklijke Philips Electronics NV 2004. All rights reserved.
 *
 * You can redistribute and/or modify this software under the terms of
 * version 2 of the GNU General Public License as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
*******************************************************************************/
#ifndef PH_ADC_PRIVATE_H_
#define PH_ADC_PRIVATE_H_

/*------------------------------------------------------------------------------
Standard include files:
------------------------------------------------------------------------------*/
#include "vhtypes.h"

/*------------------------------------------------------------------------------
Project include files:
------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
Types and defines:
------------------------------------------------------------------------------*/
/**
 *
 * @brief Return value when the channel is not enabled for conversion.
 *
 * Return value when the channel is not enabled for conversion.
 */
#define PH_ADC_ERR_CH_NOT_ENABLED 0x0066    /* Channel not selected */


/*------------------------------------------------------------------------------
Global variables:
------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
Function protoypes:
------------------------------------------------------------------------------*/
/**
* @brief The ADC phAdc_Init service function. (HwApi layer)
*
* This function is used to initialise the ADC interface/conversion. The
* application programmer has the ability to setup the ADC for `Continuous`
* scanning mode or `Single scan` mode.
*
* REMARK: `Single scan mode` and `Continuous scan mode` are mutually exclusive.
*
* - The `ADC_ENABLE` bit is set.
* - The `ADC_CSCAN`  bit is set in case of `Continuous scan` mode
*
* Function is not re-entrant.
*
* @see                   phAdcScanMode_en_t
* @param                 eScanMode : ADC scanning mode
* @returns               Returns an error status code
* @retval                PH_OK : in case of success
* @retval                PH_ERR_INIT_FAILED : could not initialise
*/
phErrorCode_t phhwAdc_Init(phAdcScanMode_en_t eScanMode);

/**
* @brief The ADC phAdc_DeInit service function. (HwApi layer)
*
* This function de-initialises the ADC.
*
* Function is not re-entrant.
*
* @see                   None
* @returns               Returns an error status code
* @retval                PH_OK in case of success
*/
phErrorCode_t phhwAdc_DeInit(void);

/**
* @brief The ADC phhwAdc_ContinuousChEnable service function. (HwApi layer)
*
* This function is used to enable channels for continuous mode.
*
* Function is not re-entrant.
*
* @see        phAdcResolution_en_t
* @param      resol channel resolution
* @param      chSelectBitmask : bitmask of the selected channels
* @returns    Returns an error status code
* @retval     PH_OK : in case of success
* @retval     PH_ADC_ERR_WRONG_ADC_MODE : in case of wrong ADC operating mode
* @retval     PH_ERR_NOT_INITIALIZED : no initialisation occured
*/
phErrorCode_t phhwAdc_ContinuousChEnable(phAdcResolution_en_t resol,
                                         UInt32                 chSelectBitmask);

/**
* @brief The ADC phhwAdc_ContinuousChDisable service function. (HwApi layer)
*
* This function is used to disable channels for continuous mode.
*
* Function is not re-entrant.
*
* @see        None
* @param      chSelectBitmask : bitmask of the selected channels
* @returns    Returns an error status code
* @retval     PH_OK : in case of success
* @retval     PH_ADC_ERR_WRONG_ADC_MODE : in case of wrong ADC operating mode
* @retval     PH_ERR_NOT_INITIALIZED : no initialisation occured
*/
phErrorCode_t phhwAdc_ContinuousChDisable(UInt32 chSelectBitmask);

/**
* @brief The ADC phhwAdc_ContinuousConversionStart function. (HwApi layer)
*
* This function is used to start and setup the conversion for continuous mode.
* The application programmer can select the input reference that should be 
* taken for the measurement.
* 
*	The ADC `Voltage reference 0` and `Voltage reference 1` are mutually exclusive
*
* - The `ADC_START` bit is set.
*
* Function is not re-entrant.
*
* @see        phAdcVref_en_t
* @param      vref : voltage reference selection
* @returns    Returns an error status code
* @retval     PH_OK : in case of success
* @retval     PH_ADC_ERR_WRONG_ADC_MODE : in case of wrong ADC operating mode
* @retval     PH_ERR_NOT_INITIALIZED : no initialisation occured
* @retval     PH_ERR_BAD_PARAMETER : bad parameter
*/
phErrorCode_t phhwAdc_ContinuousConversionStart(phAdcVref_en_t vref);

/**
* @brief The ADC phAdc_SingleConversionStart function. (HwApi layer)
*
* This function is used to start and setup the conversion for single mode.
* The application programmer can select the input reference, resolution and
* which analog input signal should be taken for the measurement.
* 
* - The `ADC_START` bit is set.
*
* Function is not re-entrant.
*
* @see                   phAdcVref_en_t
* @see                   phAdcResolution_en_t
* @see                   phAdcChannel_en_t
* @param                 vref : voltage reference selection
* @param                 resol : resolution selection
* @param                 channel : channel to read from
* @returns               Returns an error status code
* @retval                PH_OK : in case of success.
* @retval                PH_ERR_NOT_INITIALIZED : no initialisation occured
* @retval                PH_ADC_ERR_WRONG_ADC_MODE : wrong ADC mode
* @retval                PH_ERR_BAD_PARAMETER : bad parameter
*/
phErrorCode_t phhwAdc_SingleConversionStart(phAdcVref_en_t       vref,
                                            phAdcResolution_en_t resol,
                                            phAdcChannel_en_t    channel);

/**
* @brief The ADC phhwAdc_Read function. (HwApi layer)
*
* This function is used to read a value from the ADC. 
* This function is a `SYNCHRONOUS` function.
*
* REMARK1: Concurrancy problems are solved in a higher level ! 
*
* Function is not re-entrant.
*
* @see    phAdcChannel_en_t
* @param  channel : channel to read from
* @param  pResult : pointer to the result
* @return Returns an error status code
* @retval PH_OK : in case of success
* @retval PH_ERR_NOT_INITIALIZED : no initialisation occured
* @retval PH_ADC_ERR_CH_NOT_ENABLED : channel is not selected for conversion
* @retval PH_ERR_BAD_PARAMETER : bad parameter
*/
phErrorCode_t phhwAdc_Read(phAdcChannel_en_t channel,
                           UInt32*             pResult);

#endif /* PH_ADC_PRIVATE_H_ */
