/*******************************************************************************
 * (C) Copyright Koninklijke Philips Electronics NV 2004. All rights reserved.
 *
 * You can redistribute and/or modify this software under the terms of
 * version 2 of the GNU General Public License as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *******************************************************************************/
/**
 *
 * The ADC is able to convert one of its 8 analog input signals with a 
 * conversion rate of 400kS/s for a 10 bits resolution. But the resolution can
 * be reduced to 2 bits and the conversion rate to 1500kSamples/s. The ADC is 
 * composed of an analog mux 8:1 to select the input to convert, two 2:1 analog
 * muxes to select the reference pair, a resistor string DAC and a successive 
 * approximation register. One 10 bits conversion needs 11 clock cycles. During
 * the first cycle the selected input signal is sampled then the successive
 * approximation determines the output code during the last 10 clock cycles.
 *
 * Actions for `single mode` scan 
 *   - step 1 : phAdc_Init(PH_ADC_SINGLE_MODE)
 *   - step 2 : phAdc_SingleRead(vref,resol,channel,&pResult)
 *   - ...
 *   - step 3 : phAdc_SingleRead(vref,resol,channel,&pResult)
 *   - step 4 : phAdc_SingleRead(vref,resol,channel,&pResult)
 *   - ...  
 *   - step 5 : phAdc_DeInit()
 *
 *
 * Actions for `continunous mode` scan 
 *   - step 1 : phAdc_Init(PH_ADC_CONTIN_MODE)
 *   - step 2 : phAdc_ContinuousSetup(vref,resol,chSelectBitmask);
 *   - step 3 : phAdc_ContinuousRead(channel,&pResult);
 *   - ...
 *   - step 4 : phAdc_ContinuousRead(channel,&pResult);
 *   - step 5 : phAdc_ContinuousRead(channel,&pResult);
 *   - ...
 *   - step 6 : phAdc_DeInit()
 *
 * Assumptions:
 * - `Single scan mode` and `Continuous scan mode` are mutually exclusive.
 * - The ADC `Voltage reference 0` and `Voltage reference 1` are mutually exclusive
 *
 * Functional limitations:
 * - Maximum conversion rate 400k samples for 10 bits resolution.
 * @{-
 */

#ifndef PH_ADC_H_
#define PH_ADC_H_

/*------------------------------------------------------------------------------
  Standard include files:
  ------------------------------------------------------------------------------*/
#include "vhtypes.h"

/*------------------------------------------------------------------------------
  Project include files:
  ------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Adaptation to Linux:
  ------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Local Types and defines:
  ------------------------------------------------------------------------------*/
/**
 * ADC API Errror Codes
 */
#define PH_ADC_ERR_BAD_PARAMETER      201
#define PH_ADC_ERR_NOT_INITIALIZED    202
#define PH_ADC_ERR_RESOURCE_OWNED     203 
#define PH_ADC_ERR_INIT_FAILED        204
#define PH_ADC_ERR_ALREADY_SETUP      205
#define PH_ADC_ERR_CH_NOTSELECTED     206
#define PH_ADC_ERR_WRONG_ADC_MODE     207

/**
 * @brief Reference voltage enumeration
 *
 * The ADC has the possibility to select between two reference voltages.
 */
typedef enum phAdcVref_en
{
  PH_ADCVREF_0 = 0, 
  PH_ADCVREF_1 = 1
} phAdcVref_en_t;

/**
 * @brief Resolution enumeration
 *
 * The resolution of the ADC can be adjusted between 2 and 10 bits.
 */
typedef enum phAdcResolution_en
{
  PH_ADCRESOLUTION_2BITS  =  2, 
  PH_ADCRESOLUTION_3BITS  =  3, 
  PH_ADCRESOLUTION_4BITS  =  4, 
  PH_ADCRESOLUTION_5BITS  =  5, 
  PH_ADCRESOLUTION_6BITS  =  6, 
  PH_ADCRESOLUTION_7BITS  =  7, 
  PH_ADCRESOLUTION_8BITS  =  8, 
  PH_ADCRESOLUTION_9BITS  =  9, 
  PH_ADCRESOLUTION_10BITS = 10 
} phAdcResolution_en_t;

/**
 * @brief Scanning Mode enumeration
 *
 * The ADC has the possibility to select between two different scanning modes:
 * - CONTINUOUS scan mode : A/D conversions scans are carried out continuously:
 *   once one scan completed, the next one is started automatically
 * - SINGLE scan mode : Only a single conversion scan is carried out, the next
 *   scan must be started explicitely by the software.
 */
enum phAdcScanMode_en
{
  PH_ADC_CONTIN_MODE = 0,   /* continuous scan mode */
  PH_ADC_SINGLE_MODE = 1		 /* single     scan mode */
};
 
typedef enum phAdcScanMode_en phAdcScanMode_en_t;

/**
 * @brief Channel selection enumeration
 *
 * The ADC has the possibility to select one of 8 analog input signals.
 */
typedef enum phAdcChannel_en
{
  PH_ADCCHANNEL_0    = 0, 
  PH_ADCCHANNEL_1    = 1,
  PH_ADCCHANNEL_2    = 2,
  PH_ADCCHANNEL_3    = 3,
  PH_ADCCHANNEL_4    = 4,
  PH_ADCCHANNEL_5    = 5,
  PH_ADCCHANNEL_6    = 6,
  PH_ADCCHANNEL_7    = 7
} phAdcChannel_en_t;

/**
 *
 * @brief Bitmask for ADC channel 0.
 *
 * Bitmask for ADC channel 0.
 */
#define PH_ADC_MSK_CHANNEL_0 0x0000000F		  /* bitmask ADC channel 0 */

/**
 *
 * @brief Bitmask for ADC channel 1.
 *
 * Bitmask for ADC channel 1.
 */
#define PH_ADC_MSK_CHANNEL_1 0x000000F0      /* bitmask ADC channel 1 */

/**
 *
 * @brief Bitmask for ADC channel 2.
 *
 * Bitmask for ADC channel 2.
 */
#define PH_ADC_MSK_CHANNEL_2 0x00000F00		 	/* bitmask ADC channel 2 */

/**
 *
 * @brief Bitmask for ADC channel 3.
 *
 * Bitmask for ADC channel 3.
 */
#define PH_ADC_MSK_CHANNEL_3 0x0000F000			/* bitmask ADC channel 3 */

/**
 *
 * @brief Bitmask for ADC channel 4.
 *
 * Bitmask for ADC channel 4.
 */
#define PH_ADC_MSK_CHANNEL_4 0x000F0000			/* bitmask ADC channel 4 */

/**
 *
 * @brief Bitmask for ADC channel 5.
 *
 * Bitmask for ADC channel 5.
 */
#define PH_ADC_MSK_CHANNEL_5 0x00F00000		  /* bitmask ADC channel 5 */

/**
 *
 * @brief Bitmask for ADC channel 6.
 *
 * Bitmask for ADC channel 6.
 */
#define PH_ADC_MSK_CHANNEL_6 0x0F000000		 	/* bitmask ADC channel 6 */

/**
 *
 * @brief Bitmask for ADC channel 7.
 *
 * Bitmask for ADC channel 7.
 */
#define PH_ADC_MSK_CHANNEL_7 0xF0000000		 	/* bitmask ADC channel 7 */

/**
 *
 * @brief Maximum number of ADC channels
 *
 * Maximum number of ADC channels
 */
#define PH_ADC_MAX_CHANNELS 8

/*------------------------------------------------------------------------------
  Global variables:
  ------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
Function protoypes:
------------------------------------------------------------------------------*/
/**
* @brief The ADC phAdc_Init service function. (IC driver layer)
*
* This function is used to initialise the ADC interface/conversion. The
* application programmer has the ability to setup the ADC for `Continuous scann`
* mode or `Single scan` mode.
*
* REMARK: `Single scan mode` and `Continuous scan mode` are mutually exclusive.
*
* Function is re-entrant.
*
* @see                 None
* @param               scanMode : ADC scanning mode
* @returns             Returns an error status code.
* @retval              PH_OK : in case of no erros
* @retval              PH_ERR_RESOURCE_OWNED : resource is already in use
* @retval              PH_LLIC_ADC_ERR_MUTEX_INIT : could not initialise mutex
*/
phErrorCode_t phAdc_Init(phAdcScanMode_en_t scanMode);

/**
* @brief The ADC phAdc_DeInit service function. (IC driver layer)
*
* This function de-initialises the ADC.
*
* Function is not re-entrant.
*
* @see                 None
* @returns             Returns an error status code.
* @retval              PH_OK : in case of no erros
*/
phErrorCode_t phAdc_DeInit(void);

/**
* @brief The ADC phAdc_ContinuousSetup function. (IC driver layer)
*
* This function is used to start and setup the conversion for continuous mode.
* The application programmer has the ability to select channels for continuous
* conversion. It is possible to setup ADC channel 2 and 7 for continuous mode. 
* The selection of these channels are determined by the `chSelectBitmask`
* parameter.
*
* - Single scan mode` and `Continuous scan mode` are mutually exclusive.
*
* Function is not re-entrant.
* 
* @see     phAdcVref_en_t
* @see     phAdcResolution_en_t
* @param   vref : voltage reference selection
* @param   resol : resolution selection 
* @param   chSelectBitmask : bitmask of the selected channels
* @returns Returns an error status code
* @retval  PH_OK : in case of success
* @retval  PH_ADC_ERR_WRONG_ADC_MODE : in case of wrong ADC operating mode
* @retval  PH_ERR_NOT_INITIALIZED : no initialisation occured
* @retval  PH_LLIC_ADC_ERR_ALREADY_SETUP : already configured for something else
*/
phErrorCode_t phAdc_ContinuousSetup(phAdcVref_en_t vref,
                                        phAdcResolution_en_t resol,
                                        UInt32                 chSelectBitmask);

/**
* @brief The ADC phAdc_ContinuousRead function. (IC driver layer)
*
* This function is used to read a value from the ADC in `continuous` mode.
* - OSAL functions are used to solve concurrency problems.
* - This function is a `SYNCHRONOUS` function.
*
* Function is re-entrant.
*
* @see           phAdcChannel_en_t
* @param         channel : input channel selection
* @param         pResult : pointer for storing the result
* @returns       Returns an error status code.
* @retval        PH_OK : in case of success
* @retval        PH_ERR_NOT_INITIALIZED : no initialisation occured
* @retval        PH_ADC_ERR_CH_NOT_SELECTED : channel is not selected
* @retval        PH_ADC_ERR_WRONG_ADC_MODE : wrong ADC operating mode
*/
phErrorCode_t phAdc_ContinuousRead(phAdcChannel_en_t channel,
                                       UInt32*             pResult);

/**
* @brief The ADC phAdc_SingleRead function. (IC driver layer)
*
* This function is used to read a value from the ADC in `single mode`.
* - OSAL functions are used to solve concurrency problems.
* - This function is a `SYNCHRONOUS` function.
*
* Function is re-entrant.
*
* @see           phAdcVref_en_t
* @see           phAdcResolution_en_t
* @see           phAdcChannel_en_t
* @param         vref : voltage reference selection
* @param         resol : resolution selection
* @param         channel : input channel selection
* @param         pResult : pointer for storing the result
* @returns       Returns an error status code.
* @retval        PH_OK : in case of success
* @retval        PH_ERR_NOT_INITIALIZED : no initialisation occured
* @retval        PH_ADC_ERR_CH_NOT_SELECTED : channel is not selected
* @retval        PH_ADC_ERR_WRONG_ADC_MODE : wrong operating mode
*/
phErrorCode_t phAdc_SingleRead(phAdcVref_en_t       vref,
                                  phAdcResolution_en_t  resol,
                                  phAdcChannel_en_t     channel,
                                  UInt32*                 pResult);

#endif /* PH_ADC_H_ */
