/*******************************************************************************
 * (C) Copyright Koninklijke Philips Electronics NV 2004. All rights reserved.
 *
 * You can redistribute and/or modify this software under the terms of
 * version 2 of the GNU General Public License as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *******************************************************************************/
/*------------------------------------------------------------------------------
  Standard include files:
  ------------------------------------------------------------------------------*/
#include "vhtypes.h"

#define PH_OK 0
typedef unsigned int phErrorCode_t; 
/*------------------------------------------------------------------------------
  Project include files:
  ------------------------------------------------------------------------------*/
#include "phadc.h"
#include "phadc_private.h"
#include "vhssa1adc.h"

/*------------------------------------------------------------------------------
  Adaptation to Linux:
  ------------------------------------------------------------------------------*/
#include <asm/semaphore.h>
#include <asm/arch/platform.h>
#include <asm/arch/hardware.h>
#include <asm/io.h>

/*------------------------------------------------------------------------------
  Local Types and defines:
  ------------------------------------------------------------------------------*/
#ifndef DEBUG
#define MLPP_PRINTK(...) do {} while(0)
#else
#define MLPP_PRINTK(...) printk(__VA_ARGS__)
#endif /* NDEBUG */

/**
* CGU - ADC REGISTERS
*/
#define PNX0105_CGU_ADC_SCR    0x004
#define PNX0105_CGU_ADC_FDC    0x4B8
#define PNX0105_CGU_ADC_ESR    0x3F8
#define PNX0105_CGU_ADC_BCR    0x498
#define PNX0105_CGU_ADC_SSR    0x0B8
#define PNX0105_CGU_ADC_PSR    0x2BC
/*------------------------------------------------------------------------------
  Global variables:
  ------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Static Variables:
  ------------------------------------------------------------------------------*/

/**
 * @brief Variable is used to indicate if the ADC is initialised.
 *
 * Variable is used to indicate if the ADC is initialised.
 */
static UInt32 sAdcIsAdcInitialised = FALSE;

/**
 * @brief Variable is used to indicate if the conversion started.
 *
 * Variable is used to indicate if the conversion started.
 */
static UInt32 sAdcDidWeSetupForConversion = FALSE;

/**
 * @brief Pointer to the Adc mutex object.
 *
 * Pointer to the Adc mutex object.
 */
static struct semaphore spAdcMutex;
/**
 * @brief Array is used to keep track of the enabled channels in 
 * continuous mode.
 *
 * Array is used to keep track of the enabled channels in continuous mode.
 *
 * @see None.
 *
 */
UInt32 sAdcEnabledChanel[PH_ADC_MAX_CHANNELS];

/**
 * @brief Pointer is used to point to the ADC registers.
 *
 * Pointer is used to point to the ADC registers..
 *
 * @see None.
 *
 */

/*------------------------------------------------------------------------------
  Static prototypes:
  ------------------------------------------------------------------------------*/
static inline unsigned long CGU_Read(unsigned int Register);
static inline void CGU_Write(unsigned int Register, unsigned long Value);

/*------------------------------------------------------------------------------
  Exported functions:
  ------------------------------------------------------------------------------*/

/**
 * @brief The ADC phAdc_Init service function. (IC driver layer)
 *
 * This function is used to initialise the ADC interface/conversion. The
 * application programmer has the ability to setup the ADC for `Continuous scan`
 * mode or `Single scan` mode.
 *
 * REMARK: `Single scan mode` and `Continuous scan mode` are mutually exclusive.
 *
 * Function is re-entrant.
 *
 * @see                 phAdcScanMode_en_t
 * @param               scanMode : ADC scanning mode
 * @returns             Returns an error status code.
 * @retval              PM_OK : in case of no erros
 * @retval              PM_ERR_RESOURCE_OWNED : resource is already in use
 * @retval              PM_LLIC_ADC_ERR_MUTEX_INIT : could not initialise mutex
 */
phErrorCode_t phAdc_Init(phAdcScanMode_en_t scanMode)
{
  UInt32 result;
  unsigned long val;
  
  /* Initialise the ADC clock */
  /* Read Current Status */
  val = CGU_Read(PNX0105_CGU_ADC_FDC);
  MLPP_PRINTK("Def CGU_Read FDC 0x%08x\n", (unsigned int)val);
  val = CGU_Read(PNX0105_CGU_ADC_ESR);
  MLPP_PRINTK("Def CGU_Read ESR 0x%08x\n", (unsigned int)val);
  val = CGU_Read(PNX0105_CGU_ADC_SCR);
  MLPP_PRINTK("Def CGU_Read SCR 0x%08x\n", (unsigned int)val);
  val = CGU_Read(PNX0105_CGU_ADC_BCR);
  MLPP_PRINTK("Def CGU_Read BCR 0x%08x\n", (unsigned int)val);
  val = CGU_Read(PNX0105_CGU_ADC_SSR);
  MLPP_PRINTK("Def CGU_Read SSR 0x%08x\n", (unsigned int)val);
  val = CGU_Read(PNX0105_CGU_ADC_PSR);
  MLPP_PRINTK("Def CGU_Read PSR 0x%08x\n", (unsigned int)val);
  
  /* Reset FD 5 */
  val = 0x0;
  CGU_Write(PNX0105_CGU_ADC_BCR,val);
  val = CGU_Read(PNX0105_CGU_ADC_BCR);
  MLPP_PRINTK("CGU_Read BCR 0x%08x\n", (unsigned int)val);
  val = CGU_Read(PNX0105_CGU_ADC_FDC);
  MLPP_PRINTK("CGU_Read FDC 0x%08x\n", (unsigned int)val);
  val = 0x2;
  CGU_Write(PNX0105_CGU_ADC_FDC,val);
  val = CGU_Read(PNX0105_CGU_ADC_FDC);
  MLPP_PRINTK("CGU_Read FDC 0x%08x\n", (unsigned int)val);

  /* Config FD 5 */
  val = (0xffffffc0 << 0x00000008 | 0x00000080) << 3 | 0x01 << 2 | 0x01;
  CGU_Write(PNX0105_CGU_ADC_FDC,val);
  val = CGU_Read(PNX0105_CGU_ADC_FDC);
  MLPP_PRINTK("CGU_Read FDC 0x%08x\n", (unsigned int)val);
  
  /* Stop FD 5 */
  val = CGU_Read(PNX0105_CGU_ADC_FDC);
  val = val & ~(1 << 0);
  CGU_Write(PNX0105_CGU_ADC_FDC,val);
  val = CGU_Read(PNX0105_CGU_ADC_FDC);
  MLPP_PRINTK("CGU_Read FDC 0x%08x\n", (unsigned int)val);
  
  /* Div Select */
  val = CGU_Read(PNX0105_CGU_ADC_ESR);
  MLPP_PRINTK("CGU_Read ESR 0x%08x\n", (unsigned int)val);
  val = ((0x05 - 0x05) << 1) | (1 << 0);
  CGU_Write(PNX0105_CGU_ADC_ESR,val);
  val = CGU_Read(PNX0105_CGU_ADC_ESR);
  MLPP_PRINTK("CGU_Read ESR 0x%08x\n", (unsigned int)val);
  
 /* Run FD 5 */
  val = CGU_Read(PNX0105_CGU_ADC_FDC);
  val = val | (1 << 0);
  CGU_Write(PNX0105_CGU_ADC_FDC,val);
  val = CGU_Read(PNX0105_CGU_ADC_FDC);
  MLPP_PRINTK("CGU_Read FDC 0x%08x\n", (unsigned int)val);

  val = 0x1;
  CGU_Write(PNX0105_CGU_ADC_BCR,val);
  val = CGU_Read(PNX0105_CGU_ADC_BCR);
  MLPP_PRINTK("CGU_Read BCR 0x%08x\n", (unsigned int)val);

  /* Run Clock Base */
  val = CGU_Read(PNX0105_CGU_ADC_SCR);
  MLPP_PRINTK("CGU_Read SCR 0x%08x\n", (unsigned int)val);
  val = (0x00000002 & 3);
  CGU_Write(PNX0105_CGU_ADC_SCR,val);
  val = CGU_Read(PNX0105_CGU_ADC_SCR);
  MLPP_PRINTK("CGU_Read SCR 0x%08x\n", (unsigned int)val);
  /**/

  /* Initialise the ADC if it is not in use */
  if(sAdcIsAdcInitialised == FALSE)
    {
      result = phhwAdc_Init(scanMode);  

      if(result == PH_OK)
	{
	  /* Create a mutex */
	  init_MUTEX(&spAdcMutex);
                     
	  sAdcIsAdcInitialised = TRUE;
	}
    }
  else
    {
      result = PH_ADC_ERR_RESOURCE_OWNED;  
    }

  return(result);
}

/**
 * @brief The ADC phAdc_DeInit service function. (IC driver layer)
 *
 * This function de-initialises the ADC.
 *
 * Function is not-entrant.
 *
 * @see                 None
 * @returns             Returns an error status code.
 * @retval              PM_OK : in case of no erros
 */
phErrorCode_t phAdc_DeInit(void)
{
  phErrorCode_t error_status;
  
  /* de-initialise */
  error_status =  phhwAdc_DeInit();
  
  sAdcIsAdcInitialised        = FALSE;
  sAdcDidWeSetupForConversion = FALSE;

  return(error_status);
}

/**
 * @brief The ADC phAdc_ContinuousSetup function. (IC driver layer)
 *
 * This function is used to start and setup the conversion for continuous mode.
 * The application programmer has the ability to select channels for continuous
 * conversion. it is possible to setup ADC channel 2 and 7 for continuous mode. 
 * The selection of these channels are determined by the `chSelectBitmask`
 * parameter.
 *
 * - Single scan mode` and `Continuous scan mode` are mutually exclusive.
 *
 * Function is not re-entrant.
 * 
 * @see        phAdcVref_en_t
 * @see        phAdcResolution_en_t
 * @param      vref : voltage reference selection
 * @param      resol : resolution selection 
 * @param      chSelectBitmask : bitmask of the selected channels
 * @returns    Returns an error status code
 * @retval     PM_OK : in case of success
 * @retval     PM_HW_ADC_ERR_WRONG_ADC_MODE : in case of wrong ADC operating mode
 * @retval     PM_ERR_NOT_INITIALIZED : no initialisation occured
 * @retval     PM_LLIC_ADC_ERR_ALREADY_SETUP : already configured for something else
 */
phErrorCode_t phAdc_ContinuousSetup(phAdcVref_en_t       vref,
				    phAdcResolution_en_t resol,
				    UInt32                 chSelectBitmask)
{
  phErrorCode_t  error_status;    

  error_status = PH_OK;

  /* check if it is not already configured */
  if(sAdcDidWeSetupForConversion == TRUE)
    {
      return(PH_ADC_ERR_ALREADY_SETUP);
    }

  /* Enable the channels for `continunous mode` */
  error_status = phhwAdc_ContinuousChEnable(resol,chSelectBitmask);
  
  if(error_status != PH_OK)
    return error_status;

  /* Select voltage reference and start the conversion for `continuous mode`*/
  error_status = phhwAdc_ContinuousConversionStart(vref);

  if(error_status != PH_OK)
    return error_status;

  sAdcDidWeSetupForConversion = TRUE;

  return(error_status);
}





/**
 * @brief The ADC phAdc_ContinuousRead function. (IC driver layer)
 *
 * This function is used to read a value from the ADC in `continuous`.
 * - This function is a `SYNCHRONOUS` function.
 *
 * Function is re-entrant.
 *
 * @see           phAdcChannel_en_t
 * @param         channel : input channel selection
 * @param         eResult : pointer for storing the result
 * @returns       Returns an error status code.
 * @retval        PH_OK : in case of success
 * @retval        PH_ADC_ERR_NOT_INITIALIZED : no initialisation occured
 * @retval        PH_ADC_ERR_CH_NOT_SELECTED : channel is not selected
 * @retval        PH_ADC_ERR_WRONG_ADC_MODE : wrong ADC operating mode
 */

phErrorCode_t phAdc_ContinuousRead(phAdcChannel_en_t channel,
				   UInt32*             pResult)
{
  phErrorCode_t  error_status;    

  /* Read */
  error_status = phhwAdc_Read(channel,pResult);

  return(error_status);
}

/**
 * @brief The ADC phAdc_SingleRead function. (IC driver layer)
 *
 * This function is used to read a value from the ADC in `single mode`.
 * - This function is a `SYNCHRONOUS` function.
 *
 * Function is re-entrant.
 *
 * @see           phAdcVref_en_t
 * @see           phAdcResolution_en_t
 * @see           phAdcChannel_en_t
 * @param         vref : voltage reference selection
 * @param         resol : resolution selection
 * @param         channel : input channel selection
 * @param         eResult : pointer for storing the result
 * @returns       Returns an error status code.
 * @retval        PH_OK : in case of success
 * @retval        PH_ADC_ERR_NOT_INITIALIZED : no initialisation occured
 * @retval        PH_ADC_ERR_CH_NOT_SELECTED : channel is not selected
 * @retval        PH_ADC_ERR_WRONG_ADC_MODE : wrong operating mode
 */
phErrorCode_t phAdc_SingleRead(phAdcVref_en_t        vref,
				   phAdcResolution_en_t  resol,
				   phAdcChannel_en_t     channel,
				   UInt32*                 pResult)
{
  phErrorCode_t  error_status;    

  /* setup for conversion */
  error_status = phhwAdc_SingleConversionStart(vref,resol,channel);

  if(error_status == PH_OK)
    {
      /* Read */
      error_status = phhwAdc_Read(channel,pResult);
    }

  return(error_status);
}
/*******************************************************************************/
/*******************************************************************************/
/**
 * @brief The ADC phhwAdc_Init service function. (HwApi layer)
 *
 * This function is used to initialise the ADC interface/conversion. The
 * application programmer has the ability to setup the ADC for `Continuous`
 * scanning mode or `Single scan` mode.
 *
 * REMARK: `Single scan mode` and `Continuous scan mode` are mutually exclusive.
 *
 * - The `ADC_ENABLE` bit is set.
 * - The `ADC_CSCAN`  bit is set in case of `Continuous scan` mode
 *
 * Function is not re-entrant.
 *
 * @see                   phhwAdcScanMode_en_t
 * @param                 eScanMode : ADC scanning mode
 * @returns               Returns an error status code
 * @retval                PH_OK : in case of success
 * @retval                PH_ADC_ERR_INIT_FAILED : could not initialise
 */
phErrorCode_t phhwAdc_Init(phAdcScanMode_en_t eScanMode)
{
  unsigned int tmp;
  /* de-initialise everything. We write `0` to all the registers in order
  ** to have a clean start 
  */
  phhwAdc_DeInit();
  
    
  /* write ADC enable bit */
  tmp = readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);
  writel(tmp|VH_ADC_CON_ADC_ENABLE_STATUS_MSK,IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);

  /* select scan mode */
  if(eScanMode == PH_ADC_CONTIN_MODE)
    {
      /* write ADC_CSCAN bit */
      tmp = readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);
      writel (tmp|VH_ADC_CON_ADC_CSCAN_STATUS_MSK,IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);
    }
  /* Initialise some variables */
  sAdcIsAdcInitialised  = TRUE;

  return(PH_OK);
}
/**
 * @brief The ADC phhwAdc_DeInit service function. (HwApi layer)
 *
 * This function de-initialises the ADC.
 *
 * Function is not re-entrant.
 *
 * @see                   None
 * @returns               Returns an error status code
 * @retval                PH_OK in case of success
 */
phErrorCode_t phhwAdc_DeInit(void)
{
  /* read the base address */

  /* ADC_SELVREF = 0 */
  /* ADC_ENABLE  = 0 */
  /* ADC_CSCAN   = 0 */
  /* ADC_START   = 0 */
  /* ADC_STATUS  = 0 */
  writel(0,IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);

  /* ADC_SCAN_INT_ENABLE = 0 */
  writel(0,IO_ADDRESS(ADC_BASE)+VH_ADC_INT_ENABLE_REG);

  /* ADC_SCAN_INT_STATUS = 0 */
  writel(0,IO_ADDRESS(ADC_BASE)+VH_ADC_INT_STATUS_REG);

  /* ADC_CSEL0 = 0   */
  /* ADC_CSEL1 = 0   */
  writel(0,IO_ADDRESS(ADC_BASE)+VH_ADC_CSEL_RES_REG);

  /* Initialise some variables */
  sAdcIsAdcInitialised = FALSE;
  sAdcEnabledChanel[0] = FALSE;
  sAdcEnabledChanel[1] = FALSE;
  sAdcEnabledChanel[2] = FALSE;
  sAdcEnabledChanel[3] = FALSE;
  sAdcEnabledChanel[4] = FALSE;
  sAdcEnabledChanel[5] = FALSE;
  sAdcEnabledChanel[6] = FALSE;
  sAdcEnabledChanel[7] = FALSE;

  /* read the base address */

  return(PH_OK);
}





/**
 * @brief The ADC phhwAdc_ContinuousChEnable service function. (HwApi layer)
 *
 * This function is used to enable channels for continuous mode.
 *
 * @see        None
 * @param      resol channel resolution
 * @param      chSelectBitmask : bitmask of the selected channels
 * @returns    Returns an error status code
 * @retval     PH_OK : in case of success
 * @retval     PH_ADC_ERR_WRONG_ADC_MODE : in case of wrong ADC operating mode
 * @retval     PH_ADC_ERR_NOT_INITIALIZED : no initialisation occured
 *
 * Function is not re-entrant.
 */
phErrorCode_t phhwAdc_ContinuousChEnable(phAdcResolution_en_t resol,
                                         UInt32               chSelectBitmask)
{
  UInt32         counter;
  UInt32         channelX;
  UInt32         regValue;
  phErrorCode_t  error_status;

  /* Are we initialised */
  if(sAdcIsAdcInitialised == FALSE)
    return(PH_ADC_ERR_NOT_INITIALIZED); 

  error_status = PH_OK;

  /* Check if we are in `continuous mode` */
  if((readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG)) & VH_ADC_CON_ADC_CSCAN_STATUS_MSK)
    {
      /* Write Select Channel Resolution register */
      regValue = 0;

      /* check wich channels we should enable */
      for(counter = 0; counter < PH_ADC_MAX_CHANNELS; counter++)
        {
	  /* do we need to enable channel x */
	  channelX = chSelectBitmask & ( 0x0000000F << (counter*4));

	  if(channelX)
            {
	      regValue |= (UInt32)(((UInt32)resol<<(counter*4)) & channelX);

	      sAdcEnabledChanel[counter] = TRUE;
            }
        }
      writel(regValue,IO_ADDRESS(ADC_BASE)+VH_ADC_CSEL_RES_REG);
    }
  else
    {
      error_status = PH_ADC_ERR_WRONG_ADC_MODE;    
    }
  return(error_status);
}

/**
 * @brief The ADC phhwAdc_ContinuousChDisable service function. (HwApi layer)
 *
 * This function is used to disable channels for continuous mode.
 *
 * Function is not re-entrant.
 *
 * @see        None
 * @param      chSelectBitmask : bitmask of the selected channels
 * @returns    Returns an error status code
 * @retval     PH_OK : in case of success
 * @retval     PH_ADC_ERR_WRONG_ADC_MODE : in case of wrong ADC operating mode
 * @retval     PH_ADC_ERR_NOT_INITIALIZED : no initialisation occured
 */
phErrorCode_t phhwAdc_ContinuousChDisable(UInt32 chSelectBitmask)
{
  UInt32         counter;
  UInt32         channelX;
  UInt32         regValue;
  phErrorCode_t  error_status;

  /* Are we initialised */
  if(sAdcIsAdcInitialised == FALSE)
    return(PH_ADC_ERR_NOT_INITIALIZED); 

  error_status = PH_OK;

  /* Check if we are in `continuous mode` */
  if((readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG)) & VH_ADC_CON_ADC_CSCAN_STATUS_MSK)
    {
      /* Read Select Channel Resolution register */
      regValue = readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CSEL_RES_REG);

      /* check wich channels we should disable */
      for(counter = 0; counter < PH_ADC_MAX_CHANNELS; counter++)
        {
	  /* do we need to disable channel x */
	  channelX = chSelectBitmask & ( 0x0000000F << (counter*4));

	  if(channelX)
            {
	      regValue &= (UInt32)(~channelX);    

	      sAdcEnabledChanel[counter] = FALSE;
            }
        }
      writel(regValue,IO_ADDRESS(ADC_BASE)+VH_ADC_CSEL_RES_REG);
    }
  else
    {
      error_status = PH_ADC_ERR_WRONG_ADC_MODE;  
    }
  return(error_status);
}

/**
 * @brief The ADC phhwAdc_ContinuousConversionStart function. (HwApi layer)
 *
 * This function is used to start and setup the conversion for continuous mode.
 * The application programmer can select the input reference that should be 
 * taken for the measurement.
 * 
 * - The `ADC_START` bit is set.
 *
 * Function is not re-entrant.
 *
 * @see        phhwAdcVref_en_t
 * @param      vref : voltage reference selection
 * @returns    Returns an error status code
 * @retval     PH_OK : in case of success
 * @retval     PH_ADC_ERR_WRONG_ADC_MODE : in case of wrong ADC operating mode
 * @retval     PH_ADC_ERR_NOT_INITIALIZED : no initialisation occured
 * @retval     PH_ADC_ERR_BAD_PARAMETER : bad parameter
 */
phErrorCode_t phhwAdc_ContinuousConversionStart(phAdcVref_en_t vref)
{
  UInt32         regValue;  
  phErrorCode_t  error_status;
  unsigned int tmp;
  error_status = PH_OK;  

  /* Are we initialised */
  if(sAdcIsAdcInitialised == FALSE)
    return(PH_ADC_ERR_NOT_INITIALIZED); 

  /* Check if we are in `continuous mode` */
  if((readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG)) & VH_ADC_CON_ADC_CSCAN_STATUS_MSK)
    {
      /* select reference voltage input */
      regValue = readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);

      switch(vref)
        {
        case PH_ADCVREF_0 :
	  regValue &= ~(VH_ADC_CON_ADC_SELREF_STATUS_MSK);
	  break;
        case PH_ADCVREF_1 :
	  regValue |= VH_ADC_CON_ADC_SELREF_STATUS_MSK;
	  break;        
        default:
	  {
	    return(PH_ADC_ERR_BAD_PARAMETER);
	  }
	  break;    
        }
      writel(regValue,IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);

      /* write ADC start bit */
      tmp = readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);
      writel(tmp|VH_ADC_CON_ADC_START_STATUS_MSK,IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);
    }
  else
    {
      error_status = PH_ADC_ERR_WRONG_ADC_MODE;  
    }
  return(error_status);
}

/**
 * @brief The ADC phhwAdc_SingleConversionStart function. (HwApi layer)
 *
 * This function is used to start and setup the conversion for single mode.
 * The application programmer can select the input reference, resolution and
 * which analog input signal should be taken for the measurement.
 * 
 * - The `ADC_START` bit is set.
 *
 * Function is not re-entrant.
 *
 * @see                   phhwAdcVref_en_t
 * @see                   phhwAdcResolution_en_t
 * @see                   phhwAdcChannel_en_t
 * @param                 vref : voltage reference selection
 * @param                 resol : resolution selection
 * @param                 channel : channel to read from
 * @returns               Returns an error status code
 * @retval                PH_OK : in case of success.
 * @retval                PH_ADC_ERR_NOT_INITIALIZED : no initialisation occured
 * @retval                PH_ADC_ERR_WRONG_ADC_MODE : wrong ADC mode
 * @retval                PH_ADC_ERR_BAD_PARAMETER : bad parameter
 */
phErrorCode_t phhwAdc_SingleConversionStart(phAdcVref_en_t       vref,
                                            phAdcResolution_en_t resol,
                                            phAdcChannel_en_t    channel)
{
  UInt32         regValue;
  phErrorCode_t  error_status;
   unsigned int tmp;

  /* Are we initialised */
  if(sAdcIsAdcInitialised == FALSE)
    return(PH_ADC_ERR_NOT_INITIALIZED); 

  /* Are we initialised for `Single mode conversion) */
  if(readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG) & VH_ADC_CON_ADC_CSCAN_STATUS_MSK)
    return(PH_ADC_ERR_WRONG_ADC_MODE);

  error_status = PH_OK;

  /* select reference voltage input */
  regValue = readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);

  /* Check if we are in `continuous mode` */
  if(!(readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG) & VH_ADC_CON_ADC_CSCAN_STATUS_MSK))		   
    {
      switch(vref)
        {
        case PH_ADCVREF_0 :
	  regValue &= ~(VH_ADC_CON_ADC_SELREF_STATUS_MSK);
	  break;
        case PH_ADCVREF_1 :
	  regValue |= VH_ADC_CON_ADC_SELREF_STATUS_MSK;
	  break;        
        default:
	  {
	    return(PH_ADC_ERR_BAD_PARAMETER);
	  }
	  break;    
        }
      writel(regValue,IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);

      /* Write Select Channel Resolution register */
      switch(channel)
        {
        case PH_ADCCHANNEL_0 :
	  regValue = 
	    (UInt32)(((UInt32)resol)     & VH_ADC_CON_ADC_CSEL0_STATUS_MSK);
	  break;
        case PH_ADCCHANNEL_1 :
	  regValue = 
	    (UInt32)(((UInt32)resol<<4)  & VH_ADC_CON_ADC_CSEL1_STATUS_MSK);
	  break;
        case PH_ADCCHANNEL_2 :
	  regValue =
	    (UInt32)(((UInt32)resol<<8)  & VH_ADC_CON_ADC_CSEL2_STATUS_MSK);
	  break;
        case PH_ADCCHANNEL_3 :
	  regValue =
	    (UInt32)(((UInt32)resol<<12) & VH_ADC_CON_ADC_CSEL3_STATUS_MSK);
	  break;
        case PH_ADCCHANNEL_4 :
	  regValue =
	    (UInt32)(((UInt32)resol<<16) & VH_ADC_CON_ADC_CSEL4_STATUS_MSK);
	  break;
        case PH_ADCCHANNEL_5 :
	  regValue =
	    (UInt32)(((UInt32)resol<<20) & VH_ADC_CON_ADC_CSEL5_STATUS_MSK);
	  break;
        case PH_ADCCHANNEL_6 :
	  regValue =
	    (UInt32)(((UInt32)resol<<24) & VH_ADC_CON_ADC_CSEL6_STATUS_MSK);
	  break;
        case PH_ADCCHANNEL_7 :
	  regValue =
	    (UInt32)(((UInt32)resol<<28) & VH_ADC_CON_ADC_CSEL7_STATUS_MSK);
	  break;
        default:
	  {
	    return(PH_ADC_ERR_BAD_PARAMETER);
	  }
	  break;    
        }
      writel(regValue,IO_ADDRESS(ADC_BASE)+VH_ADC_CSEL_RES_REG);

      sAdcEnabledChanel[channel] = TRUE;

      /* write ADC start bit */
      tmp = readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);
      writel(tmp | VH_ADC_CON_ADC_START_STATUS_MSK,IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);
    }
  return(error_status);
}

/**
 * @brief The ADC phhwAdc_Read function. (HwApi layer)
 *
 * This function is used to read a value from the ADC. 
 * This function is a `SYNCHRONOUS` function.
 *
 * REMARK1: Concurrancy problems are solved in a higher level ! 
 *
 * Function is not re-entrant.
 *
 * @see     phhwAdcChannel_en_t
 * @param   channel : channel to read from
 * @param   pResult : pointer to the result
 * @return  Returns an error status code
 * @retval  PH_OK : in case of success
 * @retval  PH_ADC_ERR_NOT_INITIALIZED : no initialisation occured
 * @retval  PH_ADC_ERR_CH_NOT_ENABLED : channel is not selected for conversion
 * @retval  PH_ADC_ERR_BAD_PARAMETER : bad parameter
 */
phErrorCode_t phhwAdc_Read(phAdcChannel_en_t channel,
			   UInt32*              pResult)

{
  phErrorCode_t  error_status;
  unsigned int tmp;

  /* Are we initialised */
  if(sAdcIsAdcInitialised == FALSE)
    {
      return(PH_ADC_ERR_NOT_INITIALIZED);  
    }

  error_status = PH_OK;

  /*In `Continuous scan` mode: read the value immediately */
  /*In `Single     scan` mode: wait untill conversion cycle has ended and read*/

    if(!(readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG) & VH_ADC_CON_ADC_CSCAN_STATUS_MSK))
      {
	/* Single scan mode */
	
      /* wait untill conversion process is finished */
	while(readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG) & VH_ADC_CON_ADC_STATUS_STATUS_MSK){
      }
	
      /* write ADC stop command */
	tmp = readl(IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);
	writel(tmp&~(VH_ADC_CON_ADC_START_STATUS_MSK),IO_ADDRESS(ADC_BASE)+VH_ADC_CON_REG);
	
    }
    /* read ADC result register */
  /* REMARK: The result is always returned as a 32-bit number */
    switch(channel)
    {
    case PH_ADCCHANNEL_0:
      {
	/* is the channel enabled */
	if(sAdcEnabledChanel[0] == TRUE)
	  {
	    *pResult=(UInt32)( readl(IO_ADDRESS(ADC_BASE)+VH_ADC_R0_REG) & VH_ADC_R0_STATUS_MSK);
	  }
	else
	  {
	    *pResult     = 0xFFFFFFFF;  
	    error_status = PH_ADC_ERR_CH_NOT_ENABLED;
	  }
      }
      break;
    case PH_ADCCHANNEL_1:
      {
	/* is the channel enabled */
	if(sAdcEnabledChanel[1] == TRUE)
	  {
	    *pResult=(UInt32)( readl(IO_ADDRESS(ADC_BASE)+VH_ADC_R1_REG) & VH_ADC_R1_STATUS_MSK);
	  }
	else
	  {
	    *pResult     = 0xFFFFFFFF;  
	    error_status = PH_ADC_ERR_CH_NOT_ENABLED;
	  }
      }
      break;   
    case PH_ADCCHANNEL_2:
      {
	/* is the channel enabled */
	if(sAdcEnabledChanel[2] == TRUE)
	  {
	    *pResult=(UInt32)( readl(IO_ADDRESS(ADC_BASE)+VH_ADC_R2_REG) & VH_ADC_R2_STATUS_MSK);
	  }
	else
	  {
	    *pResult     = 0xFFFFFFFF;  
	    error_status = PH_ADC_ERR_CH_NOT_ENABLED;
	  }
      }
      break;
    case PH_ADCCHANNEL_3:
      {
	/* is the channel enabled */
	if(sAdcEnabledChanel[3] == TRUE)
	  {
	    *pResult=(UInt32)( readl(IO_ADDRESS(ADC_BASE)+VH_ADC_R3_REG) & VH_ADC_R3_STATUS_MSK);
	  }
	else
	  {
	    *pResult     = 0xFFFFFFFF;  
	    error_status = PH_ADC_ERR_CH_NOT_ENABLED;
	  }
      }
      break;        
    case PH_ADCCHANNEL_4:
      {
	/* is the channel enabled */
	if(sAdcEnabledChanel[4] == TRUE)
	  {
	    *pResult=(UInt32)( readl(IO_ADDRESS(ADC_BASE)+VH_ADC_R4_REG) & VH_ADC_R4_STATUS_MSK);
	  }
	else
	  {
	    *pResult     = 0xFFFFFFFF;  
	    error_status = PH_ADC_ERR_CH_NOT_ENABLED;
	  }
      }
      break;
    case PH_ADCCHANNEL_5:
      {
	/* is the channel enabled */
	if(sAdcEnabledChanel[5] == TRUE)
	  {
	    *pResult=(UInt32)( readl(IO_ADDRESS(ADC_BASE)+VH_ADC_R5_REG) & VH_ADC_R5_STATUS_MSK);
	  }
	else
	  {
	    *pResult     = 0xFFFFFFFF;  
	    error_status = PH_ADC_ERR_CH_NOT_ENABLED;
	  }
      }
      break;        
    case PH_ADCCHANNEL_6:
      {
	/* is the channel enabled */
	if(sAdcEnabledChanel[6] == TRUE)
	  {
	    *pResult=(UInt32)( readl(IO_ADDRESS(ADC_BASE)+VH_ADC_R6_REG) & VH_ADC_R6_STATUS_MSK);
	  }
	else
	  {
	    *pResult     = 0xFFFFFFFF;  
	    error_status = PH_ADC_ERR_CH_NOT_ENABLED;
	  }
      }
      break;    
    case PH_ADCCHANNEL_7:
      {
	/* is the channel enabled */
	if(sAdcEnabledChanel[7] == TRUE)
	  {
	    *pResult=(UInt32)( readl(IO_ADDRESS(ADC_BASE)+VH_ADC_R7_REG) & VH_ADC_R7_STATUS_MSK);
	  }
	else
	  {
	    *pResult     = 0xFFFFFFFF;  
	    error_status = PH_ADC_ERR_CH_NOT_ENABLED;
	  }
      }
      break;    
    default:
      {
	return(PH_ADC_ERR_BAD_PARAMETER);
      }
      break;    
    }
  return(error_status);
}

static inline unsigned long CGU_Read(unsigned int Register)
{
	return readl(IO_ADDRESS(CGU_TOTAL_BASE) + Register);
}

static inline void CGU_Write(unsigned int Register, unsigned long Value)
{
	writel(Value, IO_ADDRESS(CGU_TOTAL_BASE) + Register);
}
