/*
 * filesyste/bootsect.h
 *
 */

#if !defined(_FILESYSTEM_FAT_H_)
#define	_FILESYSTEM_FAT_H_

#include <linux/config.h>
#include <asm/types.h>

/*
 * ERROR CODE
 */

#define	OK				0
#define INVALID_SIGNATURE		-(0x0101)
#define INVALID_BPB_PARAM		-(0x0102)
#define NOTEXIST_FILE			-(0x0103)
#define INVALID_FILE_STRUCT		-(0x0104)
#define INVALID_FILE_SIZE		-(0x0105)
#define INVALID_KERNEL_SIZE		-(0x0106)
#define INVALID_MAGICNUMBER		-(0x0107)
#define NOTEXIST_ACTIVEPARTITION	-(0x0108)


#pragma pack(1)


#define	MAX_PARTITION_ENTRY		4
#define SECTOR_SIZE				512

#define	DIRENT_IN_SECTOR	(SECTOR_SIZE / sizeof(Direntry))


/*
 * Format of a Partition Table Entry
 */
typedef struct _PartitionEntry {
	u8		ActiveFlag;			/* Active Flag */
#define	PARTITION_ACTIVE	0x80
	u8		StartSecs[3];		/* Start Sector in CHS */
	u8		PartitionType;		/* Partition Type */
#define	TYPE_FAT12			0x01
#define	TYPE_FAT16_SMALL	0x04
#define	TYPE_EXT_DOS		0x05
#define	TYPE_FAT16_LARGE	0x06
#define	TYPE_FAT16_LBA		0x0E
#define	TYPE_EXT_LBA		0x0F
	u8		EndSecs[3];			/* End Sector in CHS */
	u32		LBAStartSector;		/* Start Sector in LBA */
	u32		PartitionSize;		/* Partition Size */
} PartitionEntry;


/*
 * Format of a BIOS Parameter Block for DOS 3.3
 */
typedef struct _Bpb33 {
	u16		BytesPerSec;		/* bytes per sector */
	u8		SecPerClust;		/* sectors per cluster */
	u16		ResSectors;			/* number of reserved sectors */
	u8		FATs;				/* number of FATs */
	u16		RootDirEnts;		/* number of root directory entries */
	u16		Sectors;			/* total number of sectors */
	u8		Media;				/* media descriptor */
	u16		FATsecs;			/* number of sectors per FAT */
	u16		SecPerTrack;		/* sectors per track */
	u16		Heads;				/* number of heads */
	u16		HiddenSecs;			/* number of hidden sectors */
} Bpb33;


/*
 * Format of a BIOS Parameter Block for DOS 5.0
 */
typedef struct _Bpb50 {
	u16		BytesPerSec;		/* bytes per sector */
	u8		SecPerClust;		/* sectors per cluster */
	u16		ResSectors;			/* number of reserved sectors */
	u8		FATs;				/* number of FATs */
	u16		RootDirEnts;		/* number of root directory entries */
	u16		Sectors;			/* total number of sectors */
	u8		Media;				/* media descriptor */
	u16		FATsecs;			/* number of sectors per FAT */
	u16		SecPerTrack;		/* sectors per track */
	u16		Heads;				/* number of heads */
	u32		HiddenSecs;			/* number of hidden sectors */
	u32		HugeSectors;		/* number of sectors if Sectors == 0 */
} Bpb50;


/*
 * Format of a boot sector.
 */
typedef struct _Bootsector33 {
	u8		Jump[3];			/* jump inst E9xxxx or EBxx90 */
	s8		OemName[8];			/* OEM name and version */
	s8		BPB[19];			/* BIOS parameter block */
	s8		DriveNumber;		/* drive number (0x80) */
	s8		BootCode[479];		/* pad so struct is 512b */
	u8		BootSectSig0;
	u8		BootSectSig1;
#define	BOOTSIG0	0x55
#define	BOOTSIG1	0xaa
} Bootsector33;

typedef struct _Extboot {
	s8		DriveNumber;		/* drive number (0x80) */
	s8		Reserved1;			/* reserved */
	s8		BootSignature;		/* ext. boot signature (0x29) */
#define	EXBOOTSIG	0x29
	s8		VolumeID[4];		/* volume ID number */
	s8		VolumeLabel[11];	/* volume label */
	s8		FileSysType[8];		/* fs type (FAT12 or FAT16) */
} Extboot;

typedef struct _Bootsector50 {
	u8		Jump[3];			/* jump inst E9xxxx or EBxx90 */
	s8		OemName[8];			/* OEM name and version */
	s8		BPB[25];			/* BIOS parameter block */
	s8		Ext[26];			/* Bootsector Extension */
	s8		BootCode[448];		/* pad so structure is 512b */
	u8		BootSectSig0;
	u8		BootSectSig1;
#define	BOOTSIG0	0x55
#define	BOOTSIG1	0xaa
} Bootsector50;



/*
 * Format of a dos directory entry.
 */
typedef struct _Direntry {
	u8		Name[8];			/* filename, blank filled */
#define	SLOT_EMPTY		0x00	/* slot has never been used */
#define	SLOT_E5			0x05	/* the real value is 0xe5 */
#define	SLOT_DELETED	0xe5	/* file in this slot deleted */
	u8		Extension[3];		/* extension, blank filled */
	u8		Attributes;			/* file attributes */
#define	ATTR_NORMAL	0x00		/* normal file */
#define	ATTR_READONLY	0x01	/* file is readonly */
#define	ATTR_HIDDEN		0x02	/* file is hidden */
#define	ATTR_SYSTEM		0x04	/* file is a system file */
#define	ATTR_VOLUME		0x08	/* entry is a volume label */
#define	ATTR_DIRECTORY	0x10	/* entry is a directory name */
#define	ATTR_ARCHIVE	0x20	/* file is new or modified */
#define	ATTR_WIN95		0x0f
	u8		Reserved;			/* reserved */
	u8		CHundredth;			/* hundredth of seconds in CTime */
	u16		CTime;				/* create time */
	u16		CDate;				/* create date */
	u16		ADate;				/* access date */
	u16		HighClust;			/* high bytes of cluster number */
	u16		MTime;				/* last update time */
	u16		MDate;				/* last update date */
	u16		StartCluster;		/* starting cluster of file */
	u32		FileSize;			/* size of file in bytes */
} Direntry;


#define	TIME2SECOND_MASK	0x001F
#define	TIME2SECOND_SHIFT	0
#define	TIME2MINUTE_MASK	0x07E0
#define	TIME2MINUTE_SHIFT	5
#define	TIME2HOUR_MASK		0xF800
#define	TIME2HOUR_SHIFT		11

#define	DATE2DAY_MASK		0x001F
#define	DATE2DAY_SHIFT		0
#define	DATE2MONTH_MASK		0x01E0
#define	DATE2MONTH_SHIFT	5
#define	DATE2YEAR_MASK		0xFE00
#define	DATE2YEAR_SHIFT		9

#define	TIME2SECOND(tm)		((((u16)(tm) & TIME2SECOND_MASK) >> TIME2SECOND_SHIFT) * 2)
#define	TIME2MINUTE(tm)		(((u16)(tm) & TIME2MINUTE_MASK) >> TIME2MINUTE_SHIFT)
#define	TIME2HOUR(tm)		(((u16)(tm) & TIME2HOUR_MASK) >> TIME2HOUR_SHIFT)

#define	DATE2DAY(dt)		(((u16)(dt) & DATE2DAY_MASK) >> DATE2DAY_SHIFT)
#define	DATE2MONTH(dt)		(((u16)(dt) & DATE2MONTH_MASK) >> DATE2MONTH_SHIFT)
#define	DATE2YEAR(dt)		((((u16)(dt) & DATE2YEAR_MASK) >> DATE2YEAR_SHIFT) + 1980)


/*
 * Format of a FAT filesystem.
 */
typedef struct _Fat_fs {
	u8		PartitionType;	/* directory entry */
	u32		LBAStartSec;		/* start sector in LBA */
	u16		ResSecs;			/* number of reserved sectors */
	u16		BytesPerSec;		/* number of bytes in Sector */
	u16		SecsPerFAT;			/* number of sectors in one FAT */
	u32		RootDirEntSec;		/* start sector of RootDirectoryEntry */
	u32		RootDirEntSize;		/* RootDirectoryEntry size (in Sectors) */
	u32		SecsPerCluster;		/* number of sectors in Cluster */
	u32		FirstCluster;		/* */
} Fat_fs;


/*
 * Some useful cluster numbers.
 */
#define	CLUSTER_FIRST	2		/* first legal cluster number */
#define	CLUSTER_RSRVD	0xfffffff6	/* reserved cluster range */
#define	CLUSTER_BAD		0xfffffff7	/* a cluster with a defect */
#define	CLUSTER_EOFS	0xfffffff8	/* start of eof cluster range */
#define	CLUSTER_EOFE	0xffffffff	/* end of eof cluster range */
#define	CLUSTER_END		CLUSTER_EOFE	/* bigger than any valid cluster */

#define	FAT12_MASK		0x00000fff	/* mask for 12 bit cluster numbers */
#define	FAT16_MASK		0x0000ffff	/* mask for 16 bit cluster numbers */
#define	FAT32_MASK		0x0fffffff	/* mask for FAT32 cluster numbers */

#define	MAX_FAT12_CLUSTER	((CLUSTER_RSRVD - CLUSTER_FIRST) & FAT12_MASK)

#define	IS_EOF(clst, eof)		((clst & eof) == eof)
#define	FAT12_EOF(cluster)		IS_EOF(cluster, (CLUSTER_EOFS & FAT12_MASK))
#define	FAT16_EOF(cluster)		IS_EOF(cluster, (CLUSTER_EOFS & FAT16_MASK))

#define	CLUSTOR2SECTOR(fs, clst)	(((Fat_fs *)fs)->FirstCluster + ((clst - CLUSTER_FIRST) * ((Fat_fs*)fs)->SecsPerCluster))


/*
 * Format of a FILE.
 */
typedef struct _File {
	u8		bInit;				/* Initialized flag */
#define	FILE_NOTINIT	0
#define	FILE_INITED		1
	Direntry	dir;	/* directory entry */
	Fat_fs		fs;		/* FAT filesystem */
} File;



/*
 * Definition of functions
 */
int init_fat_fs(void);
int open_file(File *fp, u8 *filename);
int read_file(File *fp, u8 *buf);


#endif	/* !defined(_FILESYSTEM_FAT_H_) */
