/*
 * filesyste/fat.c
 *
 * history
 * 2003/05/07  A.I.   1.01    Disabled active partition check.
 *
 */

#include "config.h"

#include <asm/page.h>
#include <asm/byteorder.h>

#include "nonstdio.h"
#include "fat.h"
#include "util.h"

#include "../sd/sdmem.h"


/**********************************************************
 * Grobal variables
 */

#pragma align 4
static u8		g_mbr[SECTOR_SIZE];
static u8		g_secwork[SECTOR_SIZE];
static Fat_fs	g_fs;


/**********************************************************
 * External variables
 */

extern unsigned int kernel_entry;
extern unsigned char *kernel_base;
extern unsigned int kernel_size;

/**********************************************************
 * read Master Boot Record
 *
 * Params
 *       mbr : buffer to store the mbr data
 *
 * Return
 *       0:SUCCESS
 *       other:FAILD
 */
int read_mbr(u8 *mbr)
{
	s32 ret;

	/* read MBR from SD card */
	ret = SD1_ReadSector(0, 1, (u16 *)mbr);
	if (ret < 0) {
		DBG_PRINT("Error SD1_ReadSector: %d\n", ret);
		return ret;
	}

	/* check MBR signature */
	if ((mbr[510] != BOOTSIG0) || (mbr[511] != BOOTSIG1)) {
		DBG_PRINT("error: BOOTSIG0[%x] BOOTSIG1[%x]\n", mbr[510], mbr[511]);
		return INVALID_SIGNATURE;
	}

	return OK;
}



/**********************************************************
 * Get active partition
 *
 * Params
 *       mbr : the mbr data buffer
 *
 * Return
 *       address:SUCCESS
 *       NULL:FAILD
 */
PartitionEntry* get_active_partition(const char *mbr)
{
	int i;
	PartitionEntry *pe;

	for (i = 0; i < MAX_PARTITION_ENTRY; i++) {
		/* get partition table entry */
		pe = (PartitionEntry *)&mbr[0x1BE + i * 16];

		/* DEBUG */
		dump_PartitionEntry(pe);

		/* check partition type */
		switch (pe->PartitionType) {
		case TYPE_FAT12:
		case TYPE_FAT16_SMALL:
		case TYPE_FAT16_LARGE:
		case TYPE_FAT16_LBA:
#if 1	/* rev 1.01 */
			/* detect active partition */
			return pe;
#else
			/* check active flag */
			if (pe->ActiveFlag == PARTITION_ACTIVE) {
				/* detect active partition */
				return pe;
			}
#endif	/* rev 1.01 */
		}
	}
	/* Not detected active partition */
	return NULL;
}



/**********************************************************
 * Get BIOS Parameter Block
 *
 * Params
 *       pe : partition table entry buffer
 *       pbp : buffer to store BIOS Parameter Block
 *
 * Return
 *       0:SUCCESS
 *       other:FAILD
 */
int get_bpb(PartitionEntry *pe, Bpb50 *bpb)
{
	s32 ret;
	u32 sector = le32_to_cpu(pe->LBAStartSector);
	Bootsector50* bs = (Bootsector50*)g_secwork;

	ret = SD1_ReadSector(sector, 1, (u16*)bs);
	if (ret < 0) {
		DBG_PRINT("Error SD1_ReadSector: %d\n", ret);
		return ret;
	}

	/* check Boot Sectore signature */
	if ((bs->BootSectSig0 != BOOTSIG0) || (bs->BootSectSig1 != BOOTSIG1)) {
		DBG_PRINT("error: BOOTSIG0[%x] BOOTSIG1[%x]\n", bs->BootSectSig0, bs->BootSectSig1);
		return INVALID_SIGNATURE;
	}

	/* copy BPB */
	*bpb = *(Bpb50 *)(bs->BPB);

	/* DEBUG */
	dump_Bpb50(bpb);

	/* check BPB parameters */
	if (bpb->RootDirEnts == 0) {
		DBG_PRINT("error: bpb->RootDirEnts is %d\n", bpb->RootDirEnts);
		return INVALID_BPB_PARAM;
	}
	return OK;
}



/**********************************************************
 * Load FAT data to Memory (for cache)
 *
 * Params
 *       fs : parameters of FAT partition
 *       base : buffer to store FAT data
 *
 * Return
 *       0:SUCCESS
 *       other:FAILD
 */
int load_fat(Fat_fs *fs, u8 *base)
{
	s32 ret, size;
	u32 sector;

	/* get FAT size, start sector */
	size = fs->SecsPerFAT * fs->BytesPerSec;
	sector = fs->LBAStartSec + fs->ResSecs;
	ret = SD1_ReadSector(sector, fs->SecsPerFAT, (u16 *)base);
	if (ret < 0) {
		DBG_PRINT("Error SD1_ReadSector: %d\n", ret);
		return ret;
	}

	return OK;
}



/**********************************************************
 * Init FAT Filesystem parameters
 *
 * Params
 *       NONE
 *
 * Return
 *       0:SUCCESS
 *       other:FAILD
 */
int init_fat_fs(void)
{
	int ret;
	s32 size;
	u32 sector, cluster;
	Bpb50 bpb;
	PartitionEntry* pe;

	/* read MBR */
	ret = read_mbr(g_mbr);
	if (ret != OK) {
		return ret;
	}

	/* get active partition */
	pe = get_active_partition(g_mbr);
	if (pe == NULL) {
		return NOTEXIST_ACTIVEPARTITION;
	}
	
	/* read BIOS Parameter Block */
	ret = get_bpb(pe, &bpb);
	if (ret != OK) {
		return ret;
	}

	g_fs.LBAStartSec = le32_to_cpu(pe->LBAStartSector);
	g_fs.ResSecs = le16_to_cpu(bpb.ResSectors);
	g_fs.BytesPerSec = le16_to_cpu(bpb.BytesPerSec);
	g_fs.SecsPerFAT = le16_to_cpu(bpb.FATsecs);
	g_fs.RootDirEntSec = g_fs.LBAStartSec + g_fs.ResSecs + (bpb.FATs * g_fs.SecsPerFAT);
	g_fs.RootDirEntSize = le16_to_cpu(bpb.RootDirEnts) * 32;		/* 32 = sizeof(Dentry) */
	g_fs.SecsPerCluster = bpb.SecPerClust;
	g_fs.FirstCluster = g_fs.RootDirEntSec + (g_fs.RootDirEntSize / g_fs.BytesPerSec);

	/* check Partition type */
	if (bpb.Sectors) {
		cluster = (le16_to_cpu(bpb.Sectors) / bpb.SecPerClust) + 1;
	} else {
		cluster = (le32_to_cpu(bpb.HugeSectors) / bpb.SecPerClust) + 1;
	}
	if (cluster <= MAX_FAT12_CLUSTER) {
		g_fs.PartitionType = TYPE_FAT12;
	} else {
		g_fs.PartitionType = TYPE_FAT16_LBA;	/* = not TYPE_FAT12 */
	}
	/* DEBUG */
	dump_Fat_fs(&g_fs);

	ret = load_fat(&g_fs, (u8 *)AVAIL_FAT_ADDR);

	/* DEBUG */
#if 0
	for (size = 0; size < g_fs.SecsPerFAT; size++) {
		dump_Sector((u8*)(AVAIL_FAT_ADDR + SECTOR_SIZE * size));
	}
//	dump_FAT(&g_fs, (u8*)AVAIL_FAT_ADDR);
	DBG_PRINT("End of DUMP ---------\n");
#endif
		
	if (ret != OK) {
		return ret;
	}

	return OK;
}



/**********************************************************
 * get next cluster number
 *
 * Params
 *       ptype : partition type (FAT12 or FAT16 only)
 *       fat : FAT area
 *       cluster : current cluster number / next cluster number
 *
 * Return
 *       0:next cluster isnot EOF
 *       EOF:next cluster is EOF
 */
int get_next_cluster(u8 ptype, const u8 *fat, u16 *cluster)
{
	u32 idx;

	/* DEBUG */
//	DBG_PRINT("cluster:%d(0x%X) -> ", *cluster, *cluster);
	if (ptype == TYPE_FAT12) {
		idx = (*cluster) * 3 / 2;
		if (*cluster & 1) {
			*cluster = ((fat[idx] >> 4) | ((u16)(fat[idx + 1]) << 4)) & FAT12_MASK;
		} else {
			*cluster = ((fat[idx] | ((u16)(fat[idx + 1]) << 8)) & FAT12_MASK);
		}
		if (FAT12_EOF(*cluster)) {
			/* DEBUG */
			DBG_PRINT("EOF(0x%x)\n", *cluster);
			return EOF;
		}
	} else {	/* FAT16 */
		idx = (*cluster) * 2;
		*cluster = le16_to_cpu(*(u16 *)&fat[idx]);
		if (FAT16_EOF(*cluster)) {
			/* DEBUG */
			DBG_PRINT("EOF(0x%x)\n", *cluster);
			return EOF;
		}
	}
	/* DEBUG */
//	DBG_PRINT("%d(0x%x)\n", *cluster, *cluster);
	return 0;
}



/**********************************************************
 * open the specified file
 *
 * Params
 *       fp : parameters of file I/O
 *       filename : name to open file
 *
 * Return
 *       0:SUCCESS
 *       other:FAILD
 */
int open_file(File *fp, u8 *filename)
{
	s32 ret, i, size;
	u32 sector;
#if 0
	char name[8] = "        ";
	char ext[3]  = "   ";
#else
	char name[8];
	char ext[3];
#endif
	Direntry* dir;

	fp->bInit = FILE_NOTINIT;
	/* split filename to basename and extension */
	split_filename(filename, name, ext);

	/* set size, start sector of RootDirectoryEntry */
	size = g_fs.RootDirEntSize * g_fs.BytesPerSec;
	sector = g_fs.RootDirEntSec;

	while (size > 0) {
		ret = SD1_ReadSector(sector, 1, (u16 *)g_secwork);
		if (ret < 0) {
			DBG_PRINT("Error SD1_ReadSector: %d\n", ret);
			return ret;
		}

		for (i = 0; i < DIRENT_IN_SECTOR; i++) {
			dir = (Direntry *)&g_secwork[i * sizeof(Direntry)];

			/* DEBUG */
			dump_Direntry(dir);
			/* check attribute */
			if (dir->Attributes == ATTR_WIN95) {	/* not support long file name */
				continue;
			}
			if (dir->Attributes & ATTR_VOLUME) {	/* ignore the volume label */
				continue;
			}
			if (dir->Attributes & ATTR_DIRECTORY) {	/* not support in this version */
				continue;
			}
			/* check filename */
			if (dir->Name[0] == 0) {	/* no more direntry */
				size = SECTOR_SIZE;
				/* DEBUG */
				DBG_PRINT("\n NO MORE DIRENTRY!!!\n");
				return NOTEXIST_FILE;
			}
			if (dir->Name[0] == SLOT_DELETED) {	/* erased direntry, skip to next */
				/* DEBUG */
				DBG_PRINT("\n SKIP IT!!!\n");
				continue;
			}
			if (memcmp(dir->Name, name, BASENAME_LEN)
				|| memcmp(dir->Extension, ext, EXTNAME_LEN)) {
				continue;
			}
			/* find out the file!!! */
			goto FINDOUT_FILE;
		}
		size -= SECTOR_SIZE;
		sector++;
	}

	/* can't find out the file */
	DBG_PRINT("Not exist such file : %s\n", filename);
	return NOTEXIST_FILE;

FINDOUT_FILE:

	/* store the DirectoryEntry to File */
	fp->fs  = g_fs;
	fp->dir = *dir;
#if 1	/* LittleEndian -> BigEndian */
	fp->dir.CTime = le16_to_cpu(dir->CTime);
	fp->dir.CDate = le16_to_cpu(dir->CDate);
	fp->dir.ADate = le16_to_cpu(dir->ADate);
	fp->dir.HighClust = le16_to_cpu(dir->HighClust);
	fp->dir.MTime = le16_to_cpu(dir->MTime);
	fp->dir.MDate = le16_to_cpu(dir->MDate);
	fp->dir.StartCluster = le16_to_cpu(dir->StartCluster);
	fp->dir.FileSize = le32_to_cpu(dir->FileSize);
#endif
	fp->bInit = FILE_INITED;

	return OK;
}



/**********************************************************
 * read the specified file
 *
 * Params
 *       fp : parameters of file I/O
 *       buf : buffer to store the read data
 *
 * Return
 *       0:SUCCESS
 *       other:FAILD
 */
int read_file(File *fp, u8 *buf)
{
	s32 ret, size;
	u32 sector;
	u16 cluster;

	/* check File Structure */
	if (fp->bInit == FILE_NOTINIT) {
		return INVALID_FILE_STRUCT;
	}

	cluster = fp->dir.StartCluster;
	size = fp->dir.FileSize;
	/* convert cluster to sector */
	sector = CLUSTOR2SECTOR(&(fp->fs), cluster);

	/* read 1st sector */
	ret = SD1_ReadSector(sector++, 1, (u16 *)g_secwork);
	if (ret < 0) {
		DBG_PRINT("Error SD1_ReadSector: %d\n", ret);
		return ret;
	}

	/* get kernel image paramters */
	kernel_size = le32_to_cpu(*((u32 *)g_secwork));
	kernel_base = (u8 *)le32_to_cpu(*(((u32 *)&g_secwork[4])));
	kernel_entry = le32_to_cpu(*(((u32 *)&g_secwork[8])));

	/* DEBUG */
	DBG_PRINT("kernel_size  : 0x%x\n", kernel_size);
	DBG_PRINT("kernel_base  : 0x%x\n", kernel_base);
	DBG_PRINT("kernel_entry : 0x%x\n", kernel_entry);

	/* check kernel size
	 * 16 = sizeof(kernel_size)
	 *    + sizeof(kernel_base)
	 *    + sizeof(kernel_entry)
	 *    + sizeof(magic_number)
	 */
	if ((kernel_size + 16) != size) {
		PRINTF("kernel size is unmatched : kernel_size(0x%x) :file_size(0x%x)\n",
			kernel_size + 16, size);
			return INVALID_KERNEL_SIZE;
	}
	size -= SECTOR_SIZE;
	if (size < 0) {
		PRINTF("kernel_size is less than %d : 0x%x\n", SECTOR_SIZE, (size + SECTOR_SIZE));
		return INVALID_FILE_SIZE;
	}

	/*
	 * 12 = sizeof(kernel_size)
	 *    + sizeof(kernel_base)
	 *    + sizeof(kernel_entry)
	 */
	/* copy file data to work ram area */
	memcpy(buf, &g_secwork[12], (SECTOR_SIZE - 12));
	buf += (SECTOR_SIZE - 12);

	/* load rest sector in cluster */
	ret = SD1_ReadSector(sector, (fp->fs.SecsPerCluster - 1), (u16 *)buf);
	if (ret < 0) {
		DBG_PRINT("Error SD1_ReadSector: %d\n", ret);
		return ret;
	}
	if (size <= (SECTOR_SIZE * (fp->fs.SecsPerCluster - 1))) {
		/* end of file */
		buf += size;
	} else {
		buf += (SECTOR_SIZE * (fp->fs.SecsPerCluster - 1));
	}
	size -= (SECTOR_SIZE * (fp->fs.SecsPerCluster - 1));

	while (size > 0) {
		/* get next cluster */
		if (get_next_cluster(fp->fs.PartitionType, (u8 *)AVAIL_FAT_ADDR, &cluster) == EOF) {
			/* file is EOF */
			break;
		}
		/* get 1st sector on cluster */
		sector = CLUSTOR2SECTOR(&(fp->fs), cluster);

		/* DEBUG */
//		DBG_PRINT("SD1_ReadSector(%d, %d, 0x%x)", sector, fp->fs.SecsPerCluster, (u32)buf);
		ret = SD1_ReadSector(sector, fp->fs.SecsPerCluster, (u16 *)buf);
		/* DEBUG */
//		DBG_PRINT("end (size:%d)\n", size);
		if (ret < 0) {
			DBG_PRINT("Error SD1_ReadSector: %d\n", ret);
			return ret;
		}
		if (size <= (SECTOR_SIZE * fp->fs.SecsPerCluster)) {
			/* end of file */
			buf += size;
		} else {
			buf += (SECTOR_SIZE * fp->fs.SecsPerCluster);
		}
		size -= (SECTOR_SIZE * fp->fs.SecsPerCluster);
	}

	/* check magic number */
#if 0
	buf -= 4;
	if (le32_to_cpu(*(u32 *)buf) != 0x123455AA) {
		PRINTF("Error MagicNumber : 0x123455AA != 0x%x\n", le32_to_cpu(*(u32 *)buf));
		return INVALID_MAGICNUMBER;
	}
#else
	if (*(--buf) != 0x12) {
		PRINTF("Error MagicNumber : 0x12 != 0x%x\n", *buf);
		return INVALID_MAGICNUMBER;
	}
	if (*(--buf) != 0x34) {
		PRINTF("Error MagicNumber : 0x34 != 0x%x\n", *buf);
		return INVALID_MAGICNUMBER;
	}
	if (*(--buf) != 0x55) {
		PRINTF("Error MagicNumber : 0x55 != 0x%x\n", *buf);
		return INVALID_MAGICNUMBER;
	}
	if (*(--buf) != 0xAA) {
		PRINTF("Error MagicNumber : 0xAA != 0x%x\n", *buf);
		return INVALID_MAGICNUMBER;
	}
#endif
	return OK;
}


