/*
 * filesyste/util.c
 *
 */

#include "config.h"

#if defined(SDBOOT_DEBUG)

#include <asm/page.h>
#include <asm/byteorder.h>

#include "fat.h"
#include "util.h"


/**********************************************************
 * Grobal variables
 */

char* g_parttype[] =
	{"UNKNOWN", "FAT12", "XENIX", "XENIX2","FAT16(<= 32MB)", "Ex Partition",
	 "FAT16(>32MB)", "HPFS(OS/2),NTFS", "AIX, CommodoreDOS", "AIX (boot)",
	 "OS/2 (boot)", "FAT32(CHS)", "FAT32(LBA)", "FAT16(LBA)","Ext(LBA)"};



/**********************************************************
 * Dump PartitionEntry
 *
 * Params
 *       pe : structure of PartitionEntry
 *
 * Return
 *       NONE
 *
 */
void dump_PartitionEntry(PartitionEntry *pe)
{
	u8	header, sector;
	u16	cylinder;
	u8	type;

	type = (pe->PartitionType > TYPE_EXT_LBA) ? 0 : pe->PartitionType;

	_printk("\n-- DUMP PartitionEntry --\n");
	_printk("Active Flag      : 0x%x\n", pe->ActiveFlag);

	SPLIT_CHS(pe->StartSecs, cylinder, header, sector);
	_printk("Start Cylinder   : 0x%x\n", cylinder);
	_printk("      Header     : 0x%x\n", header);
	_printk("      Sector     : 0x%x\n", sector);
	_printk("PartitionType    : 0x%x[%s]\n", pe->PartitionType, g_parttype[type]);

	SPLIT_CHS(pe->EndSecs, cylinder, header, sector);
	_printk("End   Cylinder   : 0x%x\n", cylinder);
	_printk("      Header     : 0x%x\n", header);
	_printk("      Sector     : 0x%x\n", sector);
	_printk("LBA Start Sector : 0x%x\n", le32_to_cpu((pe->LBAStartSector)));
	_printk("Partition Size   : 0x%x\n", le32_to_cpu((pe->PartitionSize)));
}



/**********************************************************
 * Dump Bpb50
 *
 * Params
 *       bpb : structure of Bpb
 *
 * Return
 *       NONE
 *
 */
void dump_Bpb50(Bpb50 *bpb)
{

	_printk("\n-- DUMP BIOS Parameter Block --\n");
	_printk("Bytes / Sector        : 0x%x\n", le16_to_cpu(bpb->BytesPerSec));
	_printk("Sectors / Cluster     : 0x%x\n", bpb->SecPerClust);
	_printk("Reserved Sector       : 0x%x\n", le16_to_cpu(bpb->ResSectors));
	_printk("number of FAT         : 0x%x\n", bpb->FATs);
	_printk("RootDirectory Entorys : 0x%x\n", le16_to_cpu(bpb->RootDirEnts));
	_printk("Total Sectors         : 0x%x\n", le16_to_cpu(bpb->Sectors));
	_printk("Media                 : 0x%x\n", bpb->Media);
	_printk("Sectors / FAT         : 0x%x\n", le16_to_cpu(bpb->FATsecs));
	_printk("Sectors / Track       : 0x%x\n", le16_to_cpu(bpb->SecPerTrack));
	_printk("Heads                 : 0x%x\n", le16_to_cpu(bpb->Heads));
	_printk("Hidden Sectors        : 0x%x\n", le16_to_cpu(bpb->HiddenSecs));
	if (bpb->Sectors == 0) {
		_printk("Huge Sectors          : 0x%x\n", le32_to_cpu(bpb->HugeSectors));
	}
}



/**********************************************************
 * Dump Direntry
 *
 * Params
 *       dir : structure of Direntry
 *
 * Return
 *       NONE
 *
 */
void dump_Direntry(Direntry *dir)
{
	int		i, j;
	char	work[20];	/* BASENAME_LEN + EXTNAME_LEN + alpha */

	j = 0;
	for (i = 0; i < BASENAME_LEN; i++) {
		if (dir->Name[i] == ' ') {
			break;
		}
		if ((i == 0) && dir->Name[i] == SLOT_DELETED) {	/* erased direntry */
			work[j++] = '*';
		} else {
			work[j++] = dir->Name[i];
		}
	}
	work[j++] = '.';
	for (i = 0; i < EXTNAME_LEN; i++) {
		if (dir->Extension[i] == ' ') {
			break;
		}
		work[j++] = dir->Extension[i];
	}
	work[j] = 0;

	_printk("\n-- DUMP Direntry --\n");
#if 0
	_printk("Name                  : \"%s\":(%x %x %x %x...)\n", work, dir->Name[0], dir->Name[1], dir->Name[2], dir->Name[3]);
#else
	_printk("Name                  : \"%s\"\n", work);
#endif
	_printk("Attributes            : 0x%x\n", dir->Attributes);
	_printk("hundredth in seconds  : 0x%x\n", dir->CHundredth);
	_printk("Create Time           : %d:%d:%d\n", TIME2HOUR(le16_to_cpu(dir->CTime)),
													TIME2MINUTE(le16_to_cpu(dir->CTime)),
													TIME2SECOND(le16_to_cpu(dir->CTime)));
	_printk("Create date           : %d/%d/%d\n", DATE2YEAR(le16_to_cpu(dir->CDate)),
													DATE2MONTH(le16_to_cpu(dir->CDate)),
													DATE2DAY(le16_to_cpu(dir->CDate)));
	_printk("Access date           : %d/%d/%d\n", DATE2YEAR(le16_to_cpu(dir->ADate)),
													DATE2MONTH(le16_to_cpu(dir->ADate)),
													DATE2DAY(le16_to_cpu(dir->ADate)));
	_printk("Cluster number        : 0x%x\n", le16_to_cpu(dir->HighClust));
	_printk("Update Time           : %d:%d:%d\n", TIME2HOUR(le16_to_cpu(dir->MTime)),
													TIME2MINUTE(le16_to_cpu(dir->MTime)),
													TIME2SECOND(le16_to_cpu(dir->MTime)));
	_printk("Update date           : %d/%d/%d\n", DATE2YEAR(le16_to_cpu(dir->MDate)),
													DATE2MONTH(le16_to_cpu(dir->MDate)),
													DATE2DAY(le16_to_cpu(dir->MDate)));
	_printk("Start Cluster         : 0x%x\n", le16_to_cpu(dir->StartCluster));
	_printk("File Size             : 0x%x(%d)\n", le32_to_cpu(dir->FileSize), le32_to_cpu(dir->FileSize));
}



/**********************************************************
 * Dump Fat_fs
 *
 * Params
 *       fs : structure of Fat_fs
 *
 * Return
 *       NONE
 *
 */
void dump_Fat_fs(Fat_fs *fs)
{
	_printk("\n-- DUMP Fat_fs --\n");
	_printk("PartitionType                         : [%s]\n", ((fs->PartitionType == TYPE_FAT12) ? "FAT12" : "FAT16"));
	_printk("LBA Start Sector                      : 0x%x\n", fs->LBAStartSec);
	_printk("Reserved Sector                       : 0x%x\n", fs->ResSecs);
	_printk("Bytes / Sector                        : 0x%x\n", fs->BytesPerSec);
	_printk("Sectors / FAT                         : 0x%x\n", fs->SecsPerFAT);
	_printk("RootDirectoryEntory Sector            : 0x%x\n", fs->RootDirEntSec);
	_printk("RootDirectoryEntory Size (in Sectors) : 0x%x\n", fs->RootDirEntSize);
	_printk("Sectors / Cluster                     : 0x%x\n", fs->SecsPerCluster);
	_printk("First Cluster                         : 0x%x\n", fs->FirstCluster);
}



/**********************************************************
 * Dump Sector
 *
 * Params
 *       sector : buffer of Sector data
 *
 * Return
 *       NONE
 *
 */
void dump_Sector(const char *sector)
{
	int	i;

	_printk("\n-- DUMP Sector --\n");
	for (i = 0; i < SECTOR_SIZE; i++) {
		if (!(i % 16)) {
			_printk("\n");
		}
		_printk("%x ", (u8)sector[i]);
	}
	_printk("\n");
}

/**********************************************************
 * Dump File Allocation Table (FAT)
 *
 * Params
 *       ptype : partition type (FAT12 or FAT16 only)
 *       fat : FAT area
 *
 * Return
 *       NONE
 */
int dump_FAT(Fat_fs* fs, const char *fat)
{
	int i, fat_size = (fs->BytesPerSec * fs->SecsPerFAT);
	u8 type = fs->PartitionType;
	u16 cluster = 0;

	_printk("\n-- DUMP FAT --\n");
	_printk(" cluster   :  value\n");
	fat_size /= 2;
	fat_size *= 3;
	for (i = 0, cluster = 0; i < fat_size; i++) {
		cluster = i;
		get_next_cluster(type, fat, &cluster);
		_printk(" %d(0x%x)   :  %d(0x%x)\n", i, i, cluster, cluster);
	}
#if 0
	if (ptype == TYPE_FAT12) {
		idx = *cluster * 3 / 2;
		if (*cluster & 1) {
			*cluster = ((fat[idx] >> 4) | ((u16)(fat[idx + 1]) << 4)) & FAT12_MASK;
		} else {
			*cluster = le16_to_cpu(*(u16 *)(&fat[idx])) & FAT12_MASK;
		}
		if (FAT12_EOF(*cluster)) {
			/* DEBUG */
			DBG_PRINT("EOF\n");
			return EOF;
		}
	} else {	/* FAT16 */
		idx = *cluster * 2;
		*cluster = le16_to_cpu(*(u16 *)(&fat[idx]));
		if (FAT16_EOF(*cluster)) {
			/* DEBUG */
			DBG_PRINT("EOF\n");
			return EOF;
		}
	}
	/* DEBUG */
	DBG_PRINT("%d\n", *cluster);
	return 0;
#endif
}

#endif	/* defined(SDBOOT_DEBUG) */

