/*
  * arch/mips/vr7701/common/pci_ops.c
  *
  * PCI support function (fixups and initialization) for NEC VR7701 processor.
  *
  * Author: MontaVista Software, Inc. <source@mvista.com>
  *
  * Based on PCI Template.
  * Copyright 2001, Pete Popov, ppopov@pacbell.net
  *
  * 2003 (c) MontaVista Software, Inc. This file is licensed under
  * the terms of the GNU General Public License version 2. This program
  * is licensed "as is" without any warranty of any kind, whether express
  * or implied.
  */

#include <linux/config.h>

#ifdef CONFIG_PCI

#include <linux/types.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <asm/pci_channel.h>
#include <asm/vr7701.h>

#define PCI_ACCESS_READ 0
#define PCI_ACCESS_WRITE 1

#undef DEBUG

#ifdef DEBUG
#define DBG(x...) printk(x)
#else
#define DBG(x...)
#endif

#define MEM_BASE    0xE0000000
#define MEM_SIZE    0x20000000
#define IO_BASE     0x00000000
#define IO_SIZE     0x00010000
#define IO_ISA_RESERVED  0x1000

#define HOST_MEM_BASE 0x00000000
#define HOST_MEM_SIZE 0x08000000

static struct resource pci_io_resource = {
	"pci IO space",
	IO_BASE + IO_ISA_RESERVED,
	IO_BASE + IO_SIZE - 1,
	IORESOURCE_IO
};

static struct resource pci_mem_resource = {
	"pci memory space",
	MEM_BASE,
	MEM_BASE + MEM_SIZE - 2,
	IORESOURCE_MEM
};

extern struct pci_ops vr7701_pci_ops;

extern phys_t(*fixup_bigphys_addr) (phys_t phys_addr, phys_t size);

/*
 * The mips_pci_channels array has all descriptors for all
 * pci bus controllers. VR7701 has single PCI controller only.
 */
struct pci_channel mips_pci_channels[] = {
	{&vr7701_pci_ops, &pci_io_resource, &pci_mem_resource, 0, 0xff},
	{NULL, NULL, NULL, 0, 0}
};

/* 
 * This is core routine which access to the PCI configuration space using
 * VR7701 PCI unit
 */
static int
vr7701_config_access(unsigned char access_type, struct pci_dev *dev,
		     unsigned char where, u32 * data)
{
	u32 addr;

	if (dev->bus->number == 0) {
		/* 
		 * device located on the same bus as we are, Type 0
		 * configuration cycle should be used 
		 */
		if (PCI_SLOT(dev->devfn) > 20)
			return -1;	/* Cannot access to device */
		addr = (((1UL) << (PCI_SLOT(dev->devfn) + 11))
			| ((PCI_FUNC(dev->devfn) & 0x07) << 8)
			| ((where & 0xFC)));
	} else {
		/* Target is on the different PCI bus, Type 1 configuration
		   cycle used to pass through bridge. */
		addr = (((dev->bus->number & 0xFF) << 16)
			| ((PCI_SLOT(dev->devfn) & 0x1F) << 11)
			| ((PCI_FUNC(dev->devfn) & 0x07) << 8)
			| ((where & 0xFC) | 0x01));
	}

	if (access_type == PCI_ACCESS_READ) {
		unsigned long flags;
		u32 saved_mask;
		save_and_cli(flags);
		saved_mask = vr7701_inl(VR7701_PCI_ERR_MASK);
		vr7701_outl(VR7701_PCI_ERR_MASK, saved_mask |
			    VR7701_PCI_ERR_STATUS_RMA |
			    VR7701_PCI_ERR_STATUS_RCO);
		vr7701_outl(VR7701_PCI_ERR_STATUS,
			    VR7701_PCI_ERR_STATUS_RMA |
			    VR7701_PCI_ERR_STATUS_RCO);
		vr7701_outl(VR7701_PCI_CFGATTR, VR7701_PCI_CFGATTR_READ);
		vr7701_outl(VR7701_PCI_CFGADDR, addr);
		*data = vr7701_inl(VR7701_PCI_CFGDATA);
		if ((vr7701_inl(VR7701_PCI_ERR_STATUS) &
		     (VR7701_PCI_ERR_STATUS_RMA |
		      VR7701_PCI_ERR_STATUS_RCO)) != 0) {
			*data = 0xFFFFFFFF;
		}
		vr7701_outl(VR7701_PCI_ERR_MASK, saved_mask);
		DBG("read  a 0x%08x d 0x%08x\n", addr, *data);
		restore_flags(flags);
	} else if (access_type == PCI_ACCESS_WRITE) {
		vr7701_outl(VR7701_PCI_CFGATTR, VR7701_PCI_CFGATTR_WRITE);
		vr7701_outl(VR7701_PCI_CFGADDR, addr);
		vr7701_outl(VR7701_PCI_CFGDATA, *data);
		DBG("write a 0x%08x d 0x%08x ", addr, *data);
		vr7701_outl(VR7701_PCI_CFGATTR, VR7701_PCI_CFGATTR_READ);
		vr7701_outl(VR7701_PCI_CFGADDR, addr);
		*data = vr7701_inl(VR7701_PCI_CFGDATA);
		DBG("written d 0x%08x\n", *data);
	} else {
		*data = 0xffffffff;
		return -1;
	}
	return 0;
}

static int
read_config_byte(struct pci_dev *dev, int where, u8 * val)
{
	u32 data = 0;

	if (vr7701_config_access(PCI_ACCESS_READ, dev, where, &data)) {
		*val = 0xff;
		return -1;
	}

	*val = (data >> ((where & 3) << 3)) & 0xff;
	DBG("cfg read byte: bus %d dev_fn %x where %x: val %x\n",
	    dev->bus->number, dev->devfn, where, *val);

	return PCIBIOS_SUCCESSFUL;
}

static int
read_config_word(struct pci_dev *dev, int where, u16 * val)
{
	u32 data = 0;

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	if (vr7701_config_access(PCI_ACCESS_READ, dev, where, &data)) {
		*val = 0xffff;
		return -1;
	}

	*val = (data >> ((where & 3) << 3)) & 0xffff;
	DBG("cfg read word: bus %d dev_fn %x where %x: val %x\n",
	    dev->bus->number, dev->devfn, where, *val);

	return PCIBIOS_SUCCESSFUL;
}

static int
read_config_dword(struct pci_dev *dev, int where, u32 * val)
{
	u32 data = 0;

	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	if (vr7701_config_access(PCI_ACCESS_READ, dev, where, &data)) {
		*val = 0xffffffff;
		return -1;
	}

	*val = data;
	DBG("cfg read dword: bus %d dev_fn %x where %x: val %x\n",
	    dev->bus->number, dev->devfn, where, *val);

	return PCIBIOS_SUCCESSFUL;
}

static int
write_config_byte(struct pci_dev *dev, int where, u8 val)
{
	u32 data = 0;

	if (vr7701_config_access(PCI_ACCESS_READ, dev, where, &data))
		return -1;

	data = (data & ~(0xff << ((where & 3) << 3))) |
	    (val << ((where & 3) << 3));
	DBG("cfg write byte: bus %d dev_fn %x where %x: val %x\n",
	    dev->bus->number, dev->devfn, where, val);

	if (vr7701_config_access(PCI_ACCESS_WRITE, dev, where, &data))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}

static int
write_config_word(struct pci_dev *dev, int where, u16 val)
{
	u32 data = 0;

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	if (vr7701_config_access(PCI_ACCESS_READ, dev, where, &data))
		return -1;

	data = (data & ~(0xffff << ((where & 3) << 3))) |
	    (val << ((where & 3) << 3));
	DBG("cfg write word: bus %d dev_fn %x where %x: val %x\n",
	    dev->bus->number, dev->devfn, where, val);

	if (vr7701_config_access(PCI_ACCESS_WRITE, dev, where, &data))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}

static int
write_config_dword(struct pci_dev *dev, int where, u32 val)
{
	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	if (vr7701_config_access(PCI_ACCESS_WRITE, dev, where, &val))
		return -1;
	DBG("cfg write dword: bus %d dev_fn %x where %x: val %x\n",
	    dev->bus->number, dev->devfn, where, val);

	return PCIBIOS_SUCCESSFUL;
}

/* 
 * VR7701 chip-specific PCI fixup. It should be invoked from board-specific
 * pcibios_fixup() routine.
 */
void
vr7701_pci_fixup(void)
{
	/* Map the host memory to PCI bus */
	vr7701_outl(VR7701_PCI_TGT1_SEG, 0);
	vr7701_outl(VR7701_PCI_TGT1_MASK,
		    (HOST_MEM_SIZE - VR7701_PCI_TGT_MASK_MIN) |
		    VR7701_PCI_TGT_MASK_EN);
	vr7701_outl(VR7701_PCI_TGT1_SWAP, VR7701_PCI_TGT_SWAP_NO);

	/* Disable second window */
	vr7701_outl(VR7701_PCI_TGT2_MASK, 0);

	/* Enable VR7701 memory space */
	vr7701_outw(VR7701_PCI_COMMAND, vr7701_inw(VR7701_PCI_COMMAND) |
		    PCI_COMMAND_MEMORY);

}

/*
 * Fixup for ioremap function. 
 * In VR7701, PCI memory window mapped to the physical space with ranges
 * 0x100000000-0xf00000000. On PCI bus 32 bit addresses are used, for
 * compatibility with 32-bit PCI peripheral, at least. This function
 * adjusting the physical and PCI address mapping, to allow to use
 * PCI addresses as a physical addresses.
 */
static phys_addr_t
vr7701_ioremap_fixup(phys_addr_t phys_addr, phys_addr_t size)
{
	/* check for pci memory window */
	if ((phys_addr >= MEM_BASE) && (size < MEM_SIZE)) {
		return (phys_addr_t) (phys_addr + VR7701_PCI_OCB_MAP);
	} else
		return phys_addr;
}

/* Initialise the VR7701 PCI-X unit properly */
void
vr7701_pci_setup(void)
{
	/* Reset the PCI bus */
	vr7701_outl(VR7701_PCI_RESET, VR7701_PCI_RESET_ASSERTED);
	udelay(1000);		/* PCI spec: Trst = 1ms */
	vr7701_outl(VR7701_PCI_RESET, VR7701_PCI_RESET_DEASSERTED);
	udelay(1000000);	/* PCI spec: we should wait 2^25 clocks to allow
				   devices complete their initialisation */

	/* Enable VR7701 bus master and memory space; enable SERR# */
	vr7701_outw(VR7701_PCI_COMMAND, vr7701_inw(VR7701_PCI_COMMAND) |
		    PCI_COMMAND_MASTER | PCI_COMMAND_MEMORY | PCI_COMMAND_SERR);

	vr7701_outb(VR7701_PCI_LAT, 64);

	/* Plain old PCI bus is supported. Therefore, lets don't
	   expose any advanced capabilities, like PCI-X or MSI */
	vr7701_outb(VR7701_PCI_CAPP, 0);

	/* Set the PCI Master segment offset */
	vr7701_outl(VR7701_PCI_MST_SEG, (0xFFFFFFFFULL &
					 ((unsigned long long) 0 -
					  VR7701_PCI_OCB_MAP))
		    | VR7701_PCI_MST_SWAP_NO);
	vr7701_outl(VR7701_PCI_MST_SEG_MSW, 0xFFFFFFFFULL &
		    (((unsigned long long) 0 - VR7701_PCI_OCB_MAP) >> 32));

	/* Set the PCI Target Base Address */
	vr7701_outl(VR7701_PCI_BAR0, 0xFFFFFFFFULL &
		    (PCI_BASE_ADDRESS_MEM_PREFETCH
		     | PCI_BASE_ADDRESS_MEM_TYPE_32
		     | PCI_BASE_ADDRESS_SPACE_MEMORY));

	/* PCI target memory disabled here. PCI target memory mapping will
	   be established in pci_fixup. */
	vr7701_outl(VR7701_PCI_TGT1_MASK, 0);
	vr7701_outl(VR7701_PCI_TGT2_MASK, 0);

	/* Set device/vendor ID to 0xFFFF to avoid self-initialisation */
	vr7701_outw(VR7701_PCI_VENDOR_ID, 0xFFFF);
	vr7701_outw(VR7701_PCI_DEVICE_ID, 0xFFFF);

	/* General unit configuration */
	vr7701_outl(VR7701_PCI_UCTL, VR7701_PCI_UCTL_INITEND |	/* Initialisation is completed */
		    VR7701_PCI_UCTL_PBA |	/* Function as arbiter */
		    VR7701_PCI_UCTL_HOST |	/* Host mode */
		    VR7701_PCI_UCTL_CLK_33 |	/* Clock is 132 MHz */
		    VR7701_PCI_UCTL_PCIMODE);	/* PCI mode of operation */

	/* Unmask the PCI device interrupts */
	vr7701_outl(VR7701_PCI_INT_MASK,
		    VR7701_PCI_INT_MASK_INTA |
		    VR7701_PCI_INT_MASK_INTB |
		    VR7701_PCI_INT_MASK_INTC | VR7701_PCI_INT_MASK_INTD);

	/* Disable the PCI errors interrupts */
	vr7701_outl(VR7701_PCI_ERR_MASK, 0);

	/* Initialize ioremap fixup pointer */
	fixup_bigphys_addr = vr7701_ioremap_fixup;

	/* Set IO base */
	set_io_port_base(VR7701_IO_BASE);

}

struct pci_ops vr7701_pci_ops = {
	read_config_byte,
	read_config_word,
	read_config_dword,
	write_config_byte,
	write_config_word,
	write_config_dword
};

#endif				/* CONFIG_PCI */
