/*
  * arch/mips/vr7701/common/nmi.c
  *
  * Error interrupt handling for NEC VR7701 processor.
  *
  * Author: MontaVista Software, Inc. <source@mvista.com>
  *
  * 2003 (c) MontaVista Software, Inc. This file is licensed under
  * the terms of the GNU General Public License version 2. This program
  * is licensed "as is" without any warranty of any kind, whether express
  * or implied.
  */

#include <linux/init.h>
#include <linux/kernel_stat.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/interrupt.h>

#include <asm/irq.h>
#include <asm/mipsregs.h>
#include <asm/vr7701.h>

static void
nmi_handler(int irq, void *arg, struct pt_regs *regs)
{
	printk("Error interrupt %d occured: %s\n", irq,
	       irq == VR7701_NMI_EXT ? "NMI button" :
	       irq == VR7701_NMI_BIU_ERR ? "Bus Interface Unit bus error" :
	       irq == VR7701_NMI_DRAM_CORR ? "DRAM ECC correctable error" :
	       irq == VR7701_NMI_DRAM_ERR ? "DRAM error" :
	       irq == VR7701_NMI_CACHE_PAR ? "Secondary cache parity error" :
	       irq == VR7701_NMI_CACHE_CORR ? "Secondary cache ECC "
	       "correctable error" :
	       irq == VR7701_NMI_CACHE_ERR ? "Secondary cache ECC error" :
	       irq == VR7701_NMI_BCU_PAR ? "Local Bus parity error between "
	       "Local Bus and OnChipBus" :
	       irq == VR7701_NMI_ETH1_PAR ? "Ethernet 1 Parity Error" :
	       irq == VR7701_NMI_ETH2_PAR ? "Ethernet 2 Parity Error" :
	       "Unknown?");
	if (irq == VR7701_NMI_BIU_ERR) {
		u32 err1, err2;
		char *info;
		err1 = vr7701_inl(VR7701_BIU_ERR_1);
		err2 = vr7701_inl(VR7701_BIU_ERR_2);
		switch (err2 & VR7701_BIU_ERR_2_INFO) {
		case VR7701_BIU_ERR_2_INFO_CMD:
			info = "command error";
			break;
		case VR7701_BIU_ERR_2_UNALIGNED_BURST:
			info = "burst access to unaligned address";
			break;
		case VR7701_BIU_ERR_2_CONN_BE:
			info = "connection byte enable error";
			break;
		case VR7701_BIU_ERR_2_NODEV:
			info = "device not detected";
			break;
		case VR7701_BIU_ERR_2_RETRY:
			info = "retry counter out";
			break;
		case VR7701_BIU_ERR_2_UNALIGNED:
			info = "access to non-byte enabled target";
			break;
		case VR7701_BIU_ERR_2_OVERSIZE_4:
			info = "burst length oversize (tgt buf 4 bytes)";
			break;
		case VR7701_BIU_ERR_2_OVERSIZE_8:
			info = "burst length oversize (tgt buf 8 bytes)";
			break;
		case VR7701_BIU_ERR_2_OVERSIZE_16:
			info = "burst length oversize (tgt buf 16 bytes)";
			break;
		case VR7701_BIU_ERR_2_OVERSIZE_32:
			info = "burst length oversize (tgt buf 32 bytes)";
			break;
		case VR7701_BIU_ERR_2_OVERSIZE_64:
			info = "burst length oversize (tgt buf 64 bytes)";
			break;
		case VR7701_BIU_ERR_2_OVERSIZE_128:
			info = "burst length oversize (tgt buf 128 bytes)";
			break;
		default:
			info = "unknown error";
		}

		printk("%s %s %s addr 0x%1x%08x len %d\n",
		       err2 & VR7701_BIU_ERR_2_MEM ? "mem" : "I/O",
		       err2 & VR7701_BIU_ERR_2_WR ? "write" : "read ",
		       info,
		       err2 & VR7701_BIU_ERR_2_ADDRHI, err1,
		       4 << ((err2 & VR7701_BIU_ERR_2_LEN) >> 6));
	}
#if 0
	show_regs(regs);
	if (irq == VR7701_NMI_BIU_ERR ||
	    irq == VR7701_NMI_DRAM_ERR ||
	    irq == VR7701_NMI_CACHE_PAR || irq == VR7701_NMI_CACHE_ERR) {
		panic("Fatal error interrupt occured!\n");
	}
#endif
}

static struct irqaction nmi_ext =
    { nmi_handler, SA_INTERRUPT, 0, "err: external", NULL, NULL };

static struct irqaction nmi_biu =
    { nmi_handler, SA_INTERRUPT, 0, "err: BIU", NULL, NULL };

static struct irqaction nmi_dram_corr =
    { nmi_handler, SA_INTERRUPT, 0, "err: DRAM correctable", NULL, NULL };

static struct irqaction nmi_dram_err =
    { nmi_handler, SA_INTERRUPT, 0, "err: DRAM unrecoverable", NULL, NULL };

static struct irqaction nmi_cache_par =
    { nmi_handler, SA_INTERRUPT, 0, "err: L2 cache parity", NULL, NULL };

static struct irqaction nmi_cache_corr =
    { nmi_handler, SA_INTERRUPT, 0, "err: L2 cache ECC correctable", NULL,
NULL };

static struct irqaction nmi_cache_err =
    { nmi_handler, SA_INTERRUPT, 0, "err: L2 cache ECC unrecoverable", NULL,
NULL };

static struct irqaction nmi_bcu =
    { nmi_handler, SA_INTERRUPT, 0, "err: BCU parity", NULL, NULL };

static struct irqaction nmi_eth1_par =
    { nmi_handler, SA_INTERRUPT, 0, "err: ethernet1 parity", NULL, NULL };

static struct irqaction nmi_eth2_par =
    { nmi_handler, SA_INTERRUPT, 0, "err: ethernet2 parity", NULL, NULL };

/* Install default error IRQ handlers */
void __init
vr7701_nmi_install(void)
{
	setup_irq(VR7701_NMI_EXT, &nmi_ext);
	setup_irq(VR7701_NMI_BIU_ERR, &nmi_biu);
	setup_irq(VR7701_NMI_DRAM_CORR, &nmi_dram_corr);
	setup_irq(VR7701_NMI_DRAM_ERR, &nmi_dram_err);
	setup_irq(VR7701_NMI_CACHE_PAR, &nmi_cache_par);
	setup_irq(VR7701_NMI_CACHE_CORR, &nmi_cache_corr);
	setup_irq(VR7701_NMI_CACHE_ERR, &nmi_cache_err);
	setup_irq(VR7701_NMI_BCU_PAR, &nmi_bcu);
	setup_irq(VR7701_NMI_ETH1_PAR, &nmi_eth1_par);
	setup_irq(VR7701_NMI_ETH2_PAR, &nmi_eth2_par);
}
