/*
  * arch/mips/vr7701/common/kgdb_io.c
  *
  * kgdb io functions for VR7701. We use the first serial port (upper one).
  * Based on arch/mips/ddb5xxx/ddb5477/kgdb_io.c.  
  *
  * Author: MontaVista Software, Inc. <source@mvista.com>
  *
  * 2001-2003 (c) MontaVista Software, Inc. This file is licensed under
  * the terms of the GNU General Public License version 2. This program
  * is licensed "as is" without any warranty of any kind, whether express
  * or implied.
  */

#include <asm/vr7701.h>

/* ======================= CONFIG ======================== */

#define         MAX_BAUD                1152000

/*
 * 0 - kgdb does serial init
 * 1 - kgdb skip serial init
 */
static int remoteDebugInitialized = 0;

/*
 * the default baud rate *if* kgdb does serial init
 */
#define		BAUD_DEFAULT		UART16550_BAUD_115200

/* ======================= END OF CONFIG ======================== */

typedef unsigned char uint8;
typedef unsigned int uint32;

#define         UART16550_BAUD_2400             2400
#define         UART16550_BAUD_4800             4800
#define         UART16550_BAUD_9600             9600
#define         UART16550_BAUD_19200            19200
#define         UART16550_BAUD_38400            38400
#define         UART16550_BAUD_57600            57600
#define         UART16550_BAUD_115200           115200

#define         UART16550_PARITY_NONE           0
#define         UART16550_PARITY_ODD            0x08
#define         UART16550_PARITY_EVEN           0x18
#define         UART16550_PARITY_MARK           0x28
#define         UART16550_PARITY_SPACE          0x38

#define         UART16550_DATA_5BIT             0x0
#define         UART16550_DATA_6BIT             0x1
#define         UART16550_DATA_7BIT             0x2
#define         UART16550_DATA_8BIT             0x3

#define         UART16550_STOP_1BIT             0x0
#define         UART16550_STOP_2BIT             0x4

void
debugInit(uint32 baud, uint8 data, uint8 parity, uint8 stop)
{
	/* disable interrupts */
	vr7701_outb(VR7701_UART1_IER, 0);

	/* set up buad rate */
	{
		uint32 divisor;

		/* set DIAB bit */
		vr7701_outb(VR7701_UART1_LCR, 0x80);

		/* set divisor */
		divisor = MAX_BAUD / baud;
		vr7701_outb(VR7701_UART1_DLL, divisor & 0xff);
		vr7701_outb(VR7701_UART1_DLM, (divisor & 0xff00) >> 8);

		/* clear DIAB bit */
		vr7701_outb(VR7701_UART1_LCR, 0x0);
	}

	/* set data format */
	vr7701_outb(VR7701_UART1_LCR, data | parity | stop);
}

uint8
getDebugChar(void)
{
	if (!remoteDebugInitialized) {
		remoteDebugInitialized = 1;
		debugInit(BAUD_DEFAULT,
			  UART16550_DATA_8BIT,
			  UART16550_PARITY_NONE, UART16550_STOP_1BIT);
	}

	while ((vr7701_inb(VR7701_UART1_LSR) & 0x1) == 0) ;
	return vr7701_inb(VR7701_UART1_RBR);
}

int
putDebugChar(uint8 byte)
{
	if (!remoteDebugInitialized) {
		remoteDebugInitialized = 1;
		debugInit(BAUD_DEFAULT,
			  UART16550_DATA_8BIT,
			  UART16550_PARITY_NONE, UART16550_STOP_1BIT);
	}

	while ((vr7701_inb(VR7701_UART1_LSR) & 0x20) == 0) ;
	vr7701_outb(VR7701_UART1_THR, byte);
	return 1;
}
