/*
  * arch/mips/vr7701/common/irq_vr7701.c
  *
  * Second-level and third-level interrupt dispatcher for NEC VR7701 
  * processor; interrupt interface primitives.
  *
  * Author: MontaVista Software, Inc. <source@mvista.com>
  *
  * 2003 (c) MontaVista Software, Inc. This file is licensed under
  * the terms of the GNU General Public License version 2. This program
  * is licensed "as is" without any warranty of any kind, whether express
  * or implied.
  */

#include <linux/irq.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/linkage.h>
#include <linux/bitops.h>

#include <asm/vr7701.h>
#include <asm/mipsregs.h>
#include <asm/io.h>
#include <asm/debug.h>

/* 
 * Board-specific code may select interrupt channel which is cascade
 * for i8259 interrupt controller.
 */
int vr7701_i8259_channel = -1;

static void
vr7701_irq_enable(unsigned int irq)
{
	db_assert(irq >= VR7701_SYS_IRQ_BASE);
	db_assert(irq < VR7701_SYS_IRQ_BASE + VR7701_NUM_SYS_IRQ);

	irq -= VR7701_SYS_IRQ_BASE;
	vr7701_outw(VR7701_INTC_INT_MASK,
		    vr7701_inw(VR7701_INTC_INT_MASK) | (1 << irq));
}

static void
vr7701_irq_disable(unsigned int irq)
{
	db_assert(irq >= VR7701_SYS_IRQ_BASE);
	db_assert(irq < VR7701_SYS_IRQ_BASE + VR7701_NUM_SYS_IRQ);

	irq -= VR7701_SYS_IRQ_BASE;
	vr7701_outw(VR7701_INTC_INT_MASK,
		    vr7701_inw(VR7701_INTC_INT_MASK) & ~(1 << irq));
}

static unsigned int
vr7701_irq_startup(unsigned int irq)
{
	vr7701_irq_enable(irq);
	return 0;
}

#define	vr7701_irq_shutdown	vr7701_irq_disable

static void
vr7701_irq_ack(unsigned int irq)
{
	db_assert(irq >= VR7701_SYS_IRQ_BASE);
	db_assert(irq < VR7701_SYS_IRQ_BASE + VR7701_NUM_SYS_IRQ);

	irq -= VR7701_SYS_IRQ_BASE;
	vr7701_outw(VR7701_INTC_INT_MASK,
		    vr7701_inw(VR7701_INTC_INT_MASK) & ~(1 << irq));
}

static void
vr7701_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		vr7701_irq_enable(irq);
}

static hw_irq_controller vr7701_irq_controller = {
	"VR7701 irq",
	vr7701_irq_startup,
	vr7701_irq_shutdown,
	vr7701_irq_enable,
	vr7701_irq_disable,
	vr7701_irq_ack,
	vr7701_irq_end,
	NULL			/* not needed for uniprocessor arch */
};

void
vr7701_irq_init(void)
{
	extern irq_desc_t irq_desc[];
	u32 i;

	for (i = VR7701_SYS_IRQ_BASE;
	     i < VR7701_SYS_IRQ_BASE + VR7701_NUM_SYS_IRQ; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &vr7701_irq_controller;
	}
	vr7701_outw(VR7701_INTC_INT_ACK, 0);
}

static void
vr7701_nmi_enable(unsigned int irq)
{
	db_assert(irq >= VR7701_NMI_IRQ_BASE);
	db_assert(irq < VR7701_NMI_IRQ_BASE + VR7701_NUM_NMI_IRQ);

	irq -= VR7701_NMI_IRQ_BASE;
	vr7701_outw(VR7701_INTC_NMI_MASK,
		    vr7701_inw(VR7701_INTC_NMI_MASK) | (1 << irq));
}

static void
vr7701_nmi_disable(unsigned int irq)
{
	db_assert(irq >= VR7701_NMI_IRQ_BASE);
	db_assert(irq < VR7701_NMI_IRQ_BASE + VR7701_NUM_NMI_IRQ);

	irq -= VR7701_NMI_IRQ_BASE;
	vr7701_outw(VR7701_INTC_NMI_MASK,
		    vr7701_inw(VR7701_INTC_NMI_MASK) & ~(1 << irq));
}

static unsigned int
vr7701_nmi_startup(unsigned int irq)
{
	vr7701_nmi_enable(irq);
	return 0;
}

#define	vr7701_nmi_shutdown	vr7701_nmi_disable

static void
vr7701_nmi_ack(unsigned int irq)
{
	db_assert(irq >= VR7701_NMI_IRQ_BASE);
	db_assert(irq < VR7701_NMI_IRQ_BASE + VR7701_NUM_NMI_IRQ);

	irq -= VR7701_NMI_IRQ_BASE;
	vr7701_outw(VR7701_INTC_NMI_MASK,
		    vr7701_inw(VR7701_INTC_NMI_MASK) & ~(1 << irq));
	vr7701_outw(VR7701_INTC_NMI_CLR, 1 << irq);
}

static void
vr7701_nmi_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		vr7701_nmi_enable(irq);
}

static hw_irq_controller vr7701_nmi_controller = {
	"VR7701 nmi",
	vr7701_nmi_startup,
	vr7701_nmi_shutdown,
	vr7701_nmi_enable,
	vr7701_nmi_disable,
	vr7701_nmi_ack,
	vr7701_nmi_end,
	NULL			/* not needed for uniprocessor architecture */
};

void
vr7701_nmi_init(void)
{
	extern irq_desc_t irq_desc[];
	u32 i;

	vr7701_outw(VR7701_INTC_NMI_CLR, 0xFFFF);

	for (i = VR7701_NMI_IRQ_BASE;
	     i < VR7701_NMI_IRQ_BASE + VR7701_NUM_NMI_IRQ; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &vr7701_nmi_controller;
	}
}

extern asmlinkage unsigned int do_IRQ(int irq, struct pt_regs *regs);

#ifdef CONFIG_I8259
/* 
 * This function is used to determine pending interrupt number in
 * configurations when it is not possible to obtain the interrupt vector
 * in different way.
 *
 * The following code assumed that 8259 remains in state when IMR can be
 * read from 0x21/0xA1 (RR == RIS == 0 in OCW3).
 */
static inline int
next_8259A_irq(void)
{
	int isr, irq;

	isr = inb(0x20) & ~inb(0x21);
	irq = ffz(~isr);
	if (irq == 2) {
		isr = inb(0xa0) & ~inb(0xA1);
		irq = 8 + ffz(~isr);
	}
	return irq;
}
#endif

asmlinkage int
vr7701_irq_dispatch(struct pt_regs *regs)
{
	u16 vector;

	vector = vr7701_inw(VR7701_INTC_INT_VECTOR) >> 3;
	vr7701_outw(VR7701_INTC_INT_ACK, 0);

#ifdef CONFIG_I8259
	if (vector == vr7701_i8259_channel) {
		int i8259_irq;
		vr7701_outw(VR7701_INTC_INT_CLR, VR7701_INTC_CHN(vector));
		if ((i8259_irq = next_8259A_irq()) < NUM_I8259_IRQ) {
			do_IRQ(I8259_IRQ_BASE + i8259_irq, regs);
		}
	} else
#endif
	if (vector > VR7701_INTCH_NMI) {
		/* 
		 * Clearing the interrupt latch has no effect for
		 * level interrupts, but it clears edge interrupt 
		 * condition, if edge mode selected for the channel.
		 */
		vr7701_outw(VR7701_INTC_INT_CLR, VR7701_INTC_CHN(vector));
		do_IRQ(vector + VR7701_SYS_IRQ_BASE - 1, regs);
	} else if (vector == VR7701_INTCH_NMI) {
		u16 s;
		int i;
		u16 bit;
		s = vr7701_inw(VR7701_INTC_NMI_PENDING);
		for (i = 0, bit = 1; i < VR7701_NUM_NMI_IRQ; bit <<= 1, i++) {
			if (s & bit) {
				do_IRQ(i + VR7701_NMI_IRQ_BASE, regs);
				break;
			}
		}
	}
	return 0;
}
