/*
  * arch/mips/vr7701/cmb-vr7701/pci_fixups.c
  *
  * CMB-VR7701 board-specific PCI fixups.
  *
  * Author: MontaVista Software, Inc. <source@mvista.com>
  *
  * Based on pci_fixups.c template:
  * Copyright 2001, Pete Popov, ppopov@pacbell.net
  *
  * 2003 (c) MontaVista Software, Inc. This file is licensed under
  * the terms of the GNU General Public License version 2. This program
  * is licensed "as is" without any warranty of any kind, whether express
  * or implied.
  */

#include <linux/config.h>

#ifdef CONFIG_PCI

#include <linux/types.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/ide.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/delay.h>

#include <asm/bootinfo.h>
#include <asm/irq.h>
#include <asm/vr7701.h>

extern void vr7701_pci_fixup(void);

#ifdef CONFIG_ROCKHOPPER
extern void __init init_i8259_irqs(void);
extern int __init rockhopper_pci_irq(struct pci_dev *dev);

extern int vr7701_i8259_channel;

static struct irqaction i8259_cascade =
    { no_action, 0, 0, "i8259 cascade", NULL, NULL };

/*
 * CMB-VR7701 CPU board requires hardware modification to handle interrupts
 * from M1535 bridge installed on the Rockhopper baseboard. M1535 has 8259A
 * interrupt controller on-chip; edge-triggered active-high output of this
 * controller routed to PCI INTC# input of VR7701. Active level couldn't
 * be selected in PCI-X block for PCI interrupts (no surprises here).
 * To workaround it, connect PCI INTC# to INTP3 interrupt line.
 *
 * This function checks that appropriate modification has been made and
 * print diagnostic message.
 *
 * Theory of operation:
 *     - disable CPU interrupts
 *     - configure INTP3 as active-high edge-triggered interrupt input
 *     - configure 8259 master PIC; mask all 8259 interrupts
 *     - configure 8253 timer to generate interrupts frequently
 *     - ensure that no interrupt condition occured on INTP3
 *     - unmask IRQ0 timer interrupt
 *     - ensure that interrupt appears on INTP3
 *     - mask IRQ0 and clear interrupt
 *     - ensure that interrupt is not occured
 *     - restore configuration
 */
static int __init
vr7701_rockhopper_detect_hardware_mod(void)
{
	unsigned long flags;
	u16 saved_gpio_sel;
	u16 saved_act_lv;
	u16 saved_lvl_edg;
	int detected = 1;

	save_flags(flags);
	cli();
	saved_gpio_sel = vr7701_inw(VR7701_GPIO_SEL);
	saved_act_lv = vr7701_inw(VR7701_INTC_ACT_LV);
	saved_lvl_edg = vr7701_inw(VR7701_INTC_LVLEDGE);

	vr7701_outw(VR7701_GPIO_SEL,
		    (saved_gpio_sel & ~VR7701_GPIO_SEL_INTP3) |
		    VR7701_GPIO_SEL_INTP3_INT);

	vr7701_outw(VR7701_INTC_ACT_LV, saved_act_lv |
		    VR7701_INTC_ACT_LV_INTP3_HIGH);

	vr7701_outw(VR7701_INTC_LVLEDGE, saved_lvl_edg |
		    VR7701_INTC_CHN(VR7701_INTCH_INTP3));

	/* Typical 8259 interrupt controller initialisation */
	outb_p(0xFF, 0x21);
	outb_p(0xFF, 0xA1);	/* Master and slave mask */
	outb_p(0x13, 0x20);
	outb_p(0x00, 0x21);	/* Master ICW1, ICW2 */
	outb_p(0x00, 0x21);
	outb_p(0x01, 0x21);	/* Master ICW3, ICW4 */

	/* 8259 End-of-interrupt sequence */
	outb_p(0xFF, 0x21);
	outb_p(0x20, 0x20);

	/* Clear locked interrupt */
	vr7701_outw(VR7701_INTC_INT_CLR, VR7701_INTC_CHN(VR7701_INTCH_INTP3));

	/* Initialize 8253 timer to generate timer interrupts (IRQ0) */
	outb_p(0x34, 0x43);
	outb_p(0x03, 0x40);
	outb_p(0x00, 0x40);
	udelay(1000);

	/* interrupts are masked, so no INTP3 pending interrupt expected */
	detected = detected && ((vr7701_inw(VR7701_INTC_INT_STATUS) &
				 VR7701_INTC_CHN(VR7701_INTCH_INTP3)) == 0);

	/* Unmask timer interrupt and check that it appears on INTP3 */
	outb_p(0xFE, 0x21);
	udelay(1000);
	detected = detected && ((vr7701_inw(VR7701_INTC_INT_STATUS) &
				 VR7701_INTC_CHN(VR7701_INTCH_INTP3)) != 0);

	/* Mask timer interrupt and check that INTP3 goes 0 */
	outb_p(0xFF, 0x21);
	udelay(10);
	/* Clear locked interrupt */
	vr7701_outw(VR7701_INTC_INT_CLR, VR7701_INTC_CHN(VR7701_INTCH_INTP3));
	udelay(1000);
	detected = detected && ((vr7701_inw(VR7701_INTC_INT_STATUS) &
				 VR7701_INTC_CHN(VR7701_INTCH_INTP3)) == 0);

	/* Clear locked interrupt */
	vr7701_outw(VR7701_INTC_INT_CLR, VR7701_INTC_CHN(VR7701_INTCH_INTP3));

	vr7701_outw(VR7701_INTC_LVLEDGE, saved_lvl_edg);
	vr7701_outw(VR7701_INTC_ACT_LV, saved_act_lv);
	vr7701_outw(VR7701_GPIO_SEL, saved_gpio_sel);
	restore_flags(flags);

	if (!detected) {
		printk("*** CMB-VR7701 board need to be modified to enable\n"
		       "*** interrupt handling from Rockhopper baseboard\n"
		       "*** peripheral.\n"
		       "*** Please read documentation for details.\n");
	}

	return detected;
}

static void __init
vr7701_rockhopper_init(void)
{
	if (!vr7701_rockhopper_detect_hardware_mod())
		return;

	/* 
	 * At this time, we mask all INTx# PCI interrupts. PCI interrupt
	 * signals routed to the rockhopper M1535 on-chip interrupt
	 * controller.
	 */
	vr7701_outl(VR7701_PCI_INT_MASK, 0);

	/* Enable i8259 cascaded interrupts handling */
	/* 
	 * Note that CMB-VR7701 board need to be modified. M1535 interrupt
	 * output is active high, but INTC# input is active low.
	 * INTP3 interrupt input is used to handle M1535 interrupts 
	 */
	vr7701_outw(VR7701_GPIO_SEL,
		    (vr7701_inw(VR7701_GPIO_SEL) & ~VR7701_GPIO_SEL_INTP3) |
		    VR7701_GPIO_SEL_INTP3_INT);
	vr7701_outw(VR7701_INTC_ACT_LV,
		    vr7701_inw(VR7701_INTC_ACT_LV) |
		    VR7701_INTC_ACT_LV_INTP3_HIGH);
	vr7701_outw(VR7701_INTC_LVLEDGE, vr7701_inw(VR7701_INTC_LVLEDGE) &
		    ~VR7701_INTC_CHN(VR7701_INTCH_INTP3));

	init_i8259_irqs();
	setup_irq(VR7701_IRQ_INTP3, &i8259_cascade);
	vr7701_i8259_channel = VR7701_INTCH_INTP3;
}
#endif

/* PCI resources fixup. Hopefully, nothing to do */
void __init
pcibios_fixup_resources(struct pci_dev *dev)
{
}

/*
 * Any board or system controller fixups go here.
 * Now, this is called after the pci_auto code (if enabled) and
 * after the linux pci scan.
 */
void __init
pcibios_fixup(void)
{
	vr7701_pci_fixup();

#ifdef CONFIG_ROCKHOPPER
	if (mips_machtype == MACH_NEC_CMB_VR7701_ROCKHOPPERII)
		vr7701_rockhopper_init();
#endif
}

/* 
 * This is very board specific. You'll have to look at
 * each pci device and assign its interrupt number.
 * This function is called last of all the fixup functions.
 */
/* 
 * PCI-X block raise the single interrupt in response to all PCI interrupt
 * requests. Rockhopper board has own interrupt controller; all interrupts
 * routed to it.
 */
void __init
pcibios_fixup_irqs(void)
{
	struct pci_dev *dev;

	pci_for_each_dev(dev) {
#ifdef CONFIG_ROCKHOPPER
		if (mips_machtype != MACH_NEC_CMB_VR7701_ROCKHOPPERII)
			dev->irq = VR7701_IRQ_PCIX;
		else
			dev->irq = rockhopper_pci_irq(dev);
#else
		dev->irq = VR7701_IRQ_PCIX;
#endif
	}
}

unsigned int
pcibios_assign_all_busses(void)
{
	return 0;
}
#endif
