/*
  * arch/mips/vr7701/cmb-vr7701/lcd44780.c
  *
  * Simple driver for a memory-mapped 44780-style LCD display.
  * Configured for CMB-VR7701/Rockhopper
  *
  * Author: MontaVista Software, Inc. <source@mvista.com>
  *
  * Based on:
  * Copyright 2001 Bradley D. LaRonde <brad@ltc.com>
  *
  * 2003 (c) MontaVista Software, Inc. This file is licensed under
  * the terms of the GNU General Public License version 2. This program
  * is licensed "as is" without any warranty of any kind, whether express
  * or implied.
  */

#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <asm/bootinfo.h>

#include "lcd44780.h"

#define LCD44780_COMMAND   ((volatile unsigned char *)0xb9020000)
#define LCD44780_DATA      ((volatile unsigned char *)0xb9020001)

#define LCD_GREETING "MontaVista Linux"

#define MISC_MAJOR 10

static int lcd_open(struct inode *inode, struct file *file);
static ssize_t lcd_read(struct file *file, char *s, size_t size, loff_t * ppos);
static ssize_t lcd_write(struct file *file, const char *s,
			 size_t size, loff_t * ppos);
static int lcd_ioctl(struct inode *inode, struct file *file,
		     unsigned int cmd, unsigned long arg);
static loff_t lcd_llseek(struct file *file, loff_t pos, int);
static int lcd_close(struct inode *inode, struct file *file);

/* LCD cursor position variable. '-1' is inconsistent value. */
static int lcd_position;

/* Misc device structures */
static struct file_operations lcd_fops = {
	llseek:lcd_llseek,
	read:lcd_read,
	write:lcd_write,
	ioctl:lcd_ioctl,
	open:lcd_open,
	release:lcd_close,
};

#ifdef	CONFIG_DEVFS_FS
static struct miscdevice lcd_dev = {
	MISC_DYNAMIC_MINOR,
	"lcd44780",
	&lcd_fops
};
#endif

/*
 * Private helpers
 */
static inline void
lcd_wait(void)
{
	int i, j;
	for (i = 0; i < 400; i++)
		for (j = 0; j < 10000; j++) ;
}

static inline void
lcd_command(unsigned char c)
{
	*(volatile u8 *) LCD44780_COMMAND = c;
	lcd_wait();
}

static inline void
lcd_writechar(unsigned char c)
{
	*(volatile u8 *) LCD44780_DATA = c;
	lcd_wait();
}

static inline unsigned char
lcd_readchar(void)
{
	unsigned char c = *(volatile u8 *) LCD44780_DATA;
	lcd_wait();
	return c;
}

static inline void
lcd_reset(void)
{
	lcd_command(LCD44780_8BIT_2LINE);
	lcd_command(LCD44780_DD_ADDRESS);
	lcd_command(LCD44780_CURSOR_BLOCK);
	lcd_command(LCD44780_CLEAR);
	lcd_position = 0;
}

static void
lcd_seek(int pos)
{
	int i, j;

	if (lcd_position == -1)
		lcd_reset();
	if (pos == lcd_position)
		return;

	lcd_command(LCD44780_HOME);
	for (i = 0; i < pos; i++) {
		lcd_command(LCD44780_RIGHT);
		if (i == 8) {
			for (j = 0; j < 32; j++)
				lcd_command(LCD44780_RIGHT);
		}
	}
	lcd_position = pos;
}

/*
 * Functions accessed from outside
 */

static int
lcd_init(void)
{
	int err;
	loff_t start = 0;

	lcd_reset();
	lcd_wait();
	lcd_write(NULL, LCD_GREETING, sizeof (LCD_GREETING) - 1, &start);

	/* Register the lcd driver */
#ifdef	CONFIG_DEVFS_FS
	err = misc_register(&lcd_dev);
#else
	err = register_chrdev(0, "lcd44780", &lcd_fops);
	printk(KERN_NOTICE "lcd44780: autoassigned major number=%d\n", err);
#endif
	if (err < 0)
		printk(KERN_WARNING
		       "failed to register lcd device: err=%d\n", err);

	return err;
}

module_init(lcd_init);

static int
lcd_open(struct inode *inode, struct file *file)
{
	if (mips_machtype != MACH_NEC_CMB_VR7701_ROCKHOPPERII)
		return -ENXIO;
	else
		return 0;
}

static int
lcd_close(struct inode *inode, struct file *file)
{
	lcd_seek(16);
	return 0;
}

static loff_t
lcd_llseek(struct file *file, loff_t pos, int orig)
{
	if (orig == 1)		/* SEEK_CUR */
		pos = file->f_pos + pos;
	else if (orig != 0)	/* SEEK_SET */
		return -EINVAL;

	if (pos < 16 && pos >= 0)
		return pos;
	else
		return -EINVAL;
}

static ssize_t
lcd_read(struct file *file, char *s, size_t size, loff_t * ppos)
{
	int i, j;

	/* Let's position the cursor */
	if (*ppos > 16 || *ppos < 0)
		return -EFAULT;
	if (*ppos == 16)
		return 0;
	lcd_seek(*ppos);
	lcd_readchar();

	/* Read characters */
	for (i = 0; i < size; i++) {
		*s = lcd_readchar();
		s++;
		lcd_position++;
		if (lcd_position == 8) {
			/* 
			 * We must write 32 of spaces to get cursor 
			 * to 2nd line 
			 */
			for (j = 0; j < 32; j++)
				lcd_readchar();
		}
		/* 
		 * We have filled all 16 character positions, 
		 * so stop outputing data 
		 */
		if (lcd_position == 16) {
			i++;
			break;
		}
	}

	*ppos = lcd_position;
	lcd_position++;
	return i;
}

static ssize_t
lcd_write(struct file *file, const char *s, size_t size, loff_t * ppos)
{
	int i, j;

	/* Let's position the cursor */
	if (*ppos >= 16 || *ppos < 0)
		return -EFAULT;
	lcd_seek(*ppos);

	/* Print characters */
	for (i = 0; i < size; i++) {
		lcd_writechar(*s);
		s++;
		lcd_position++;
		if (lcd_position == 8) {
			/* 
			 * We must write 32 of spaces to get cursor 
			 * to 2nd line 
			 */
			for (j = 0; j < 32; j++)
				lcd_writechar(' ');
		}
		/* 
		 * We have filled all 16 character positions, 
		 * so stop outputing data 
		 */
		if (lcd_position == 16) {
			i++;
			break;
		}
	}

	*ppos = lcd_position;
	return i;
}

static int
lcd_ioctl(struct inode *inode, struct file *file,
	  unsigned int cmd, unsigned long arg)
{
	switch (cmd) {
	case LCD_IOCTL_ACTION:
		lcd_command(arg);
		lcd_position = -1;
		return 0;
		break;

	case LCD_IOCTL_READ:
		*(unsigned char *) arg = lcd_readchar();
		return 0;
		break;

	case LCD_IOCTL_WRITE:
		lcd_writechar(arg);
		return 0;
		break;

	default:
		return -EINVAL;
	}
}
