/*
 * arch/mips/vr41xx/nec-cmbvr41xx/m1535plus.c
 *
 * Initialize for ALi M1535+(included M5229 and M5237).
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/serial.h>

#include <asm/vr41xx/cmbvr41xx.h>

#ifdef CONFIG_PCI
#include <linux/pci.h>

#include <asm/io.h>

#define M1535PLUS_BASE_BAUD	(1843200 / 16)

#define CONFIG_PORT(port)	((port) ? 0x3f0 : 0x370)
#define DATA_PORT(port)		((port) ? 0x3f1 : 0x371)
#define INDEX_PORT(port)	CONFIG_PORT(port)

#define ENTER_CONFIG_MODE(port)				\
	do {						\
		outb_p(0x51, CONFIG_PORT(port));	\
		outb_p(0x23, CONFIG_PORT(port));	\
	} while(0)

#define SELECT_LOGICAL_DEVICE(port, dev_no)		\
	do {						\
		outb_p(0x07, INDEX_PORT(port));		\
		outb_p((dev_no), DATA_PORT(port));	\
	} while(0)

#define WRITE_CONFIG_DATA(port,index,data)		\
	do {						\
		outb_p((index), INDEX_PORT(port));	\
		outb_p((data), DATA_PORT(port));	\
	} while(0)

#define EXIT_CONFIG_MODE(port)	outb(0xbb, CONFIG_PORT(port))

#define PCI_CONFIG_ADDR	KSEG1ADDR(0x0f000c18)
#define PCI_CONFIG_DATA	KSEG1ADDR(0x0f000c14)

#ifdef CONFIG_BLK_DEV_FD

void __devinit ali_m1535plus_fdc_init(int port)
{
	ENTER_CONFIG_MODE(port);
	SELECT_LOGICAL_DEVICE(port, 0);		/* FDC */
	WRITE_CONFIG_DATA(port, 0x30, 0x01);	/* FDC: enable */
	WRITE_CONFIG_DATA(port, 0x60, 0x03);	/* I/O port base: 0x3f0 */
	WRITE_CONFIG_DATA(port, 0x61, 0xf0);
	WRITE_CONFIG_DATA(port, 0x70, 0x06);	/* IRQ: 6 */
	WRITE_CONFIG_DATA(port, 0x74, 0x02);	/* DMA: channel 2 */
	WRITE_CONFIG_DATA(port, 0xf0, 0x08);
	WRITE_CONFIG_DATA(port, 0xf1, 0x00);
	WRITE_CONFIG_DATA(port, 0xf2, 0xff);
	WRITE_CONFIG_DATA(port, 0xf4, 0x00);
	EXIT_CONFIG_MODE(port);
}

#endif

void __devinit ali_m1535plus_parport_init(int port)
{
	ENTER_CONFIG_MODE(port);
	SELECT_LOGICAL_DEVICE(port, 3);		/* Parallel Port */
	WRITE_CONFIG_DATA(port, 0x30, 0x01);
	WRITE_CONFIG_DATA(port, 0x60, 0x03);	/* I/O port base: 0x378 */
	WRITE_CONFIG_DATA(port, 0x61, 0x78);
	WRITE_CONFIG_DATA(port, 0x70, 0x07);	/* IRQ: 7 */
	WRITE_CONFIG_DATA(port, 0x74, 0x04);	/* DMA: None */
	WRITE_CONFIG_DATA(port, 0xf0, 0x8c);	/* IRQ polarity: Active Low */
	WRITE_CONFIG_DATA(port, 0xf1, 0xc5);
	EXIT_CONFIG_MODE(port);
}

#ifdef CONFIG_SERIAL

void __devinit ali_m1535plus_uart1_init(int port)
{
	struct serial_struct s;

	ENTER_CONFIG_MODE(port);
	SELECT_LOGICAL_DEVICE(port, 4);		/* UART1 */
	WRITE_CONFIG_DATA(port, 0x30, 0x01);
	WRITE_CONFIG_DATA(port, 0x60, 0x03);	/* I/O port base: 0x3f8 */
	WRITE_CONFIG_DATA(port, 0x61, 0xf8);
	WRITE_CONFIG_DATA(port, 0x70, 0x04);	/* IRQ: 4 */
	WRITE_CONFIG_DATA(port, 0xf0, 0x00);
	WRITE_CONFIG_DATA(port, 0xf1, 0x00);
	WRITE_CONFIG_DATA(port, 0xf2, 0x00);
	EXIT_CONFIG_MODE(port);

	memset(&s, 0, sizeof(s));

	s.line = vr41xx_serial_ports;
	s.baud_base = M1535PLUS_BASE_BAUD;
	s.port = 0x3f8;
	s.irq = UART1_IRQ;
	s.flags = ASYNC_BOOT_AUTOCONF | ASYNC_SKIP_TEST;
	s.io_type = SERIAL_IO_PORT;
	if (early_serial_setup(&s) != 0)
		printk(KERN_ERR "ALi M1535+ UART1 setup failed!\n");
	else
		vr41xx_serial_ports++;
}

void __devinit ali_m1535plus_uart2_init(int port)
{
	ENTER_CONFIG_MODE(port);
	SELECT_LOGICAL_DEVICE(port, 5);		/* UART2 */
	WRITE_CONFIG_DATA(port, 0x30, 0x00);	/* UART2 disable */
	EXIT_CONFIG_MODE(port);
}

void __devinit ali_m1535plus_uart3_init(int port)
{
	struct serial_struct s;

	ENTER_CONFIG_MODE(port);
	SELECT_LOGICAL_DEVICE(port, 0xb);	/* UART3 */
	WRITE_CONFIG_DATA(port, 0x30, 0x01);	/* UART3: enable */
	WRITE_CONFIG_DATA(port, 0x60, 0x02);    /* IO port base: 0x280 */
	WRITE_CONFIG_DATA(port, 0x61, 0xf8);
	WRITE_CONFIG_DATA(port, 0x70, 0x03);    /* IRQ: 3 */
	WRITE_CONFIG_DATA(port, 0xf0, 0x00);
	WRITE_CONFIG_DATA(port, 0xf1, 0x00);
	WRITE_CONFIG_DATA(port, 0xf2, 0x00);
	EXIT_CONFIG_MODE(port);

	memset(&s, 0, sizeof(s));

	s.line = vr41xx_serial_ports;
	s.baud_base = M1535PLUS_BASE_BAUD;
	s.port = 0x2f8;
	s.irq = UART3_IRQ;
	s.flags = ASYNC_BOOT_AUTOCONF | ASYNC_SKIP_TEST;
	s.io_type = SERIAL_IO_PORT;
	if (early_serial_setup(&s) != 0)
		printk(KERN_ERR "ALi M1535+ UART3 setup failed!\n");
	else
		vr41xx_serial_ports++;
}

#endif

#ifdef CONFIG_PC_KEYB

void __devinit ali_m1535plus_keyboard_init(int port)
{
	ENTER_CONFIG_MODE(port);
	SELECT_LOGICAL_DEVICE(port, 7);		/* KEYBOARD */
	WRITE_CONFIG_DATA(port, 0x30, 0x01);	/* KEYBOARD: eable */
	WRITE_CONFIG_DATA(port, 0x70, 0x01);	/* IRQ: 1 */
	WRITE_CONFIG_DATA(port, 0x72, 0x0c);	/* PS/2 Mouse IRQ: 12 */
	WRITE_CONFIG_DATA(port, 0xf0, 0x00);
	EXIT_CONFIG_MODE(port);
}

#endif

void __devinit ali_m1535plus_hotkey_init(int port)
{
	ENTER_CONFIG_MODE(port);
	SELECT_LOGICAL_DEVICE(port, 0xc);	/* HOTKEY */
	WRITE_CONFIG_DATA(port, 0x30, 0x00);
	WRITE_CONFIG_DATA(port, 0xf0, 0x35);
	WRITE_CONFIG_DATA(port, 0xf1, 0x14);
	WRITE_CONFIG_DATA(port, 0xf2, 0x11);
	WRITE_CONFIG_DATA(port, 0xf3, 0x71);
	WRITE_CONFIG_DATA(port, 0xf5, 0x05);
	EXIT_CONFIG_MODE(port);
}

void __init ali_m1535plus_init(struct pci_dev *dev)
{
	pci_write_config_byte(dev, 0x40, 0x18);
	pci_write_config_byte(dev, 0x41, 0xc0);
	pci_write_config_byte(dev, 0x42, 0x41);
	pci_write_config_byte(dev, 0x43, 0x00);
	pci_write_config_byte(dev, 0x44, 0x5d);
	pci_write_config_byte(dev, 0x45, 0x0b);
	pci_write_config_byte(dev, 0x47, 0x00);

	/* IRQ routing */
	pci_write_config_byte(dev, 0x48, 0x03); /* INTA IRQ10, INTB disable */
	pci_write_config_byte(dev, 0x49, 0x00); /* INTC and INTD disable */
	pci_write_config_byte(dev, 0x4a, 0x00); /* INTE and INTF disable */
	pci_write_config_byte(dev, 0x4b, 0x90); /* Audio IRQ11, Modem disable */

	pci_write_config_word(dev, 0x50, 0x0000);
	pci_write_config_word(dev, 0x52, 0x0000);
	pci_write_config_word(dev, 0x54, 0x0002);
	pci_write_config_word(dev, 0x56, 0x0002);

	pci_write_config_byte(dev, 0x59, 0x00);
	pci_write_config_byte(dev, 0x5a, 0x00);
	pci_write_config_byte(dev, 0x5b, 0x00);
	pci_write_config_word(dev, 0x5c, 0x0000);
	pci_write_config_byte(dev, 0x5e, 0x00);
	pci_write_config_byte(dev, 0x5f, 0x00);
	pci_write_config_word(dev, 0x60, 0x0000);

	pci_write_config_byte(dev, 0x6c, 0x00);
	pci_write_config_byte(dev, 0x6d, 0x04);
	pci_write_config_byte(dev, 0x6e, 0x00);

	pci_write_config_byte(dev, 0x70, 0x12);
	pci_write_config_byte(dev, 0x71, 0x00);
	pci_write_config_byte(dev, 0x72, 0x00);
	pci_write_config_byte(dev, 0x73, 0x00);

	/*
	 * IRQ setup ALi M5237 USB Host Controller
	 * IRQ: 9
	 */
	pci_write_config_byte(dev, 0x74, 0x01);

	pci_write_config_byte(dev, 0x75, 0x1f);
	pci_write_config_byte(dev, 0x76, 0x80);
	pci_write_config_byte(dev, 0x77, 0x20);
	pci_write_config_dword(dev, 0x78, 0x20000000);
	pci_write_config_byte(dev, 0x7c, 0x00);
	pci_write_config_byte(dev, 0x81, 0x00);
	pci_write_config_byte(dev, 0x90, 0x00);
	pci_write_config_word(dev, 0xa4, 0x0000);

#ifdef CONFIG_BLK_DEV_FD
	ali_m1535plus_fdc_init(1);
#endif

#ifdef CONFIG_PARPORT
	ali_m1535plus_parport_init(1);
#endif

#ifdef CONFIG_SERIAL
	ali_m1535plus_uart3_init(1);
	ali_m1535plus_uart1_init(1);
	ali_m1535plus_uart2_init(1);
#endif

#ifdef CONFIG_PC_KEYB
	ali_m1535plus_keyboard_init(1);
	ali_m1535plus_hotkey_init(1);
#endif
}

static inline void ali_config_writeb(u8 reg, u8 val)
{
	u32 data;
	int shift;

	writel((1UL << 0x12) | (reg & 0xfc), PCI_CONFIG_ADDR);
        data = readl(PCI_CONFIG_DATA);

	shift = (reg & 3) << 3;
	data &= ~(0xff << shift);
	data |= (((u32)val) << shift);

	writel(data, PCI_CONFIG_DATA);
}

static inline u16 ali_config_readw(u8 reg)
{
	u32 data;

	writel(((1UL << 0x12) | (reg & 0xfc)) , PCI_CONFIG_ADDR);
	data = readl(PCI_CONFIG_DATA);

	return (u16)(data >> ((reg & 2) << 3));
}

void __init ali_m5229_preinit(void)
{
	if (ali_config_readw(PCI_VENDOR_ID) == PCI_VENDOR_ID_AL &&
	    ali_config_readw(PCI_DEVICE_ID) == PCI_DEVICE_ID_AL_M1533) {
		/*
		 * Enable ALi M5229 IDE Controller
		 * IDSEL: A27
		 */
		 ali_config_writeb(0x58, 0x4c);
	}
}

void __init ali_m5229_init(struct pci_dev *dev)
{
	/*
	 * Setup base addresses
	 */
	pci_write_config_dword(dev, PCI_BASE_ADDRESS_0, 0x1f1);
	pci_write_config_dword(dev, PCI_BASE_ADDRESS_1, 0x3f5);
	pci_write_config_dword(dev, PCI_BASE_ADDRESS_2, 0x171);
	pci_write_config_dword(dev, PCI_BASE_ADDRESS_3, 0x375);
	pci_write_config_dword(dev, PCI_BASE_ADDRESS_4, 0xf001);

	/*
	 * Enable Primary/Secondary Channel Cable Detect 40-Pin
	 */
	pci_write_config_word(dev, 0x4a, 0xc023);

	/*
	 * Set only the 3rd byteis for the master IDE's cycle and
	 * enable Internal IDE Function
	 */
	pci_write_config_byte(dev, 0x50, 0x23);

	pci_write_config_byte(dev, 0x09, 0xff);
	pci_write_config_byte(dev, 0x52, 0x00);
	pci_write_config_byte(dev, 0x58, 0x03);
	pci_write_config_byte(dev, 0x59, 0x0a);
	pci_write_config_byte(dev, 0x5a, 0x0a);
	pci_write_config_byte(dev, 0x5b, 0x0a);
	pci_write_config_byte(dev, 0x5c, 0x03);
	pci_write_config_byte(dev, 0x5d, 0x0a);
	pci_write_config_byte(dev, 0x5e, 0x0a);
	pci_write_config_byte(dev, 0x5f, 0x0a);
	pci_write_config_byte(dev, PCI_LATENCY_TIMER, 0x20);
	pci_write_config_word(dev, PCI_COMMAND,
	                           PCI_COMMAND_PARITY | PCI_COMMAND_MASTER |
				   PCI_COMMAND_IO);
}

#endif
