/*
 * arch/mips/vr41xx/nec-cmbvr41xx/irq.c
 *
 * Interrupt routines for the NEC CMB-VR4122/VR4131 board.
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/bitops.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>

#include <asm/io.h>
#include <asm/vr41xx/cmbvr41xx.h>

extern void enable_8259A_irq(unsigned int irq);
extern void disable_8259A_irq(unsigned int irq);
extern void mask_and_ack_8259A(unsigned int irq);
extern void init_8259A(int hoge);

static unsigned int startup_i8259_irq(unsigned int irq)
{
	enable_8259A_irq(irq - I8259_IRQ_BASE);

	return 0;
}

static void shutdown_i8259_irq(unsigned int irq)
{
	disable_8259A_irq(irq - I8259_IRQ_BASE);
}

static void enable_i8259_irq(unsigned int irq)
{
	enable_8259A_irq(irq - I8259_IRQ_BASE);
}

static void disable_i8259_irq(unsigned int irq)
{
	disable_8259A_irq(irq - I8259_IRQ_BASE);
}

static void ack_i8259_irq(unsigned int irq)
{
	mask_and_ack_8259A(irq - I8259_IRQ_BASE);
}

static void end_i8259_irq(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		enable_8259A_irq(irq - I8259_IRQ_BASE);
}

static struct hw_interrupt_type i8259_irq_type = {
	.typename	= "XT-PIC",
	.startup	= startup_i8259_irq,
	.shutdown	= shutdown_i8259_irq,
	.enable		= enable_i8259_irq,
	.disable	= disable_i8259_irq,
	.ack		= ack_i8259_irq,
	.end		= end_i8259_irq,
};

static int i8259_get_irq_number(int irq)
{
	unsigned long isr;

	isr = inb(0x20);
	irq = ffz(~isr);
	if (irq == 2) {
		isr = inb(0xa0);
		irq = 8 + ffz(~isr);
	}

	if (irq < 0 || irq > 15)
		return -EINVAL;

	return I8259_IRQ_BASE + irq;
}

static struct irqaction i8259_slave_cascade = {
	.handler	= &no_action,
	.name		= "cascade",
};

void __init cmbvr41xx_irq_init(void)
{
	int i;

	outb(0x11, 0x20);		/* Master ICW1 */
	outb(I8259_IRQ_BASE, 0x21);	/* Master ICW2 */
	outb(0x04, 0x21);		/* Master ICW3 */
	outb(0x01, 0x21);		/* Master ICW4 */
	outb(0xff, 0x21);		/* Master IMW */

	outb(0x11, 0xa0);		/* Slave ICW1 */
	outb(I8259_IRQ_BASE + 8, 0xa1);	/* Slave ICW2 */
	outb(0x02, 0xa1);		/* Slave ICW3 */
	outb(0x01, 0xa1);		/* Slave ICW4 */
	outb(0xff, 0xa1);		/* Slave IMW */

	outb(0x00, 0x4d0);
	outb(0x02, 0x4d1);	/* USB IRQ9 is level */

	for (i = I8259_IRQ_BASE; i <= I8259_IRQ_LAST; i++)
		irq_desc[i].handler = &i8259_irq_type;

	setup_irq(I8259_SLAVE_IRQ, &i8259_slave_cascade);

	vr41xx_set_irq_trigger(CMBVR41XX_INTC_PIN, TRIGGER_LEVEL, SIGNAL_THROUGH);
	vr41xx_set_irq_level(CMBVR41XX_INTC_PIN, LEVEL_HIGH);
	vr41xx_cascade_irq(CMBVR41XX_INTC_IRQ, i8259_get_irq_number);
}
