/*
 * arch/mips/vr41xx/nec-cmbvr4133/ricoh_rtc.c
 *
 * A simple Real Time Clock interface for Ricoh RV5C387A RTC chip
 * on NEC CMB-VR4133 boards.
 *
 * Author: Alex Sapkov <asapkov@ru.mvista.com> or <source@mvista.com>
 *
 * 2004 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

/*	This is an RTC driver interface for NEC CMB-VR4133 
 *	with Ricoh RV5C387A I2C RTC chip. RV5C387A on this board
 *	is not connected to any type of I2C controller. It is
 * 	just wired to GIU pins 6 & 35. So we do not use common Linux I2C 
 *	interface drivers to access it. This is highly platform 
 *	dependent but who cares: it cannot be used on any other boards
 *	anyway.
 *
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/delay.h>

#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>

#include <asm-mips/time.h>

/************************* I2C part *****************************/

/* GIU registers' addresses */
static volatile u16 *giu_iosell = (u16 *)0xaf000140; /* GIUIOSELL reg */
static volatile u16 *giu_piodl = (u16 *)0xaf000144; /* GIUPIODL reg */
static volatile u16 *giu_podat = (u16 *)0xaf00015e; /* GIUPODAT reg */

/* Vars to hold registers' values for a while. Can be global since
   the operations are protected by a spinlock anyway. */
static u16 piodl_val;
static u16 podat_val;

/* masks to set and clear GIU registers' bits */
#define GPIO6_1 0x40
#define GPIO6_0 0xffbf
#define GPO35_1 0x8
#define GPO35_0 0xfff7

#define SET_GPIO6(x) do { *giu_piodl = piodl_val | GPIO6_1; } while(0)
#define SET_GPO35(x) do { *giu_podat = podat_val | GPO35_1; } while(0)
#define CLEAR_GPIO6(x) do { *giu_piodl = piodl_val & GPIO6_0; } while(0)
#define CLEAR_GPO35(x) do { *giu_podat = podat_val & GPO35_0; } while(0)

#define GIU_DELAY 40 /* the hardware delay in microseconds */

/* spinlock to protect access to GIU registers */
static spinlock_t giu_lock;

static unsigned int bcdtobin(unsigned int bcd)
{
	return(((bcd >> 4) & 0x0f) * 10 + (bcd & 0x0f));
}

static unsigned int bintobcd(unsigned int bin)
{
	return(((bin / 10) << 4) + bin % 10);
}

static void i2c_write_bit(unsigned char bit)
{
	CLEAR_GPO35();

	if(bit)
		SET_GPIO6();    /* output 1 */
	else
		CLEAR_GPIO6();  /* output 0 */

	udelay(GIU_DELAY);

	SET_GPO35();
	
	udelay(GIU_DELAY);

	CLEAR_GPO35();
}

static unsigned char i2c_read_bit(void)
{
	volatile unsigned char bit;

	u16 iosell_val = *giu_iosell;

	CLEAR_GPO35();
	*giu_iosell = iosell_val & GPIO6_0; /* set GPIO6 to input */

	udelay(GIU_DELAY);

	SET_GPO35();
	
	udelay(GIU_DELAY);

        bit = (unsigned char)((*giu_piodl & GPIO6_1) >> 6); /* read the bit */
	CLEAR_GPO35();

	udelay(GIU_DELAY);

	return bit;
}

static void i2c_start(void)
{
	piodl_val = *giu_piodl;
	podat_val = *giu_podat;

	SET_GPIO6();
	SET_GPO35();

	udelay(GIU_DELAY);

	CLEAR_GPIO6();

	udelay(GIU_DELAY);

	CLEAR_GPO35();

	udelay(GIU_DELAY);

	/* send I2C slave address: 0x32 */
	i2c_write_bit(0);
	i2c_write_bit(1);
	i2c_write_bit(1);
	i2c_write_bit(0);
	i2c_write_bit(0);
	i2c_write_bit(1);
	i2c_write_bit(0);
	
}

static void i2c_stop(void)
{
	CLEAR_GPIO6();
	CLEAR_GPO35();

	udelay(GIU_DELAY);

	SET_GPO35();
	
	udelay(GIU_DELAY);

	SET_GPIO6();
}

static void chg_sda_2_output(void) 
{
	u16 iosell_val = *giu_iosell;

	*giu_iosell = iosell_val | GPIO6_1;
}

static unsigned char i2c_read_byte(void)
{
	volatile unsigned char val;

	val = (i2c_read_bit() << 7);
	val |= (i2c_read_bit() << 6);
	val |= (i2c_read_bit() << 5);
	val |= (i2c_read_bit() << 4);
	val |= (i2c_read_bit() << 3);
	val |= (i2c_read_bit() << 2);
	val |= (i2c_read_bit() << 1);
	val |= i2c_read_bit();
	chg_sda_2_output();
	return val;
}

static void i2c_write_byte(unsigned char val)
{
	unsigned char i;

	for(i = 0x80; i; ) {
		if(i & val)
			i2c_write_bit(1);
		else
			i2c_write_bit(0);
		i = i >> 1;
	}
}

unsigned long vr4133_ricoh_rtc_get_time(void)
{
	unsigned int year, mon, day, hour, min, sec, wday;

	spin_lock_irq(&giu_lock);

	i2c_start(); 			/* start i2c & write RTC addr */
	i2c_write_bit(0);		/* write 0 for write operation */

	/* read acknowledge */
	if(i2c_read_bit() != 0) {
		printk(KERN_ERR "vr4133_ricoh_rtc: I2C error!\n");
		chg_sda_2_output();
		i2c_stop();
		return 0;
	}

	chg_sda_2_output();
	/* internal addr point = 0 */
	i2c_write_bit(0);
	i2c_write_bit(0);
	i2c_write_bit(0);
	i2c_write_bit(0);
	/* 0x4 for transmission format */
	i2c_write_bit(0);
	i2c_write_bit(1);
	i2c_write_bit(0);
	i2c_write_bit(0);

	/* read acknowledge */
	if(i2c_read_bit() != 0) {
		printk(KERN_ERR "vr4133_ricoh_rtc: I2C error!\n");
		chg_sda_2_output();
		i2c_stop();
		return 0;
	}
	sec = (i2c_read_byte()) & 0x7f;
	i2c_write_bit(0);     /* write 0 to acknowledge */

	min = (i2c_read_byte()) & 0x7f;
	i2c_write_bit(0);     /* write 0 to acknowledge */

	hour = (i2c_read_byte()) & 0x3f;
	i2c_write_bit(0);     /* write 0 to acknowledge */

	wday = (i2c_read_byte()) & 0x7;
	i2c_write_bit(0);     /* write 0 to acknowledge */

	day = (i2c_read_byte()) & 0x3f;
	i2c_write_bit(0);     /* write 0 to acknowledge */

	mon = i2c_read_byte();
	i2c_write_bit(0);     /* write 0 to acknowledge */

	year = (i2c_read_byte()) & 0xff;

	sec=bcdtobin(sec);
	min=bcdtobin(min);
	hour=bcdtobin(hour);
	wday=bcdtobin(wday);
	day=bcdtobin(day);
	year=bcdtobin(year);

	i2c_write_bit(1);     /* write 1 before stop */
	i2c_stop();

	spin_unlock_irq(&giu_lock);


	if(hour == 12)
		hour = 0;
	else if(hour == 32)
		hour = 12;
	else if(hour >= 21)
		hour -= 8;

	if(mon & 0x80) 
		year += 100;  /* if RTC is counting from 2000 */

	mon = bcdtobin(mon & 0x1f);

	return  mktime(year + 1900, mon, day, hour, min, sec);
}

int vr4133_ricoh_rtc_set_time(unsigned long time)
{
	struct rtc_time tm;

	to_tm(time, &tm);

	if(tm.tm_hour == 0)
		tm.tm_hour = 12;
	else if(tm.tm_hour == 12)
		tm.tm_hour = 32;
	else if(tm.tm_hour >= 13)
		tm.tm_hour += 8;
	
	tm.tm_sec = bintobcd(tm.tm_sec);
	tm.tm_min = bintobcd(tm.tm_min);
	tm.tm_hour = bintobcd(tm.tm_hour);
	tm.tm_mday = bintobcd(tm.tm_mday);
	tm.tm_mon = bintobcd(tm.tm_mon + 1) | ((tm.tm_year / 100 == 20) ? 0x80 : 0);
	tm.tm_year = bintobcd(tm.tm_year % 100);
	tm.tm_wday = bintobcd(tm.tm_wday);

	spin_lock_irq(&giu_lock);
	
	i2c_start(); 		/* start i2c & write rtc address */
	i2c_write_bit(0);	/* write 0 for write operation */ 

	/* read acknowlegde */
	if(i2c_read_bit())
		goto error_stop_i2;

	chg_sda_2_output();

	i2c_write_bit(0);	/* internal addr point = 0 */
	i2c_write_bit(0);
	i2c_write_bit(0);
	i2c_write_bit(0);
	i2c_write_bit(0);	/* write transmission format */
	i2c_write_bit(0);
	i2c_write_bit(0);
	i2c_write_bit(0);

	/* read acknowlegde */
	if(i2c_read_bit())
		goto error_stop_i2;

	chg_sda_2_output();
	i2c_write_byte(tm.tm_sec);
	/* read acknowlegde */
	if(i2c_read_bit())
		goto error_stop_i2;


	chg_sda_2_output();
	i2c_write_byte(tm.tm_min);
	/* read acknowlegde */
	if(i2c_read_bit())
		goto error_stop_i2;

	chg_sda_2_output();
	i2c_write_byte(tm.tm_hour);
	/* read acknowlegde */
	if(i2c_read_bit())
		goto error_stop_i2;

	chg_sda_2_output();
	i2c_write_byte(tm.tm_wday);
	/* read acknowlegde */
	if(i2c_read_bit())
		goto error_stop_i2;

	chg_sda_2_output();
	i2c_write_byte(tm.tm_mday);
	/* read acknowlegde */
	if(i2c_read_bit())
		goto error_stop_i2;

	chg_sda_2_output();
	i2c_write_byte(tm.tm_mon);
	/* read acknowlegde */
	if(i2c_read_bit())
		goto error_stop_i2;

	chg_sda_2_output();
	i2c_write_byte(tm.tm_year);
	/* read acknowlegde */
	if(i2c_read_bit())
		goto error_stop_i2;
	
	chg_sda_2_output();
	i2c_stop();
	spin_unlock_irq(&giu_lock);

	return 0;

error_stop_i2:
	chg_sda_2_output();
	i2c_stop();
	spin_unlock_irq(&giu_lock);

	printk(KERN_ERR "vr4133_ricoh_rtc: I2C error!\n");
	return 1;
}

EXPORT_SYMBOL(vr4133_ricoh_rtc_get_time);
EXPORT_SYMBOL(vr4133_ricoh_rtc_set_time);

