/*
 * arch/mips/vr41xx/common/serial.c
 *
 * Serial Interface Unit routines for NEC VR4100 series.
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2002-2004 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
/*
 * Changes:
 *  MontaVista Software Inc. <yyuasa@mvista.com> or <source@mvista.com>
 *  - New creation, NEC VR4122 and VR4131 are supported.
 *  - Added support for NEC VR4111 and VR4121.
 *  - Added KGDB support for NEC VR4100 series.
 *
 *  MontaVista Software Inc. <asapkov@ru.mvista.com> or <source@mvista.com>
 *  - Added support for NEC VR4133
 */
#include <linux/init.h>
#include <linux/types.h>
#include <linux/serial.h>

#include <asm/addrspace.h>
#include <asm/cpu.h>
#include <asm/io.h>
#include <asm/vr41xx/vr41xx.h>

/* VR4111 and VR4121 SIU Registers */
#define VR4111_SIURB		KSEG1ADDR(0x0c000000)
#define VR4111_SIUIRSEL		KSEG1ADDR(0x0c000008)

/* VR4122 and VR4131/4133 SIU Registers */
#define VR4122_SIURB		KSEG1ADDR(0x0f000800)
#define VR4122_SIUIRSEL		KSEG1ADDR(0x0f000808)

 #define USE_RS232C		0x00
 #define USE_IRDA		0x01
 #define SIU_USES_IRDA		0x00
 #define FIR_USES_IRDA		0x02
 #define IRDA_MODULE_SHARP	0x00
 #define IRDA_MODULE_TEMIC	0x04
 #define IRDA_MODULE_HP		0x08
 #define TMICTX			0x10
 #define TMICMODE		0x20

 #define SIURB			0x00
 #define SIUTH			0x00
 #define SIUDLL			0x00
 #define SIUIE			0x01
 #define SIUDLM			0x01
 #define SIULC			0x03
  #define SELECT_DIVISOR	0x80
 #define SIULS			0x05
  #define TX_DATA_EMPUTY	0x20
  #define RX_DATA_READY		0x01

#define SIU_BASE_BAUD		1152000
#define SIU_CLOCK		0x0102

/* VR4122 and VR4131 DSIU Registers */
#define DSIURB			KSEG1ADDR(0x0f000820)

#define MDSIUINTREG		KSEG1ADDR(0x0f000096)
 #define INTDSIU		0x0800

#define DSIU_BASE_BAUD		1152000
#define DSIU_CLOCK		0x0802

int vr41xx_serial_ports = 0;

void vr41xx_siu_ifselect(int interface, int module)
{
	u16 val = USE_RS232C;	/* Select RS-232C */

	/* Select IrDA */
	if (interface == SIU_IRDA) {
		switch (module) {
		case IRDA_SHARP:
			val = IRDA_MODULE_SHARP;
			break;
		case IRDA_TEMIC:
			val = IRDA_MODULE_TEMIC;
			break;
		case IRDA_HP:
			val = IRDA_MODULE_HP;
			break;
		}
		val |= USE_IRDA | SIU_USES_IRDA;
	}

	switch (current_cpu_data.cputype) {
	case CPU_VR4111:
	case CPU_VR4121:
		writew(val, VR4111_SIUIRSEL);
		break;
	case CPU_VR4122:
	case CPU_VR4131:
	case CPU_VR4133:
		writew(val, VR4122_SIUIRSEL);
		break;
	default:
		printk(KERN_INFO "Unexpected CPU of NEC VR4100 series\n");
		break;
	}
}

void __init vr41xx_siu_init(int interface, int module)
{
	struct serial_struct s;

	vr41xx_siu_ifselect(interface, module);

	memset(&s, 0, sizeof(s));

	s.line = vr41xx_serial_ports;
	s.baud_base = SIU_BASE_BAUD;
	s.irq = SIU_IRQ;
	s.flags = ASYNC_BOOT_AUTOCONF | ASYNC_SKIP_TEST;
	switch (current_cpu_data.cputype) {
	case CPU_VR4111:
	case CPU_VR4121:
		s.iomem_base = (unsigned char *)VR4111_SIURB;
		break;
	case CPU_VR4122:
	case CPU_VR4131:
	case CPU_VR4133:
		s.iomem_base = (unsigned char *)VR4122_SIURB;
		break;
	default:
		panic("Unexpected CPU of NEC VR4100 series");
		break;
	}
	s.iomem_reg_shift = 0;
	s.io_type = SERIAL_IO_MEM;
	if (early_serial_setup(&s) != 0)
		printk(KERN_ERR "SIU setup failed!\n");

	vr41xx_clock_supply(SIU_CLOCK);

	vr41xx_serial_ports++;
}

void __init vr41xx_dsiu_init(void)
{
	struct serial_struct s;

	if (current_cpu_data.cputype != CPU_VR4122 &&
	    current_cpu_data.cputype != CPU_VR4131 && 
	    current_cpu_data.cputype != CPU_VR4133)
		return;

	memset(&s, 0, sizeof(s));

	s.line = vr41xx_serial_ports;
	s.baud_base = DSIU_BASE_BAUD;
	s.irq = DSIU_IRQ;
	s.flags = ASYNC_BOOT_AUTOCONF | ASYNC_SKIP_TEST;
	s.iomem_base = (unsigned char *)DSIURB;
	s.iomem_reg_shift = 0;
	s.io_type = SERIAL_IO_MEM;
	if (early_serial_setup(&s) != 0)
		printk(KERN_ERR "DSIU setup failed!\n");

	vr41xx_clock_supply(DSIU_CLOCK);

	writew(INTDSIU, MDSIUINTREG);

	vr41xx_serial_ports++;
}

#ifdef CONFIG_KGDB

static u32 siu_debug_base;

void __init
vr41xx_siu_debug_init(unsigned int divisor, unsigned char data,
                      unsigned char parity, unsigned char stop)
{
	vr41xx_clock_supply(SIU_CLOCK);

	switch (current_cpu_data.cputype) {
	case CPU_VR4111:
	case CPU_VR4121:
		siu_debug_base = VR4111_SIURB;
		break;
	case CPU_VR4122:
	case CPU_VR4131:
	case CPU_VR4133:
		siu_debug_base = VR4122_SIURB;
		break;
	default:
		panic("Unexpected CPU of NEC VR4100 series");
		break;
	}

	writeb(0, siu_debug_base + SIUIE);

	writeb(SELECT_DIVISOR, siu_debug_base + SIULC);

	writeb((u8) divisor, siu_debug_base + SIUDLL);
	writeb((u8) (divisor >> 8), siu_debug_base + SIUDLM);

	writeb(0, siu_debug_base + SIULC);

	writeb(data | parity | stop, siu_debug_base + SIULC);
}

void __init
vr41xx_dsiu_debug_init(unsigned int divisor, unsigned char data,
                       unsigned char parity, unsigned char stop)
{
	if (current_cpu_data.cputype != CPU_VR4122 &&
	    current_cpu_data.cputype != CPU_VR4131 &&
	    current_cpu_data.cputype != CPU_VR4133)
		return;

	vr41xx_clock_supply(DSIU_CLOCK);

	if (current_cpu_data.cputype != CPU_VR4133)
		siu_debug_base = DSIURB;
	else { 
		/* for vr 4133 we use UART0 for kdbg output. UART1 seems 
		to be not connected on this board. */
		siu_debug_base = VR4122_SIURB;
		return;
	}

	writeb(0, siu_debug_base + SIUIE);

	writeb(SELECT_DIVISOR, siu_debug_base + SIULC);

	writeb((u8) divisor, siu_debug_base + SIUDLL);
	writeb((u8) (divisor >> 8), siu_debug_base + SIUDLM);

	writeb(0, siu_debug_base + SIULC);

	writeb(data | parity | stop, siu_debug_base + SIULC);
}

char
getDebugChar(void)
{
	while ((readb(siu_debug_base + SIULS) & RX_DATA_READY) == 0) ;

	return readb(siu_debug_base + SIURB);
}

int
putDebugChar(char c)
{
	while ((readb(siu_debug_base + SIULS) & TX_DATA_EMPUTY) == 0) ;

	writeb(c, siu_debug_base + SIUTH);

	return 1;
}
#endif
