/*
 * arch/mips/vr4181a/nec-smvr4181a/setup.c
 *
 * Setup for the NEC SMVR4181A board.
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2002-2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/console.h>
#include <linux/ide.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/string.h>

#include <asm/bootinfo.h>
#include <asm/pci_channel.h>
#include <asm/processor.h>
#include <asm/reboot.h>
#include <asm/time.h>
#include <asm/vr4181a/vr4181a.h>
#include <asm/vr4181a/smvr4181a.h>

#ifdef CONFIG_BLK_DEV_INITRD
extern unsigned long initrd_start, initrd_end;
extern void *__rd_start, *__rd_end;
#endif

#ifdef CONFIG_MIPS_RTC
extern void smvr4181a_time_init(void);
#endif

static struct vr4181a_address_maps smvr4181a_address_maps __initdata = {
	.pcs3 = {
		 .base_address = EXTERNAL_PCS3_BASE,
		 .size = EXTERNAL_PCS3_SIZE,
		 .data_width = DATA_WIDTH_16BIT,
		 .pci_master_access = PCI_MASTER_ACCESS_PROHIBIT,
		 },
	.isa_bus = {
		    .base_address = EXTERNAL_ISA_MEMORY_BASE,
		    .size = EXTERNAL_ISA_MEMORY_SIZE + EXTERNAL_ISA_IO_SIZE,
		    .data_width = DATA_WIDTH_16BIT,
		    .pci_master_access = PCI_MASTER_ACCESS_PROHIBIT,
		    },
#ifdef CONFIG_PCI
	.pci_bus0 = {
		     .base_address = INTERNAL_PCI_WINDOW0_BASE,
		     .size = INTERNAL_PCI_WINDOW0_SIZE,
		     .data_width = DATA_WIDTH_32BIT,
		     .pci_master_access = PCI_MASTER_ACCESS_PROHIBIT,
		     },
	.pci_bus1 = {
		     .base_address = INTERNAL_PCI_WINDOW1_BASE,
		     .size = INTERNAL_PCI_WINDOW1_SIZE,
		     .data_width = DATA_WIDTH_32BIT,
		     .pci_master_access = PCI_MASTER_ACCESS_PROHIBIT,
		     },
#endif
	.rom = {
		.base_address = ROM_BASE,
		.size = ROM_SIZE,
		.data_width = DATA_WIDTH_16BIT,
		.pci_master_access = PCI_MASTER_ACCESS_PROHIBIT,
		},
};

static struct vr4181a_access_setup smvr4181a_access_setup __initdata = {
	.pcs3 = {
		 .page_access_length = 4,
		 .control_signal_active_level = ACTIVE_LOW,
		 .chip_select_active_level = ACTIVE_LOW,
		 .control_signal_deassert_timing = 1,
		 .iordy_sync_times = LATCH_2_TIMES,
		 .ready_mode = READY_USE,
		 .bus_idel_time = 0,
		 .control_signal_width_min = 2,
		 .control_signal_hold_time = 1,
		 .ready_signal_timeout_counter = 256,
		 .control_signal_setup_time = 1,
		 .config_active = 1,
		 },
	.isa_bus = {
		    .io_signal_setup_time = 4,
		    .page_access_length = 4,
		    .control_signal_deassert_timing = 1,
		    .iordy_sync_times = LATCH_2_TIMES,
		    .ready_mode = READY_USE,
		    .bus_idel_time = 2,
		    .control_signal_width_min = 8,
		    .control_signal_hold_time = 1,
		    .ready_signal_timeout_counter = 256,
		    .control_signal_setup_time = 1,
		    .config_active = 1,
		    },
	.rom = {
		.page_access_length = 4,
		.control_signal_active_level = ACTIVE_LOW,
		.chip_select_active_level = ACTIVE_LOW,
		.control_signal_deassert_timing = 0,
		.ready_mode = READY_UNUSE,
		.bus_idel_time = 0,
		.control_signal_hold_time = 0,
		.control_signal_assert_time_page_access = 2,
		.control_signal_assert_time = 6,
		.control_signal_setup_time = 0,
		.config_active = 1,
		},
};

#ifdef CONFIG_PCI

static struct vr4181a_pci_config smvr4181a_pci_config __initdata = {
	.dram = {
		 .pci_base_address = DRAM_BASE,
		 .prefetch = PREFETCH_ENABLE,
		 .address_active = 1,
		 },
	.pci_window0 = {
			.pci_base_address = INTERNAL_PCI_WINDOW0_BASE,
			.access_width = ACCESS_WIDTH_32BIT,
			.command_type = COMMAND_TYPE_MEMORY,
			},
	.pci_window1 = {
			.pci_base_address = INTERNAL_PCI_WINDOW1_BASE,
			.access_width = ACCESS_WIDTH_32BIT,
			.command_type = COMMAND_TYPE_IO,
			},
	.master_latency = 128,
	.delayed_transaction_abort = 32768,
	.retry_limit = 256,
};

static struct resource smvr4181a_pci_io_resource = {
	.name = "PCI I/O space",
	.start = PCI_IO_RESOURCE_START,
	.end = PCI_IO_RESOURCE_END,
	.flags = IORESOURCE_IO,
};

static struct resource smvr4181a_pci_mem_resource = {
	.name = "PCI memory space",
	.start = PCI_MEM_RESOURCE_START,
	.end = PCI_MEM_RESOURCE_END,
	.flags = IORESOURCE_MEM,
};

struct pci_channel mips_pci_channels[2] = {
	{
	 .pci_ops = &vr4181a_pci_ops,
	 .io_resource = &smvr4181a_pci_io_resource,
	 .mem_resource = &smvr4181a_pci_mem_resource,
	 .first_devfn = 0,
	 .last_devfn = 256,
	 },
};

#endif

const char *
get_system_type(void)
{
	return "NEC SMVR4181A";
}

unsigned long __init
vr4181a_platform_init(int argc, char **argv, char **envp,
		      int *prom_vec, char *arcs_cmdline)
{
	add_memory_region(0, 64 << 20, BOOT_MEM_RAM);

	return MACH_NEC_SMVR4181A;
}

void __init
vr4181a_platform_setup(void)
{
	int ecu_mode;

	set_vr4181a_internal_registers_base(INTERNAL_REGISTERS_BASE);

	set_io_port_base(IO_PORT_BASE);
	ioport_resource.start = IO_PORT_RESOURCE_START;
	ioport_resource.end = IO_PORT_RESOURCE_END;
	iomem_resource.start = IO_MEM_RESOURCE_START;
	iomem_resource.end = IO_MEM_RESOURCE_END;

#ifdef CONFIG_BLK_DEV_INITRD
	ROOT_DEV = MKDEV(RAMDISK_MAJOR, 0);
	initrd_start = (unsigned long) &__rd_start;
	initrd_end = (unsigned long) &__rd_end;
#endif

	cpu_wait = vr4181a_wait;

	_machine_restart = vr4181a_restart;
	_machine_halt = vr4181a_halt;
	_machine_power_off = vr4181a_power_off;

#ifdef CONFIG_MIPS_RTC
	board_time_init = smvr4181a_time_init;
#endif
	board_timer_setup = vr4181a_timer_setup;

#ifdef CONFIG_FB
	conswitchp = &dummy_con;
#endif

	/*
	 * Power up of Peripheral I/O device
	 * (This setup has already been performed by bootstrap code).
	 */
	vr4181a_set_gpio_mode(45, GPIO_OUTPUT, GPIO_INPUT_PROHIBIT);
	vr4181a_set_gpio_data(45, 1);

	vr4181a_ccu_init(ECU_SYSCLOCK_MAX);

	vr4181a_hbu_init(&smvr4181a_address_maps);

	vr4181a_exibu_init(&smvr4181a_access_setup, LCLOCK_MAX);

	/*
	 * The power of USB is saved when not using it.
	 */
	vr4181a_power_save_usb_host();
	vr4181a_power_save_usb_function();

#if defined(CONFIG_USB_OHCI) || defined(CONFIG_USB_OHCI_MODULE)
	/*
	 * Enable CLK48 pin
	 */
	vr4181a_set_pinmode2(PINMODE2_USB_ENABLE);

	/*
	 * Enable clock supply to USB(48MHz).
	 */
	vr4181a_set_gpio_mode(43, GPIO_OUTPUT, GPIO_INPUT_PROHIBIT);
	vr4181a_set_gpio_data(43, 1);

	vr4181a_clock_supply(USB_HOST_PCI_CLOCK);
	vr4181a_clock_supply(USB_HOST_CLOCK);

	vr4181a_power_active_usb_host();
#endif

#ifdef CONFIG_PCI
	vr4181a_iopciu_init(&smvr4181a_pci_config);
#endif

#ifdef CONFIG_SERIAL
	vr4181a_set_gpio_mode(7, GPIO_INPUT, GPIO_INPUT_ENABLE);	/* RI0# */
	vr4181a_siu_init(0, 0);
#endif

#if defined(CONFIG_SERIAL) && !defined(CONFIG_KGDB)
	vr4181a_set_gpio_mode(6, GPIO_INPUT, GPIO_INPUT_ENABLE);	/* RI2# */
	vr4181a_siu_init(2, 1);
#endif

#ifdef CONFIG_KGDB
	vr4181a_siu_debug_init(2, BAUD_RATE_38400, DATA_8BIT, PARITY_NONE,
			       STOP_1BIT);
#endif

	/*
	 * GPIO3 is used for IRQ from SMSC LAN91C113I ch1.
	 */
	vr4181a_set_gpio_mode(3, GPIO_INPUT, GPIO_INPUT_ENABLE);

	/*
	 * GPIO4 is used for IRQ from SMSC LAN91C113I ch2.
	 */
	vr4181a_set_gpio_mode(4, GPIO_INPUT, GPIO_INPUT_ENABLE);

	/*
	 * GPIO12 is used for IRQ from debug board.
	 */
	vr4181a_set_gpio_mode(12, GPIO_INPUT, GPIO_INPUT_ENABLE);

	/*
	 * ECU is set default as CompactFlash Interface.
	 */
	ecu_mode = ECU_MODE_CF;

	vr4181a_set_gpio_mode(37, GPIO_INPUT, GPIO_INPUT_ENABLE);
	if (vr4181a_get_gpio_data(37) == 1) {
#ifdef CONFIG_BLK_DEV_IDE
		vr4181a_set_gpio_mode(24, GPIO_OUTPUT, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_data(24, 0);

		vr4181a_set_gpio_mode(35, GPIO_INPUT, GPIO_INPUT_ENABLE);
		vr4181a_set_gpio_mode(36, GPIO_INPUT, GPIO_INPUT_ENABLE);

		ecu_mode = ECU_MODE_IDE;
#endif
	}
	vr4181a_ecu_init(ecu_mode);
}
