/*
 * FILE NAME
 *	arch/mips/vr4181a/nec-smvr4181a/rtc_4543.c
 *
 * BRIEF MODULE DESCRIPTION
 *	low-level RTC hookups for EPSON RTC-4543SA/SB chip.
 *
 * Copyright 2002 NEC Micro Systems, Ltd.
 * Author: Takahiro Miwa
 *         t-miwa@pi.jp.nec.com
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 *  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 *  OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 *  TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 *  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <linux/rtc.h>

#include <asm/time.h>
#include <asm/vr4181a/vr4181a.h>

#define CSIMODE			0xb900
#define CSIRXDATA		0xb902
#define CSITXDATA		0xb904
#define CSILSTAT		0xb906
#define CSIINTMSK		0xb908
#define CSIINTSTAT		0xb90a
#define CSITXBLEN		0xb90c
#define CSIRXBLEN		0xb90e
#define CSICLKSEL		0xb910
#define CSICLK_0_144MHZ	0x0007
#define CSICLK_0_288MHZ	0x0006
#define CSICLK_0_576MHZ	0x0005
#define CSICLK_1_152MHZ	0x0004
#define CSICLK_2_304MHZ	0x0003
#define CSICLK_4_608MHZ	0x0002

#define RTC_4543_EPOCH		2000
#define READ_WRITE_PIN		62
#define CHIP_SELECT_PIN		63
#define RTC_4543_READ		0
#define RTC_4543_WRITE		1
#define RTC_4543_CS_DISABLE	0
#define RTC_4543_CS_ENABLE	1

#ifndef BCD_TO_BIN
#define BCD_TO_BIN(val)	(((val)&15) + ((val)>>4)*10)
#endif
#ifndef BIN_TO_BCD
#define BIN_TO_BCD(val)	((((val)/10)<<4) + (val)%10)
#endif

static unsigned long
rtc_4543_get_time(void)
{
	int i;
	unsigned int year, month, day, hour, minute, second;
	unsigned short rdata[4];

	vr4181a_set_gpio_data(READ_WRITE_PIN, RTC_4543_READ);
	vr4181a_set_gpio_data(CHIP_SELECT_PIN, RTC_4543_CS_ENABLE);

	/* Clear the data FIFO */
	vr4181a_writew(0x3331, CSIMODE);
	vr4181a_write_fixed;

	/* recieve mode */
	vr4181a_writew(0x3631, CSIMODE);
	vr4181a_write_fixed;
	vr4181a_writew(0x00ff, CSIINTSTAT);
	vr4181a_write_fixed;

	/* read data */
	rdata[0] = vr4181a_readw(CSIRXDATA);	/* Kick the CSI */
	while (!(vr4181a_readw(CSIINTSTAT) & 0x0004)) {
	}
	for (i = 0; i < 4; i++) {
		rdata[i] = vr4181a_readw(CSIRXDATA);
	}

	vr4181a_set_gpio_data(CHIP_SELECT_PIN, RTC_4543_CS_DISABLE);

	/* read time data */
	second = BCD_TO_BIN(rdata[0] & 0x7f);
	minute = BCD_TO_BIN((rdata[0] >> 8) & 0x7f);
	hour = BCD_TO_BIN(rdata[1] & 0x3f);
	day = BCD_TO_BIN(((rdata[2] << 4) | (rdata[1] >> 12)) & 0x3f);
	month = BCD_TO_BIN((rdata[2] >> 4) & 0x1f);
	year = BCD_TO_BIN(((rdata[3] << 4) | (rdata[2] >> 12)) & 0xff);
	year += RTC_4543_EPOCH;

	return mktime(year, month, day, hour, minute, second);
}

static int
rtc_4543_set_time(unsigned long t)
{
	int i;
	struct rtc_time tm;
	unsigned int year, month, week, day, hour, minute, second;
	unsigned short rdata[4];

	/* RTC4543 write enable */
	vr4181a_set_gpio_data(CHIP_SELECT_PIN, RTC_4543_CS_DISABLE);
	vr4181a_set_gpio_data(READ_WRITE_PIN, RTC_4543_WRITE);

	/* clear the FIFO */
	vr4181a_writew(0x3311, CSIMODE);
	vr4181a_write_fixed;

	vr4181a_set_gpio_data(CHIP_SELECT_PIN, RTC_4543_CS_ENABLE);

	/* convert */
	to_tm(t, &tm);

	second = BIN_TO_BCD(tm.tm_sec);
	minute = BIN_TO_BCD(tm.tm_min);
	hour = BIN_TO_BCD(tm.tm_hour);
	week = BIN_TO_BCD(tm.tm_wday);
	day = BIN_TO_BCD(tm.tm_mday);
	month = BIN_TO_BCD(tm.tm_mon + 1);	// tm_mon is from 0 to 11
	year = BIN_TO_BCD(tm.tm_year - RTC_4543_EPOCH);

	rdata[0] = (minute << 8) | second;
	rdata[1] = ((day & 0x0f) << 12) | (week << 8) | hour;
	rdata[2] = ((year & 0x0f) << 12) | (month << 4) | (day >> 4);
	rdata[3] = year >> 4;

	/* send data */
	vr4181a_writew(0x2311, CSIMODE);
	vr4181a_write_fixed;
	vr4181a_writew(0x6311, CSIMODE);
	vr4181a_write_fixed;
	for (i = 0; i < 4; i++) {
		vr4181a_writew(rdata[i], CSITXDATA);
		vr4181a_write_fixed;
		if (i == 0) {
			vr4181a_writew(0x0f0f, CSIINTSTAT);
			vr4181a_write_fixed;
		}
	}

	while (!(vr4181a_readw(CSIINTSTAT) & 0x0400)) ;

	/* RTC4543 Disable */
	vr4181a_set_gpio_data(READ_WRITE_PIN, RTC_4543_READ);
	vr4181a_set_gpio_data(CHIP_SELECT_PIN, RTC_4543_CS_DISABLE);

	return 0;
}

void
smvr4181a_time_init(void)
{
	/* CSI and RTC setup */
	vr4181a_clock_supply(CSU_CLOCK);	/* enable CSI clock */

	vr4181a_set_gpio_mode(READ_WRITE_PIN, GPIO_OUTPUT, GPIO_INPUT_PROHIBIT);
	vr4181a_set_gpio_mode(CHIP_SELECT_PIN, GPIO_OUTPUT,
			      GPIO_INPUT_PROHIBIT);

	vr4181a_set_gpio_data(READ_WRITE_PIN, RTC_4543_READ);
	vr4181a_set_gpio_data(CHIP_SELECT_PIN, RTC_4543_CS_DISABLE);

	vr4181a_writew(0, CSIINTMSK);
	vr4181a_write_fixed;
	vr4181a_writew(52, CSITXBLEN);	/* transmit data count */
	vr4181a_write_fixed;
	vr4181a_writew(64, CSIRXBLEN);	/* recieve data count */
	vr4181a_write_fixed;
	vr4181a_writew(CSICLK_0_288MHZ, CSICLKSEL);
	vr4181a_write_fixed;
	vr4181a_writew(0x8080, CSILSTAT);	/* FIFO length = 8 */
	vr4181a_write_fixed;

	/* set the function pointers */
	rtc_get_time = rtc_4543_get_time;
	rtc_set_time = rtc_4543_set_time;
}
