/*
 * arch/mips/vr4181a/common/siu.c
 *
 * Serial Interface Unit routines for NEC VR4181A.
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2002-2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/serial.h>
#include <linux/string.h>
#include <linux/types.h>

#include <asm/vr4181a/vr4181a.h>

#define SIU_UNITS	3

#define SIU0_IRQ	20
#define SIU1_IRQ	21
#define SIU2_IRQ	22

#define SIURB_0		0xc040
#define SIURB_1		0xc010
#define SIURB_2		0xc000

#define SIURB		0x00
#define SIUTH		0x00
#define SIUDLL		0x00
#define SIUIE		0x01
#define SIUDLM		0x01
#define SIULC		0x03
#define SELECT_DIVISOR	0x80
#define SIULS		0x05
#define TX_DATA_EMPUTY	0x20
#define RX_DATA_READY	0x01

#define SIU_BASE_BAUD	1152000

static int siu_irq[SIU_UNITS] __initdata = {
	SIU0_IRQ,
	SIU1_IRQ,
	SIU2_IRQ,
};

static unsigned short siu_base_offset[SIU_UNITS] __initdata = {
	SIURB_0,
	SIURB_1,
	SIURB_2,
};

static unsigned short siu_clock[SIU_UNITS] __initdata = {
	SIU0_CLOCK,
	SIU1_CLOCK,
	SIU2_CLOCK,
};

void __init
vr4181a_siu_init(unsigned char unit, int line)
{
	struct serial_struct s;

	if (unit > 2)
		return;

	vr4181a_clock_supply(siu_clock[unit]);

	switch (unit) {
	case 0:
		vr4181a_set_gpio_mode(16, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(18, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(19, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_pinmode0(PINMODE0_SIU0);
		break;
	case 1:
		vr4181a_set_pinmode0(PINMODE0_SIU1);
		vr4181a_set_pinmode1(PINMODE1_SIU1);
		break;
	case 2:
		vr4181a_set_pinmode2(PINMODE2_SIU2);
		break;
	}

	memset(&s, 0, sizeof (s));

	s.type = PORT_16550A;
	s.line = line;
	s.irq = siu_irq[unit];
	s.flags = ASYNC_SKIP_TEST;
	s.baud_base = SIU_BASE_BAUD;
	s.io_type = SERIAL_IO_MEM;
	s.iomem_base =
	    (unsigned char *) (vr4181a_internal_registers_base +
			       siu_base_offset[unit]);
	s.iomem_reg_shift = 0;
	if (early_serial_setup(&s) != 0)
		printk(KERN_ERR "SIU%d setup failed!\n", unit);
}

#ifdef CONFIG_KGDB

unsigned short siu_debug_base;

void
vr4181a_siu_debug_init(unsigned char unit, unsigned int divisor,
		       unsigned char data, unsigned char parity,
		       unsigned char stop)
{
	if (unit > 2)
		return;

	vr4181a_clock_supply(siu_clock[unit]);

	switch (unit) {
	case 0:
		vr4181a_set_gpio_mode(16, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(18, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(19, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_pinmode0(PINMODE0_SIU0);
		break;
	case 1:
		vr4181a_set_pinmode0(PINMODE0_SIU1);
		vr4181a_set_pinmode1(PINMODE1_SIU1);
		break;
	case 2:
		vr4181a_set_pinmode2(PINMODE2_SIU2);
		break;
	}

	siu_debug_base = siu_base_offset[unit];

	vr4181a_writeb(0, siu_debug_base + SIUIE);
	vr4181a_write_fixed;

	vr4181a_writeb(SELECT_DIVISOR, siu_debug_base + SIULC);
	vr4181a_write_fixed;

	vr4181a_writeb((uint8_t) divisor, siu_debug_base + SIUDLL);
	vr4181a_write_fixed;
	vr4181a_writeb((uint8_t) (divisor >> 8), siu_debug_base + SIUDLM);
	vr4181a_write_fixed;

	vr4181a_writeb(0, siu_debug_base + SIULC);
	vr4181a_write_fixed;

	vr4181a_writeb(data | parity | stop, siu_debug_base + SIULC);
	vr4181a_write_fixed;
}

char
getDebugChar(void)
{
	while ((vr4181a_readb(siu_debug_base + SIULS) & RX_DATA_READY) == 0) ;

	return vr4181a_readb(siu_debug_base + SIURB);
}

int
putDebugChar(char c)
{
	while ((vr4181a_readb(siu_debug_base + SIULS) & TX_DATA_EMPUTY) == 0) ;

	vr4181a_writeb(c, siu_debug_base + SIUTH);
	vr4181a_write_fixed;

	return 1;
}
#endif
