/*
 * arch/mips/vr4181a/common/iopciu.c
 *
 * Internal PCI Bridge Unit routines for the NEC VR4181A.
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2002-2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/types.h>

#include <asm/addrspace.h>
#include <asm/vr4181a/vr4181a.h>

#define IOPCIU_BASE		0x0200
#define PCICMD			0x04
#define SERREN			0x0100
#define PEREN			0x0040
#define PCISTS			0x06
#define DPE			0x8000
#define SSE			0x4000
#define RMA			0x2000
#define RTA			0x1000
#define STA			0x0800
#define DPR			0x0100
#define BAR_INTCS		0x10
#define BAR_SDRAM		0x18
#define BAR_SFLASH		0x20
#define BAR_PCS0		0x40
#define BAR_PCS1		0x48
#define BAR_PCS2		0x50
#define BAR_PCS3		0x58
#define BAR_PCS4		0x60
#define BAR_ISAW		0x68
#define BAR_ROMCS		0x80
#define PCIERR			0xb8
#define PCIERR_ADDR_MASK	0xfffffffc
#define IS_CPU			0x00000001
#define PCICTRL_L		0xe0
#define PCICTRL_H		0xe4
#define PCICRST		0x80000000
#define PCIWRST		0x40000000
#define IBERIN			0x02000000
#define IBERSE			0x01000000
#define IBERCH			0x00800000
#define AERIN			0x00200000
#define DTIMIN			0x00100000
#define PERIN			0x00080000
#define RTYIN			0x00040000
#define MAIN			0x00020000
#define TAIN			0x00010000
#define AERSE			0x00002000
#define DTIMSE			0x00001000
#define PERSE			0x00000800
#define RTYSE			0x00000400
#define MASE			0x00000200
#define TASE			0x00000100
#define ERRTYPE_MASK		0x000000e0
#define DTIMCH			0x00000010
#define PERCH			0x00000008
#define RTYCH			0x00000004
#define MACH			0x00000002
#define TACH			0x00000001
#define PCIARB			0xe8
#define ROUNDROBIN		0x00000000
#define PCIINIT01		0xf0
#define PCIINIT11		0xf8
#define PCI_BASE_ADDRESS	0xffe00000
#define CONFIGTYPE_0		0x00000000
#define CONFIGTYPE_1		0x00000200
#define ACCESS_32		0x00000010
#define TYPE_CONFIGURATION	0x0000000a
#define TYPE_MEMORY		0x00000006
#define TYPE_IO		0x00000002

#define iopciu_readb(offset)		vr4181a_readb(IOPCIU_BASE + (offset))
#define iopciu_readw(offset)		vr4181a_readw(IOPCIU_BASE + (offset))
#define iopciu_readl(offset)		vr4181a_readl(IOPCIU_BASE + (offset))

#define iopciu_writeb(val, offset)	vr4181a_writeb(val, IOPCIU_BASE + (offset))
#define iopciu_writew(val, offset)	vr4181a_writew(val, IOPCIU_BASE + (offset))
#define iopciu_writel(val, offset)	vr4181a_writel(val, IOPCIU_BASE + (offset))

static inline u32
iopci_read_config_dword(unsigned char bus, unsigned char slot, int where)
{
	u32 pciinit, val;

	pciinit = iopciu_readl(PCIINIT01);
	iopciu_writel(1UL << slot | CONFIGTYPE_0 | ACCESS_32 |
		      TYPE_CONFIGURATION, PCIINIT01);

	val = readl(KSEG1ADDR((pciinit & 0xffe00000) + (where & 0xfc)));

	iopciu_writel(pciinit, PCIINIT01);

	return val;
}

static inline void
iopci_write_config_dword(unsigned char bus, unsigned char slot,
			 int where, u32 val)
{
	u32 pciinit;

	pciinit = iopciu_readl(PCIINIT01);
	iopciu_writel(1UL << slot | CONFIGTYPE_0 | ACCESS_32 |
		      TYPE_CONFIGURATION, PCIINIT01);

	writel(val, KSEG1ADDR((pciinit & 0xffe00000) + (where & 0xfc)));

	iopciu_writel(pciinit, PCIINIT01);
}

static int
vr4181a_pci_config_read_byte(struct pci_dev *dev, int where, u8 * val)
{
	unsigned char bus = dev->bus->number;
	unsigned char slot = PCI_SLOT(dev->devfn);
	u32 data;

	*val = 0xff;

	if (bus != 0 || slot < 29 || slot > 31 || where > 255)
		return PCIBIOS_DEVICE_NOT_FOUND;

	data = iopci_read_config_dword(bus, slot, where);

	*val = (u8) (data >> ((where & 3) << 3));

	return PCIBIOS_SUCCESSFUL;

}

static int
vr4181a_pci_config_read_word(struct pci_dev *dev, int where, u16 * val)
{
	unsigned char bus = dev->bus->number;
	unsigned char slot = PCI_SLOT(dev->devfn);
	u32 data;

	*val = 0xffff;

	if (bus != 0 || slot < 29 || slot > 31 || where > 255)
		return PCIBIOS_DEVICE_NOT_FOUND;

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	data = iopci_read_config_dword(bus, slot, where);

	*val = (u16) (data >> ((where & 2) << 3));

	return PCIBIOS_SUCCESSFUL;
}

static int
vr4181a_pci_config_read_dword(struct pci_dev *dev, int where, u32 * val)
{
	unsigned char bus = dev->bus->number;
	unsigned char slot = PCI_SLOT(dev->devfn);

	*val = 0xffffffffUL;

	if (bus != 0 || slot < 29 || slot > 31 || where > 255)
		return PCIBIOS_DEVICE_NOT_FOUND;

	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	*val = iopci_read_config_dword(bus, slot, where);

	return PCIBIOS_SUCCESSFUL;
}

static int
vr4181a_pci_config_write_byte(struct pci_dev *dev, int where, u8 val)
{
	unsigned char bus = dev->bus->number;
	unsigned char slot = PCI_SLOT(dev->devfn);
	u32 data;
	int shift;

	if (bus != 0 || slot < 29 || slot > 31 || where > 255)
		return PCIBIOS_DEVICE_NOT_FOUND;

	data = iopci_read_config_dword(bus, slot, where);

	shift = (where & 3) << 3;
	data &= ~(0xff << shift);
	data |= (u32) val << shift;

	iopci_write_config_dword(bus, slot, where, data);

	return PCIBIOS_SUCCESSFUL;
}

static int
vr4181a_pci_config_write_word(struct pci_dev *dev, int where, u16 val)
{
	unsigned char bus = dev->bus->number;
	unsigned char slot = PCI_SLOT(dev->devfn);
	u32 data;
	int shift;

	if (bus != 0 || slot < 29 || slot > 31 || where > 255)
		return PCIBIOS_DEVICE_NOT_FOUND;

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	data = iopci_read_config_dword(bus, slot, where);

	shift = (where & 2) << 3;
	data &= ~(0xffff << shift);
	data |= (u32) val << shift;

	iopci_write_config_dword(bus, slot, where, data);

	return PCIBIOS_SUCCESSFUL;
}

static int
vr4181a_pci_config_write_dword(struct pci_dev *dev, int where, u32 val)
{
	unsigned char bus = dev->bus->number;
	unsigned char slot = PCI_SLOT(dev->devfn);

	if (bus != 0 || slot < 29 || slot > 31 || where > 255)
		return PCIBIOS_DEVICE_NOT_FOUND;

	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	iopci_write_config_dword(bus, slot, where, val);

	return PCIBIOS_SUCCESSFUL;
}

struct pci_ops vr4181a_pci_ops = {
	.read_byte = vr4181a_pci_config_read_byte,
	.read_word = vr4181a_pci_config_read_word,
	.read_dword = vr4181a_pci_config_read_dword,
	.write_byte = vr4181a_pci_config_write_byte,
	.write_word = vr4181a_pci_config_write_word,
	.write_dword = vr4181a_pci_config_write_dword,
};

static inline void
write_bar_value(struct vr4181a_pci_base_address *base, u16 offset)
{
	u32 bar;

	if (base->address_active) {
		bar = base->pci_base_address;

		if (base->prefetch == PREFETCH_ENABLE) ;
		bar |= 0x0008;

		iopciu_writel(bar, offset);
	}
}

static inline void
write_pciinit_value(struct vr4181a_pci_window_config *config, u16 offset)
{
	u32 pciinit;

	pciinit = config->pci_base_address & 0xffe00000;

	if (config->access_width == ACCESS_WIDTH_32BIT)
		pciinit |= ACCESS_32;

	switch (config->command_type) {
	case COMMAND_TYPE_IO:
		pciinit |= TYPE_IO;
		break;
	case COMMAND_TYPE_MEMORY:
		pciinit |= TYPE_MEMORY;
		break;
	case COMMAND_TYPE_CONFIGURATION:
		pciinit |= TYPE_CONFIGURATION;
		break;
	}

	iopciu_writel(pciinit, offset);
}

void __init
vr4181a_iopciu_init(struct vr4181a_pci_config *config)
{
	vr4181a_clock_supply(PCI_BRIDGE_CLOCK);

	iopciu_writel(PCICRST | PCIWRST, PCICTRL_H);

	/* dummy read */
	vr4181a_read_nvreg(0);

	iopciu_writel(0, PCICTRL_H);

	iopciu_writeb(config->master_latency, PCI_LATENCY_TIMER);

	write_bar_value(&config->internal_registers, BAR_INTCS);
	write_bar_value(&config->dram, BAR_SDRAM);
	write_bar_value(&config->sync_flash_memory, BAR_SFLASH);
	write_bar_value(&config->pcs0, BAR_PCS0);
	write_bar_value(&config->pcs1, BAR_PCS1);
	write_bar_value(&config->pcs2, BAR_PCS2);
	write_bar_value(&config->pcs3, BAR_PCS3);
	write_bar_value(&config->pcs4, BAR_PCS4);
	write_bar_value(&config->isa_bus, BAR_ISAW);
	write_bar_value(&config->rom, BAR_ROMCS);

	iopciu_writel((u32) (config->delayed_transaction_abort >> 8) << 24 |
		      (u32) (config->retry_limit >> 8) << 16, PCICTRL_L);

	write_pciinit_value(&config->pci_window0, PCIINIT01);
	write_pciinit_value(&config->pci_window1, PCIINIT11);

	iopciu_writew(SERREN | PEREN, PCICMD);
}
