/*
 * arch/mips/vr4181a/common/icu.c
 *
 * Interrupt Control Unit routines for the NEC VR4181A.
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2002-2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/types.h>

#include <asm/gdb-stub.h>
#include <asm/vr4181a/vr4181a.h>

#define MIPS_CPU_IRQ_BASE	0
#define SYSINT0_IRQ_BASE	8
#define SYSINT0_IRQ_LAST	23
#define SYSINT1_IRQ_BASE	24
#define SYSINT1_IRQ_LAST	39
#define SYSINT2_IRQ_BASE	40
#define SYSINT2_IRQ_LAST	55
#define SYSINT3_IRQ_BASE	56
#define SYSINT3_IRQ_LAST	71

#define INTASSIGN0		0xb080
#define INTASSIGN1		0xb082
#define INTASSIGN2		0xb084
#define INTASSIGN3		0xb086
#define SYSINT0REG		0xb088
#define SYSINT1REG		0xb08a
#define SYSINT2REG		0xb08c
#define SYSINT3REG		0xb08e
#define MSYSINT0REG		0xb090
#define MSYSINT1REG		0xb092
#define MSYSINT2REG		0xb094
#define MSYSINT3REG		0xb096
#define NMIREG			0xb09c
#define SOFTINTREG		0xb09e

static inline u16
set_icu(u16 offset, u16 set)
{
	u16 res;

	res = vr4181a_readw(offset);
	res |= set;
	vr4181a_writew(res, offset);
	vr4181a_write_fixed;

	return res;
}

static inline u16
clear_icu(u16 offset, u16 clear)
{
	u16 res;

	res = vr4181a_readw(offset);
	res &= ~clear;
	vr4181a_writew(res, offset);
	vr4181a_write_fixed;

	return res;
}

/*=======================================================================*/

static void
enable_sysint0_irq(unsigned int irq)
{
	set_icu(MSYSINT0REG, 1 << (irq - SYSINT0_IRQ_BASE));
}

static void
disable_sysint0_irq(unsigned int irq)
{
	clear_icu(MSYSINT0REG, 1 << (irq - SYSINT0_IRQ_BASE));
}

static unsigned int
startup_sysint0_irq(unsigned int irq)
{
	set_icu(MSYSINT0REG, 1 << (irq - SYSINT0_IRQ_BASE));

	return 0;		/* never anything pending */
}

#define shutdown_sysint0_irq	disable_sysint0_irq
#define ack_sysint0_irq		disable_sysint0_irq

static void
end_sysint0_irq(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		set_icu(MSYSINT0REG, 1 << (irq - SYSINT0_IRQ_BASE));
}

static struct hw_interrupt_type sysint0_irq_type = {
	.typename = "SYSINT0",
	.startup = startup_sysint0_irq,
	.shutdown = shutdown_sysint0_irq,
	.enable = enable_sysint0_irq,
	.disable = disable_sysint0_irq,
	.ack = ack_sysint0_irq,
	.end = end_sysint0_irq,
};

/*=======================================================================*/
static void
enable_sysint1_irq(unsigned int irq)
{
	set_icu(MSYSINT1REG, 1 << (irq - SYSINT1_IRQ_BASE));
}

static void
disable_sysint1_irq(unsigned int irq)
{
	clear_icu(MSYSINT1REG, 1 << (irq - SYSINT1_IRQ_BASE));
}

static unsigned int
startup_sysint1_irq(unsigned int irq)
{
	set_icu(MSYSINT1REG, 1 << (irq - SYSINT1_IRQ_BASE));

	return 0;		/* never anything pending */
}

#define shutdown_sysint1_irq	disable_sysint1_irq
#define ack_sysint1_irq		disable_sysint1_irq

static void
end_sysint1_irq(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		set_icu(MSYSINT1REG, 1 << (irq - SYSINT1_IRQ_BASE));
}

static struct hw_interrupt_type sysint1_irq_type = {
	.typename = "SYSINT1",
	.startup = startup_sysint1_irq,
	.shutdown = shutdown_sysint1_irq,
	.enable = enable_sysint1_irq,
	.disable = disable_sysint1_irq,
	.ack = ack_sysint1_irq,
	.end = end_sysint1_irq,
};

/*=======================================================================*/

static void
enable_sysint2_irq(unsigned int irq)
{
	set_icu(MSYSINT2REG, 1 << (irq - SYSINT2_IRQ_BASE));
}

static void
disable_sysint2_irq(unsigned int irq)
{
	clear_icu(MSYSINT2REG, 1 << (irq - SYSINT2_IRQ_BASE));
}

static unsigned int
startup_sysint2_irq(unsigned int irq)
{
	set_icu(MSYSINT2REG, 1 << (irq - SYSINT2_IRQ_BASE));

	return 0;		/* never anything pending */
}

#define shutdown_sysint2_irq	disable_sysint2_irq
#define ack_sysint2_irq		disable_sysint2_irq

static void
end_sysint2_irq(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		set_icu(MSYSINT2REG, 1 << (irq - SYSINT2_IRQ_BASE));
}

static struct hw_interrupt_type sysint2_irq_type = {
	.typename = "SYSINT2",
	.startup = startup_sysint2_irq,
	.shutdown = shutdown_sysint2_irq,
	.enable = enable_sysint2_irq,
	.disable = disable_sysint2_irq,
	.ack = ack_sysint2_irq,
	.end = end_sysint2_irq,
};

/*=======================================================================*/

static void
enable_sysint3_irq(unsigned int irq)
{
	set_icu(MSYSINT3REG, 1 << (irq - SYSINT3_IRQ_BASE));
}

static void
disable_sysint3_irq(unsigned int irq)
{
	clear_icu(MSYSINT3REG, 1 << (irq - SYSINT3_IRQ_BASE));
}

static unsigned int
startup_sysint3_irq(unsigned int irq)
{
	set_icu(MSYSINT3REG, 1 << (irq - SYSINT3_IRQ_BASE));

	return 0;		/* never anything pending */
}

#define shutdown_sysint3_irq	disable_sysint3_irq
#define ack_sysint3_irq		disable_sysint3_irq

static void
end_sysint3_irq(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		set_icu(MSYSINT3REG, 1 << (irq - SYSINT3_IRQ_BASE));
}

static struct hw_interrupt_type sysint3_irq_type = {
	.typename = "SYSINT3",
	.startup = startup_sysint3_irq,
	.shutdown = shutdown_sysint3_irq,
	.enable = enable_sysint3_irq,
	.disable = disable_sysint3_irq,
	.ack = ack_sysint3_irq,
	.end = end_sysint3_irq,
};

/*=======================================================================*/

#define INT0_IRQ	2
#define INT1_IRQ	3
#define INT2_IRQ	4
#define INT3_IRQ	5
#define INT4_IRQ	6

extern void init_generic_irq(void);
extern void mips_cpu_irq_init(u32 irq_base);

extern void vr4181a_handle_interrupt(void);
extern void vr4181a_gpint_init(void);

static unsigned int intassign[32];

static struct irqaction icu_cascade = {
	.handler = &no_action,
	.name = "cascade",
};

static void __init
vr4181a_icu_init(void)
{
	u16 assign;
	int i;

	vr4181a_writew(0, MSYSINT0REG);
	vr4181a_write_fixed;
	vr4181a_writew(0, MSYSINT1REG);
	vr4181a_write_fixed;
	vr4181a_writew(0, MSYSINT2REG);
	vr4181a_write_fixed;
	vr4181a_writew(0, MSYSINT3REG);
	vr4181a_write_fixed;

	assign = vr4181a_readw(INTASSIGN0);
	for (i = 0; i < 8; i++)
		intassign[i] = assign & 0x0003 << (i << 1);

	assign = vr4181a_readw(INTASSIGN1);
	for (i = 0; i < 8; i++)
		intassign[i + 8] = assign & 0x0003 << (i << 1);

	assign = vr4181a_readw(INTASSIGN2);
	for (i = 0; i < 8; i++)
		intassign[i + 16] = assign & 0x0003 << (i << 1);

	assign = vr4181a_readw(INTASSIGN3);
	for (i = 0; i < 8; i++)
		intassign[i + 24] = assign & 0x0003 << (i << 1);

	for (i = 0; i < NR_IRQS; i++) {
		if (i >= SYSINT0_IRQ_BASE && i <= SYSINT0_IRQ_LAST)
			irq_desc[i].handler = &sysint0_irq_type;
		else if (i >= SYSINT1_IRQ_BASE && i <= SYSINT1_IRQ_LAST)
			irq_desc[i].handler = &sysint1_irq_type;
		else if (i >= SYSINT2_IRQ_BASE && i <= SYSINT2_IRQ_LAST)
			irq_desc[i].handler = &sysint2_irq_type;
		else if (i >= SYSINT3_IRQ_BASE && i <= SYSINT3_IRQ_LAST)
			irq_desc[i].handler = &sysint3_irq_type;
	}

	if (setup_irq(INT0_IRQ, &icu_cascade) < 0)
		printk(KERN_ERR "ICU: Can not cascade IRQ %d.\n", INT0_IRQ);
	if (setup_irq(INT1_IRQ, &icu_cascade) < 0)
		printk(KERN_ERR "ICU: Can not cascade IRQ %d.\n", INT1_IRQ);
	if (setup_irq(INT2_IRQ, &icu_cascade) < 0)
		printk(KERN_ERR "ICU: Can not cascade IRQ %d.\n", INT2_IRQ);
	if (setup_irq(INT3_IRQ, &icu_cascade) < 0)
		printk(KERN_ERR "ICU: Can not cascade IRQ %d.\n", INT3_IRQ);
	if (setup_irq(INT4_IRQ, &icu_cascade) < 0)
		printk(KERN_ERR "ICU: Can not cascade IRQ %d.\n", INT4_IRQ);
}

void __init
init_IRQ(void)
{
	memset(irq_desc, 0, sizeof (irq_desc));

	init_generic_irq();

	mips_cpu_irq_init(MIPS_CPU_IRQ_BASE);

	vr4181a_icu_init();

	vr4181a_gpint_init();

	set_except_vector(0, vr4181a_handle_interrupt);

#ifdef CONFIG_KGDB
	printk("Setting debug traps - please connect the remote debugger.\n");
	set_debug_traps();
	breakpoint();
#endif
}

/*=======================================================================*/

extern unsigned int gpint_irqdispatch(unsigned int giuintr);
extern void spurious_interrupt(void);
extern unsigned int do_IRQ(int irq, struct pt_regs *regs);

asmlinkage unsigned int
int0_irqdispatch(unsigned int intnum, struct pt_regs *regs)
{
	u32 mask, pend;
	int irq, i;

	irq = -1;

	pend = vr4181a_readw(SYSINT0REG);
	mask = vr4181a_readw(MSYSINT0REG);
	pend |= (u32) vr4181a_readw(SYSINT1REG) << 16;
	mask |= (u32) vr4181a_readw(MSYSINT1REG) << 16;
	pend &= mask;
	if (!(pend & 0x00ff) && pend & 0x0f00) {
		pend &= 0x0f00;
		for (i = 0; i < 4; i++) {
			if (pend & (0x0100 << i))
				irq = gpint_irqdispatch(i);
		}
	} else {
		for (i = 0; i < 32; i++) {
			if (intnum == intassign[i] && pend & (1UL << i))
				irq = SYSINT0_IRQ_BASE + i;
		}
	}

	if (irq < 0)
		spurious_interrupt();

	return do_IRQ(irq, regs);
}

asmlinkage unsigned int
int4_irqdispatch(unsigned int intnum, struct pt_regs *regs)
{
	u32 pend, mask;
	int irq, i;

	irq = -1;

	pend = vr4181a_readw(SYSINT2REG);
	mask = vr4181a_readw(MSYSINT2REG);
	pend |= (u32) vr4181a_readw(SYSINT3REG) << 16;
	mask |= (u32) vr4181a_readw(MSYSINT3REG) << 16;
	pend &= mask;
	for (i = 0; i < 32; i++) {
		if (pend & (1UL << i))
			irq = SYSINT2_IRQ_BASE + i;
	}

	if (irq < 0)
		spurious_interrupt();

	return do_IRQ(irq, regs);
}
