/*
 * arch/mips/vr4181a/common/giu.c
 *
 * General-purpose I/O Unit routines for NEC VR4181A.
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * 2002-2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/types.h>

#include <asm/vr4181a/vr4181a.h>

#define GPINT0_IRQ_BASE	72
#define GPINT0_IRQ_LAST	87
#define GPINT1_IRQ_BASE	88
#define GPINT1_IRQ_LAST	103
#define GPINT2_IRQ_BASE	104
#define GPINT2_IRQ_LAST	119
#define GPINT3_IRQ_BASE	120
#define GPINT3_IRQ_LAST	133

#define GPMODE0			0xb300
#define GPMODE1			0xb302
#define GPMODE2			0xb304
#define GPMODE3			0xb306
#define GPMODE4			0xb308
#define GPMODE5			0xb30a
#define GPMODE6			0xb30c
#define GPMODE7			0xb30e
#define GPDATA0			0xb310
#define GPDATA1			0xb312
#define GPDATA2			0xb314
#define GPDATA3			0xb316
#define GPINEN0			0xb318
#define GPINEN1			0xb31a
#define GPINEN2			0xb31c
#define GPINEN3			0xb31e
#define GPINTMSK0		0xb320
#define GPINTMSK1		0xb322
#define GPINTMSK2		0xb324
#define GPINTMSK3		0xb326
#define GPINTTYP0		0xb328
#define GPINTTYP1		0xb32a
#define GPINTTYP2		0xb32c
#define GPINTTYP3		0xb32e
#define GPINTTYP4		0xb330
#define GPINTTYP5		0xb332
#define GPINTTYP6		0xb334
#define GPINTTYP7		0xb336
#define GPINTSTAT0		0xb338
#define GPINTSTAT1		0xb33a
#define GPINTSTAT2		0xb33c
#define GPINTSTAT3		0xb33e
#define PINMODE			0xb340
#define CF1RSTEN		0x0300
#define CF1RSTEN_ENABLE	0x0300
#define CF1RSTEN_LOW		0x0200
#define CF1RSTEN_HIGH		0x0100
#define CF1RSTEN_PROHIBIT	0x0000
#define G9MODE			0x0002
#define G8MODE			0x0001
#define SDRAMACT		0xb342
#define SDACT			0x0001
#define NVREG0			0xb370
#define NVREG1			0xb372
#define NVREG2			0xb374
#define NVREG3			0xb376
#define PINMODE0		0xb380
#define G51MODE		0x4000
#define G50MODE		0x2000
#define G49MODE		0x1000
#define G48MODE		0x0800
#define G47MODE		0x0400
#define G46MODE		0x0200
#define G39MODE		0x0100
#define G38MODE		0x0080
#define G37MODE		0x0040
#define G23MODE		0x0020
#define G22MODE		0x0010
#define G21MODE		0x0008
#define G20MODE		0x0004
#define G17MODE		0x0002
#define G15MODE		0x0001
#define PINMODE1		0xb382
#define G14MODE		0x0010
#define G13MODE		0x0008
#define G12MODE		0x0004
#define G11MODE		0x0002
#define G10MODE		0x0001
#define PINMODE2		0xb384
#define CMODE			0xc000
#define CMODE_PROHIBIT		0x0000
#define CMODE_ENABLE		0x4000
#define S6MODE			0x3000
#define S6MODE_SRESET		0x3000
#define S6MODE_DSR2		0x1000
#define S6MODE_PROHIBIT	0x0000
#define S5MODE			0x0c00
#define S5MODE_SDATAIN		0x0c00
#define S5MODE_SDI		0x0800
#define S5MODE_DCD2		0x0400
#define S5MODE_PROHIBIT	0x0000
#define S4MODE			0x0300
#define S4MODE_SDATAOUT	0x0300
#define S4MODE_SDO		0x0200
#define S4MODE_DTR2		0x0100
#define S4MODE_PROHIBIT	0x0000
#define S3MODE			0x00c0
#define S3MODE_SYNC		0x00c0
#define S3MODE_WS		0x0080
#define S3MODE_RTS2		0x0040
#define S3MODE_PROHIBIT	0x0000
#define S2MODE			0x0030
#define S2MODE_BITCLK		0x0030
#define S2MODE_SCLK		0x0020
#define S2MODE_CTS2		0x0010
#define S2MODE_PROHIBIT	0x0000
#define S1MODE			0x000c
#define S1MODE_IRDOUT		0x000c
#define S1MODE_TXD2		0x0004
#define S1MODE_PROHIBIT	0x0000
#define S0MODE			0x0003
#define S0MODE_IRDIN		0x0003
#define S0MODE_RXD2		0x0001
#define S0MODE_PROHIBIT	0x0000
#define ECU1SIGCTRL		0xb390
#define CF1_CD2		0x0004
#define CF1_CD1		0x0002
#define CF1_IOIS16		0x0001
#define USBSIGCTRL		0xb392
#define IENF			0x0008
#define PSF			0x0004
#define IENH			0x0002
#define PSH			0x0001

static inline u16
set_gpint(u16 offset, u16 set)
{
	u16 res;

	res = vr4181a_readw(offset);
	res |= set;
	vr4181a_writew(res, offset);
	vr4181a_write_fixed;

	return res;
}

static inline u16
clear_gpint(u16 offset, u16 clear)
{
	u16 res;

	res = vr4181a_readw(offset);
	res &= ~clear;
	vr4181a_writew(res, offset);
	vr4181a_write_fixed;

	return res;
}

/*=======================================================================*/

static void
enable_gpint0_irq(unsigned int irq)
{
	set_gpint(GPINTMSK0, 1 << (irq - GPINT0_IRQ_BASE));
}

static void
disable_gpint0_irq(unsigned int irq)
{
	clear_gpint(GPINTMSK0, 1 << (irq - GPINT0_IRQ_BASE));
}

static unsigned int
startup_gpint0_irq(unsigned int irq)
{
	set_gpint(GPINTMSK0, 1 << (irq - GPINT0_IRQ_BASE));

	return 0;		/* never anything pending */
}

#define shutdown_gpint0_irq	disable_gpint0_irq

static void
ack_gpint0_irq(unsigned int irq)
{
	u16 status;

	status = 1 << (irq - GPINT0_IRQ_BASE);
	clear_gpint(GPINTMSK0, status);
	vr4181a_writew(status, GPINTSTAT0);
	vr4181a_write_fixed;
}

static void
end_gpint0_irq(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		set_gpint(GPINTMSK0, 1 << (irq - GPINT0_IRQ_BASE));
}

static struct hw_interrupt_type gpint0_irq_type = {
	.typename = "GPINT0",
	.startup = startup_gpint0_irq,
	.shutdown = shutdown_gpint0_irq,
	.enable = enable_gpint0_irq,
	.disable = disable_gpint0_irq,
	.ack = ack_gpint0_irq,
	.end = end_gpint0_irq,
};

/*=======================================================================*/

static void
enable_gpint1_irq(unsigned int irq)
{
	set_gpint(GPINTMSK1, 1 << (irq - GPINT1_IRQ_BASE));
}

static void
disable_gpint1_irq(unsigned int irq)
{
	clear_gpint(GPINTMSK1, 1 << (irq - GPINT1_IRQ_BASE));
}

static unsigned int
startup_gpint1_irq(unsigned int irq)
{
	set_gpint(GPINTMSK1, 1 << (irq - GPINT1_IRQ_BASE));

	return 0;		/* never anything pending */
}

#define shutdown_gpint1_irq	disable_gpint1_irq

static void
ack_gpint1_irq(unsigned int irq)
{
	u16 status;

	status = 1 << (irq - GPINT1_IRQ_BASE);
	clear_gpint(GPINTMSK1, status);
	vr4181a_writew(status, GPINTSTAT1);
	vr4181a_write_fixed;
}

static void
end_gpint1_irq(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		set_gpint(GPINTMSK1, 1 << (irq - GPINT1_IRQ_BASE));
}

static struct hw_interrupt_type gpint1_irq_type = {
	.typename = "GPINT1",
	.startup = startup_gpint1_irq,
	.shutdown = shutdown_gpint1_irq,
	.enable = enable_gpint1_irq,
	.disable = disable_gpint1_irq,
	.ack = ack_gpint1_irq,
	.end = end_gpint1_irq,
};

/*=======================================================================*/

static void
enable_gpint2_irq(unsigned int irq)
{
	set_gpint(GPINTMSK2, 1 << (irq - GPINT2_IRQ_BASE));
}

static void
disable_gpint2_irq(unsigned int irq)
{
	clear_gpint(GPINTMSK2, 1 << (irq - GPINT2_IRQ_BASE));
}

static unsigned int
startup_gpint2_irq(unsigned int irq)
{
	set_gpint(GPINTMSK2, 1 << (irq - GPINT2_IRQ_BASE));

	return 0;		/* never anything pending */
}

#define shutdown_gpint2_irq	disable_gpint2_irq

static void
ack_gpint2_irq(unsigned int irq)
{
	u16 status;

	status = 1 << (irq - GPINT2_IRQ_BASE);
	clear_gpint(GPINTMSK2, status);
	vr4181a_writew(status, GPINTSTAT2);
	vr4181a_write_fixed;
}

static void
end_gpint2_irq(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		set_gpint(GPINTMSK2, 1 << (irq - GPINT2_IRQ_BASE));
}

static struct hw_interrupt_type gpint2_irq_type = {
	.typename = "GPINT2",
	.startup = startup_gpint2_irq,
	.shutdown = shutdown_gpint2_irq,
	.enable = enable_gpint2_irq,
	.disable = disable_gpint2_irq,
	.ack = ack_gpint2_irq,
	.end = end_gpint2_irq,
};

/*=======================================================================*/

static void
enable_gpint3_irq(unsigned int irq)
{
	set_gpint(GPINTMSK3, 1 << (irq - GPINT3_IRQ_BASE));
}

static void
disable_gpint3_irq(unsigned int irq)
{
	clear_gpint(GPINTMSK3, 1 << (irq - GPINT3_IRQ_BASE));
}

static unsigned int
startup_gpint3_irq(unsigned int irq)
{
	set_gpint(GPINTMSK3, 1 << (irq - GPINT3_IRQ_BASE));

	return 0;		/* never anything pending */
}

#define shutdown_gpint3_irq	disable_gpint3_irq

static void
ack_gpint3_irq(unsigned int irq)
{
	u16 status;

	status = 1 << (irq - GPINT3_IRQ_BASE);
	clear_gpint(GPINTMSK3, status);
	vr4181a_writew(status, GPINTSTAT3);
	vr4181a_write_fixed;
}

static void
end_gpint3_irq(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		set_gpint(GPINTMSK3, 1 << (irq - GPINT3_IRQ_BASE));
}

static struct hw_interrupt_type gpint3_irq_type = {
	.typename = "GPINT3",
	.startup = startup_gpint3_irq,
	.shutdown = shutdown_gpint3_irq,
	.enable = enable_gpint3_irq,
	.disable = disable_gpint3_irq,
	.ack = ack_gpint3_irq,
	.end = end_gpint3_irq,
};

/*=======================================================================*/

static inline void
gpint_clear(unsigned short pin)
{
	if (pin < 16)
		vr4181a_writew(1 << pin, GPINTSTAT0);
	else if (pin < 32)
		vr4181a_writew(1 << (pin - 16), GPINTSTAT1);
	else if (pin < 48)
		vr4181a_writew(1 << (pin - 32), GPINTSTAT2);
	else if (pin < 62)
		vr4181a_writew(1 << (pin - 48), GPINTSTAT3);
	vr4181a_write_fixed;
}

void
vr4181a_set_irq_trigger(unsigned short pin, int trigger)
{
	u16 offset, type;

	if (pin < 8) {
		offset = GPINTTYP0;
		type = 2 << (pin << 1);
	} else if (pin < 16) {
		offset = GPINTTYP1;
		type = 2 << ((pin - 8) << 1);
	} else if (pin < 24) {
		offset = GPINTTYP2;
		type = 2 << ((pin - 16) << 1);
	} else if (pin < 32) {
		offset = GPINTTYP3;
		type = 2 << ((pin - 24) << 1);
	} else if (pin < 40) {
		offset = GPINTTYP4;
		type = 2 << ((pin - 32) << 1);
	} else if (pin < 48) {
		offset = GPINTTYP5;
		type = 2 << ((pin - 40) << 1);
	} else if (pin < 56) {
		offset = GPINTTYP6;
		type = 2 << ((pin - 48) << 1);
	} else if (pin < 62) {
		offset = GPINTTYP7;
		type = 2 << ((pin - 56) << 1);
	} else {
		return;
	}

	if (trigger == TRIGGER_LEVEL)
		set_gpint(offset, type);
	else
		clear_gpint(offset, type);

	gpint_clear(pin);
}

void
vr4181a_set_irq_level(unsigned short pin, int level)
{
	u16 offset, type;

	if (pin < 8) {
		offset = GPINTTYP0;
		type = 1 << (pin << 1);
	} else if (pin < 16) {
		offset = GPINTTYP1;
		type = 1 << ((pin - 8) << 1);
	} else if (pin < 24) {
		offset = GPINTTYP2;
		type = 1 << ((pin - 16) << 1);
	} else if (pin < 32) {
		offset = GPINTTYP3;
		type = 1 << ((pin - 24) << 1);
	} else if (pin < 40) {
		offset = GPINTTYP4;
		type = 1 << ((pin - 32) << 1);
	} else if (pin < 48) {
		offset = GPINTTYP5;
		type = 1 << ((pin - 40) << 1);
	} else if (pin < 56) {
		offset = GPINTTYP6;
		type = 1 << ((pin - 48) << 1);
	} else if (pin < 62) {
		offset = GPINTTYP7;
		type = 1 << ((pin - 56) << 1);
	} else {
		return;
	}

	if (level == LEVEL_HIGH)
		set_gpint(offset, type);
	else
		clear_gpint(offset, type);

	gpint_clear(pin);
}

/*=======================================================================*/

#define GPINT0_CASCADE_IRQ	16
#define GPINT1_CASCADE_IRQ	17
#define GPINT2_CASCADE_IRQ	18
#define GPINT3_CASCADE_IRQ	19

static struct irqaction gpint_cascade = {
	.handler = &no_action,
	.name = "cascade",
};

void __init
vr4181a_gpint_init(void)
{
	int i;

	for (i = 0; i < NR_IRQS; i++) {
		if (i >= GPINT0_IRQ_BASE && i <= GPINT0_IRQ_LAST)
			irq_desc[i].handler = &gpint0_irq_type;
		if (i >= GPINT1_IRQ_BASE && i <= GPINT1_IRQ_LAST)
			irq_desc[i].handler = &gpint1_irq_type;
		if (i >= GPINT2_IRQ_BASE && i <= GPINT2_IRQ_LAST)
			irq_desc[i].handler = &gpint2_irq_type;
		if (i >= GPINT3_IRQ_BASE && i <= GPINT3_IRQ_LAST)
			irq_desc[i].handler = &gpint3_irq_type;
	}

	if (setup_irq(GPINT0_CASCADE_IRQ, &gpint_cascade) < 0)
		printk(KERN_ERR "GIU: Can not cascade IRQ %d.\n",
		       GPINT0_CASCADE_IRQ);
	if (setup_irq(GPINT1_CASCADE_IRQ, &gpint_cascade) < 0)
		printk(KERN_ERR "GIU: Can not cascade IRQ %d.\n",
		       GPINT1_CASCADE_IRQ);
	if (setup_irq(GPINT2_CASCADE_IRQ, &gpint_cascade) < 0)
		printk(KERN_ERR "GIU: Can not cascade IRQ %d.\n",
		       GPINT2_CASCADE_IRQ);
	if (setup_irq(GPINT3_CASCADE_IRQ, &gpint_cascade) < 0)
		printk(KERN_ERR "GIU: Can not cascade IRQ %d.\n",
		       GPINT3_CASCADE_IRQ);
}

static inline unsigned int
gpint0_irqdispatch(void)
{
	u16 pend, mask;
	int i;

	pend = vr4181a_readw(GPINTSTAT0);
	mask = vr4181a_readw(GPINTMSK0);
	pend &= mask;
	for (i = 0; i < 16; i++) {
		if (pend & (1 << i))
			return GPINT0_IRQ_BASE + i;
	}

	return -1;
}

static inline unsigned int
gpint1_irqdispatch(void)
{
	u16 pend, mask;
	int i;

	pend = vr4181a_readw(GPINTSTAT1);
	mask = vr4181a_readw(GPINTMSK1);
	pend &= mask;
	for (i = 0; i < 16; i++) {
		if (pend & (1 << i))
			return GPINT1_IRQ_BASE + i;
	}

	return -1;
}

static inline unsigned int
gpint2_irqdispatch(void)
{
	u16 pend, mask;
	int i;

	pend = vr4181a_readw(GPINTSTAT2);
	mask = vr4181a_readw(GPINTMSK2);
	pend &= mask;
	for (i = 0; i < 16; i++) {
		if (pend & (1 << i))
			return GPINT2_IRQ_BASE + i;
	}

	return -1;
}

static inline int
gpint3_irqdispatch(void)
{
	u16 pend, mask;
	int i;

	pend = vr4181a_readw(GPINTSTAT3);
	mask = vr4181a_readw(GPINTMSK3);
	pend &= mask;
	for (i = 0; i < 16; i++) {
		if (pend & (1 << i))
			return GPINT3_IRQ_BASE + i;
	}

	return -1;
}

int
gpint_irqdispatch(unsigned int giuintr)
{
	switch (giuintr) {
	case 0:
		return gpint0_irqdispatch();
	case 1:
		return gpint1_irqdispatch();
	case 2:
		return gpint2_irqdispatch();
	case 3:
		return gpint3_irqdispatch();
	default:
		break;
	}

	return -1;
}

/*=======================================================================*/

int
vr4181a_set_gpio_mode(unsigned short pin, int mode, int input_mode)
{
	u16 gpmode, gpinen;
	unsigned int gpen, gien;

	if (pin < 16) {
		if (pin < 8) {
			gpmode = vr4181a_readw(GPMODE0);
			gpen = pin * 2;
		} else {
			gpmode = vr4181a_readw(GPMODE1);
			gpen = (pin - 8) * 2;
		}
		gpinen = vr4181a_readw(GPINEN0);
		gien = pin;
	} else if (pin < 32) {
		if (pin < 24) {
			gpmode = vr4181a_readw(GPMODE2);
			gpen = (pin - 16) * 2;
		} else {
			gpmode = vr4181a_readw(GPMODE3);
			gpen = (pin - 24) * 2;
		}
		gpinen = vr4181a_readw(GPINEN1);
		gien = pin - 16;
	} else if (pin < 48) {
		if (pin < 40) {
			gpmode = vr4181a_readw(GPMODE4);
			gpen = (pin - 32) * 2;
		} else {
			gpmode = vr4181a_readw(GPMODE5);
			gpen = (pin - 40) * 2;
		}
		gpinen = vr4181a_readw(GPINEN2);
		gien = pin - 32;
	} else if (pin < 64) {
		if (pin < 56) {
			gpmode = vr4181a_readw(GPMODE6);
			gpen = (pin - 48) * 2;
		} else {
			gpmode = vr4181a_readw(GPMODE7);
			gpen = (pin - 56) * 2;
		}
		gpinen = vr4181a_readw(GPINEN3);
		gien = pin - 48;
	} else
		return -ENODEV;

	if (mode != GPIO_OTHER) {
		if (mode == GPIO_OUTPUT)
			gpmode |= 0x0001 << (gpen + 1);
		else
			gpmode &= ~(0x0001 << (gpen + 1));
		gpmode |= 0x0001 << gpen;
	} else {
		gpmode &= ~(0x0001 << gpen);
	}

	if (input_mode == GPIO_INPUT_ENABLE)
		gpinen |= 0x0001 << gien;
	else
		gpinen &= ~(0x0001 << gien);

	if (pin < 16) {
		if (pin < 8)
			vr4181a_writew(gpmode, GPMODE0);
		else
			vr4181a_writew(gpmode, GPMODE1);
		vr4181a_write_fixed;
		vr4181a_writew(gpinen, GPINEN0);
		vr4181a_write_fixed;
	} else if (pin < 32) {
		if (pin < 24)
			vr4181a_writew(gpmode, GPMODE2);
		else
			vr4181a_writew(gpmode, GPMODE3);
		vr4181a_write_fixed;
		vr4181a_writew(gpinen, GPINEN1);
		vr4181a_write_fixed;
	} else if (pin < 48) {
		if (pin < 40)
			vr4181a_writew(gpmode, GPMODE4);
		else
			vr4181a_writew(gpmode, GPMODE5);
		vr4181a_write_fixed;
		vr4181a_writew(gpinen, GPINEN2);
		vr4181a_write_fixed;
	} else if (pin < 64) {
		if (pin < 56)
			vr4181a_writew(gpmode, GPMODE6);
		else
			vr4181a_writew(gpmode, GPMODE7);
		vr4181a_write_fixed;
		vr4181a_writew(gpinen, GPINEN3);
		vr4181a_write_fixed;
	}

	return 0;
}

int
vr4181a_get_gpio_data(unsigned short pin)
{
	u16 data;

	if (pin < 16) {
		data = vr4181a_readw(GPDATA0);
		data >>= pin;
	} else if (pin < 32) {
		data = vr4181a_readw(GPDATA1);
		data >>= pin - 16;
	} else if (pin < 48) {
		data = vr4181a_readw(GPDATA2);
		data >>= pin - 32;
	} else if (pin < 64) {
		data = vr4181a_readw(GPDATA3);
		data >>= pin - 48;
	} else
		return -ENODEV;

	return data & 1;
}

int
vr4181a_set_gpio_data(unsigned short pin, int data)
{
	u16 value;

	if (pin < 16) {
		value = vr4181a_readw(GPDATA0);
		value &= ~(0x0001 << pin);
		value |= (u16) (data & 1) << pin;
		vr4181a_writew(value, GPDATA0);
	} else if (pin < 32) {
		value = vr4181a_readw(GPDATA1);
		value &= ~(0x0001 << (pin - 16));
		value |= (u16) (data & 1) << (pin - 16);
		vr4181a_writew(value, GPDATA1);
	} else if (pin < 48) {
		value = vr4181a_readw(GPDATA2);
		value &= ~(0x0001 << (pin - 32));
		value |= (u16) (data & 1) << (pin - 32);
		vr4181a_writew(value, GPDATA2);
	} else if (pin < 64) {
		value = vr4181a_readw(GPDATA3);
		value &= ~(0x0001 << (pin - 48));
		value |= (u16) (data & 1) << (pin - 48);
		vr4181a_writew(value, GPDATA3);
	} else
		return -ENODEV;

	vr4181a_write_fixed;

	return data & 1;
}

static unsigned int pinmode, pinmode0, pinmode1, pinmode2;
static u16 vr4181a_pinmode, vr4181a_pinmode0, vr4181a_pinmode1,
    vr4181a_pinmode2;

int
vr4181a_get_pinmode(void)
{
	return pinmode;
}

void
vr4181a_set_pinmode(int mode)
{
	if (mode & (PINMODE_PWM0 | PINMODE_KSCAN7)) {
		if (mode & PINMODE_PWM0) {
			pinmode &= ~PINMODE_KSCAN7;
			pinmode |= PINMODE_PWM0;
			vr4181a_pinmode &= ~G8MODE;
		} else {
			pinmode &= ~PINMODE_PWM0;
			pinmode |= PINMODE_KSCAN7;
			vr4181a_pinmode |= G8MODE;
		}
		vr4181a_set_gpio_mode(8, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else {
		pinmode &= ~(PINMODE_PWM0 | PINMODE_KSCAN7);
	}

	if (mode & (PINMODE_PWM1 | PINMODE_KSCAN6)) {
		if (mode & PINMODE_PWM1) {
			pinmode &= ~PINMODE_KSCAN6;
			pinmode |= PINMODE_PWM1;
			vr4181a_pinmode &= ~G9MODE;
		} else {
			pinmode &= ~PINMODE_PWM1;
			pinmode |= PINMODE_KSCAN6;
			vr4181a_pinmode |= G9MODE;
		}
		vr4181a_set_gpio_mode(9, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else {
		pinmode &= ~(PINMODE_PWM1 | PINMODE_KSCAN6);
	}

	switch (mode & 0xf0) {
	case PINMODE_CF1_RESET_PROHIBIT:
		pinmode &=
		    ~(PINMODE_CF1_RESET_FIXED1 | PINMODE_CF1_RESET_FIXED0 |
		      PINMODE_CF1_RESET_ENABLE);
		pinmode |= PINMODE_CF1_RESET_PROHIBIT;
		vr4181a_pinmode &= ~CF1RSTEN;
		break;
	case PINMODE_CF1_RESET_FIXED1:
		pinmode &=
		    ~(PINMODE_CF1_RESET_PROHIBIT | PINMODE_CF1_RESET_FIXED0 |
		      PINMODE_CF1_RESET_ENABLE);
		vr4181a_pinmode &= ~CF1RSTEN;
		vr4181a_pinmode |= CF1RSTEN_HIGH;
		break;
	case PINMODE_CF1_RESET_FIXED0:
		pinmode &=
		    ~(PINMODE_CF1_RESET_PROHIBIT | PINMODE_CF1_RESET_FIXED1 |
		      PINMODE_CF1_RESET_ENABLE);
		vr4181a_pinmode &= ~CF1RSTEN;
		vr4181a_pinmode |= CF1RSTEN_LOW;
		break;
	case PINMODE_CF1_RESET_ENABLE:
		pinmode &=
		    ~(PINMODE_CF1_RESET_PROHIBIT | PINMODE_CF1_RESET_FIXED1 |
		      PINMODE_CF1_RESET_FIXED0);
		vr4181a_pinmode |= CF1RSTEN_ENABLE;
		break;
	default:
		break;
	}

	vr4181a_writew(vr4181a_pinmode, PINMODE);
	vr4181a_write_fixed;
}

int
vr4181a_get_pinmode0(void)
{
	return pinmode0;
}

void
vr4181a_set_pinmode0(int mode)
{
	if (mode & (PINMODE0_SIU0 | PINMODE0_SIU1)) {
		if (mode & PINMODE0_SIU0) {
			pinmode0 &= ~PINMODE0_SIU1;
			pinmode0 |= PINMODE0_SIU0;
			vr4181a_pinmode0 &= ~G15MODE;
			vr4181a_pinmode0 |= G17MODE;
		} else {
			pinmode0 &= ~PINMODE0_SIU0;
			pinmode0 |= PINMODE0_SIU1;
			vr4181a_pinmode0 &= ~G17MODE;
			vr4181a_pinmode0 |= G15MODE;
		}
		vr4181a_set_gpio_mode(15, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(17, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else {
		pinmode0 &= ~(PINMODE0_SIU0 | PINMODE0_SIU1);
	}

	if (mode & PINMODE0_CSI) {
		pinmode0 &=
		    ~(PINMODE0_KSCAN8 | PINMODE0_KSCAN9 | PINMODE0_KSCAN10);
		pinmode0 |= PINMODE0_CSI;
		vr4181a_pinmode0 &= ~(G20MODE | G21MODE | G22MODE);
		vr4181a_set_gpio_mode(20, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(21, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(22, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_KSCAN8) {
		pinmode0 &= ~PINMODE0_CSI;
		pinmode0 |= PINMODE0_KSCAN8;
		vr4181a_pinmode0 |= G20MODE;
		vr4181a_set_gpio_mode(20, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_KSCAN9) {
		pinmode0 &= ~PINMODE0_CSI;
		pinmode0 |= PINMODE0_KSCAN9;
		vr4181a_pinmode0 |= G21MODE;
		vr4181a_set_gpio_mode(21, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_KSCAN10) {
		pinmode0 &= ~PINMODE0_CSI;
		pinmode0 |= PINMODE0_KSCAN10;
		vr4181a_pinmode0 |= G22MODE;
		vr4181a_set_gpio_mode(22, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else {
		pinmode0 &= ~(PINMODE0_CSI | PINMODE0_KSCAN8 | PINMODE0_KSCAN9 |
			      PINMODE0_KSCAN10);
	}

	if (mode & (PINMODE0_I2S | PINMODE0_KSCAN11)) {
		if (mode & PINMODE0_I2S) {
			pinmode0 &= ~PINMODE0_KSCAN11;
			pinmode0 |= PINMODE0_I2S;
			vr4181a_pinmode0 &= ~G23MODE;
		} else {
			pinmode0 &= ~PINMODE0_I2S;
			pinmode0 |= PINMODE0_KSCAN11;
			vr4181a_pinmode0 |= G23MODE;
		}
		vr4181a_set_gpio_mode(23, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else {
		pinmode0 &= ~(PINMODE0_I2S | PINMODE0_KSCAN11);
	}

	if (mode & PINMODE0_ECU1) {
		pinmode0 &=
		    ~(PINMODE0_KSCAN4 | PINMODE0_KPORT5 | PINMODE0_KPORT4 |
		      PINMODE0_FPD10 | PINMODE0_FPD11 | PINMODE0_FPD12 |
		      PINMODE0_FPD13 | PINMODE0_FPD14 | PINMODE0_FPD15);
		pinmode0 |= PINMODE0_ECU1;
		vr4181a_pinmode0 &=
		    ~(G46MODE | G47MODE | G48MODE | G49MODE | G50MODE |
		      G51MODE);
		vr4181a_pinmode0 |= G37MODE | G38MODE | G39MODE;
		vr4181a_set_gpio_mode(37, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(38, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(39, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(46, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(47, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(48, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(49, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(50, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(51, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_KSCAN4) {
		pinmode0 &= ~PINMODE0_ECU1;
		pinmode0 |= PINMODE0_KSCAN4;
		vr4181a_pinmode0 &= ~G37MODE;
		vr4181a_set_gpio_mode(37, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_KPORT5) {
		pinmode0 &= ~PINMODE0_ECU1;
		pinmode0 |= PINMODE0_KPORT5;
		vr4181a_pinmode0 &= ~G38MODE;
		vr4181a_set_gpio_mode(38, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_KPORT4) {
		pinmode0 &= ~PINMODE0_ECU1;
		pinmode0 |= PINMODE0_KPORT4;
		vr4181a_pinmode0 &= ~G39MODE;
		vr4181a_set_gpio_mode(39, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_FPD10) {
		pinmode0 &= ~PINMODE0_ECU1;
		pinmode0 |= PINMODE0_FPD10;
		vr4181a_pinmode0 &= ~G46MODE;
		vr4181a_set_gpio_mode(46, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_FPD11) {
		pinmode0 &= ~PINMODE0_ECU1;
		pinmode0 |= PINMODE0_FPD11;
		vr4181a_pinmode0 &= ~G47MODE;
		vr4181a_set_gpio_mode(47, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_FPD12) {
		pinmode0 &= ~PINMODE0_ECU1;
		pinmode0 |= PINMODE0_FPD12;
		vr4181a_pinmode0 &= ~G48MODE;
		vr4181a_set_gpio_mode(48, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_FPD13) {
		pinmode0 &= ~PINMODE0_ECU1;
		pinmode0 |= PINMODE0_FPD13;
		vr4181a_pinmode0 &= ~G49MODE;
		vr4181a_set_gpio_mode(49, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_FPD14) {
		pinmode0 &= ~PINMODE0_ECU1;
		pinmode0 |= PINMODE0_FPD14;
		vr4181a_pinmode0 &= ~G50MODE;
		vr4181a_set_gpio_mode(50, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE0_FPD15) {
		pinmode0 &= ~PINMODE0_ECU1;
		pinmode0 |= PINMODE0_FPD15;
		vr4181a_pinmode0 &= ~G51MODE;
		vr4181a_set_gpio_mode(51, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else {
		pinmode0 &=
		    ~(PINMODE0_ECU1 | PINMODE0_KSCAN4 | PINMODE0_KPORT5 |
		      PINMODE0_KPORT4 | PINMODE0_FPD10 | PINMODE0_FPD11 |
		      PINMODE0_FPD12 | PINMODE0_FPD13 | PINMODE0_FPD14 |
		      PINMODE0_FPD15);
	}

	vr4181a_writew(vr4181a_pinmode0, PINMODE0);
	vr4181a_write_fixed;
}

int
vr4181a_get_pinmode1(void)
{
	return pinmode1;
}

void
vr4181a_set_pinmode1(int mode)
{
	if (mode & (PINMODE1_PWM2 | PINMODE1_KSCAN5)) {
		if (mode & PINMODE1_PWM2) {
			pinmode1 &= ~PINMODE1_KSCAN5;
			pinmode1 |= PINMODE1_PWM2;
			vr4181a_pinmode1 &= ~G10MODE;
		} else {
			pinmode1 &= ~PINMODE1_PWM2;
			pinmode1 |= PINMODE1_KSCAN5;
			vr4181a_pinmode1 |= G10MODE;
		}
		vr4181a_set_gpio_mode(10, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else {
		pinmode1 &= ~(PINMODE1_PWM2 | PINMODE1_KSCAN5);
	}

	if (mode & PINMODE1_I2C0) {
		pinmode1 &= ~(PINMODE1_KPORT6 | PINMODE1_KPORT7);
		pinmode1 |= PINMODE1_I2C0;
		vr4181a_pinmode1 &= ~(G11MODE | G12MODE);
		vr4181a_set_gpio_mode(11, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(12, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE1_KPORT6) {
		pinmode1 &= ~PINMODE1_I2C0;
		pinmode1 |= PINMODE1_KPORT6;
		vr4181a_pinmode1 |= G11MODE;
		vr4181a_set_gpio_mode(11, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else if (mode & PINMODE1_KPORT7) {
		pinmode1 &= ~PINMODE1_I2C0;
		pinmode1 |= PINMODE1_KPORT7;
		vr4181a_pinmode1 |= G12MODE;
		vr4181a_set_gpio_mode(12, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else {
		pinmode1 &=
		    ~(PINMODE1_I2C0 | PINMODE1_KPORT6 | PINMODE1_KPORT7);
	}

	if (mode & (PINMODE1_I2C1 | PINMODE1_SIU1)) {
		if (mode & PINMODE1_I2C1) {
			pinmode1 &= ~PINMODE1_SIU1;
			pinmode1 |= PINMODE1_I2C1;
			vr4181a_pinmode1 &= ~(G13MODE | G14MODE);
		} else {
			pinmode1 &= ~PINMODE1_I2C1;
			pinmode1 |= PINMODE1_SIU1;
			vr4181a_pinmode1 |= G13MODE | G14MODE;
		}
		vr4181a_set_gpio_mode(13, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(14, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	} else {
		pinmode1 &= ~(PINMODE1_I2C1 | PINMODE1_SIU1);
	}

	vr4181a_writew(vr4181a_pinmode1, PINMODE1);
	vr4181a_write_fixed;
}

int
vr4181a_get_pinmode2(void)
{
	return pinmode2;
}

void
vr4181a_set_pinmode2(int mode)
{
	if (mode & PINMODE2_PROHIBIT || mode == 0) {
		pinmode2 = PINMODE2_PROHIBIT;
		vr4181a_pinmode2 = 0;
	} else {
		pinmode2 &= ~PINMODE2_PROHIBIT;
		vr4181a_pinmode2 &= ~(S0MODE | S1MODE | S2MODE | S3MODE |
				      S4MODE | S5MODE | S6MODE);
		if (mode & (PINMODE2_SIU2 | PINMODE2_SIU2_IRDA)) {
			pinmode2 &= ~(PINMODE2_I2S | PINMODE2_AC97);
			if (mode & PINMODE2_SIU2) {
				pinmode2 &= ~PINMODE2_SIU2_IRDA;
				vr4181a_pinmode2 |= S0MODE_RXD2 | S1MODE_TXD2;
			} else {
				pinmode2 &= ~PINMODE2_SIU2;
				vr4181a_pinmode2 |=
				    S0MODE_IRDIN | S1MODE_IRDOUT;
			}
			vr4181a_pinmode2 |=
			    S2MODE_CTS2 | S3MODE_RTS2 | S4MODE_DTR2 |
			    S5MODE_DCD2 | S6MODE_DSR2;
		} else if (mode & PINMODE2_I2S) {
			pinmode2 &= ~(PINMODE2_SIU2 | PINMODE2_SIU2_IRDA |
				      PINMODE2_AC97);
			pinmode2 |= PINMODE2_I2S;
			vr4181a_pinmode2 |=
			    S2MODE_SCLK | S3MODE_WS | S4MODE_SDO | S5MODE_SDI;
		} else if (mode & PINMODE2_AC97) {
			pinmode2 &= ~(PINMODE2_SIU2 | PINMODE2_SIU2_IRDA |
				      PINMODE2_I2S);
			pinmode2 |= PINMODE2_AC97;
			vr4181a_pinmode2 |= S2MODE_BITCLK | S3MODE_SYNC |
			    S4MODE_SDATAOUT | S5MODE_SDATAIN | S6MODE_SRESET;
		} else if (mode & PINMODE2_USB_ENABLE) {
			pinmode2 |= PINMODE2_USB_ENABLE;
			vr4181a_pinmode2 &= ~CMODE;
			vr4181a_pinmode2 |= CMODE_ENABLE;
		}
	}

	vr4181a_writew(vr4181a_pinmode2, PINMODE2);
	vr4181a_write_fixed;
}

u16
vr4181a_read_nvreg(int regno)
{
	u16 value = 0;

	switch (regno) {
	case 0:
		value = vr4181a_readw(NVREG0);
		break;
	case 1:
		value = vr4181a_readw(NVREG1);
		break;
	case 2:
		value = vr4181a_readw(NVREG2);
		break;
	case 3:
		value = vr4181a_readw(NVREG3);
		break;
	}

	return value;
}

void
vr4181a_write_nvreg(int regno, u16 value)
{
	switch (regno) {
	case 0:
		vr4181a_writew(value, NVREG0);
		break;
	case 1:
		vr4181a_writew(value, NVREG1);
		break;
	case 2:
		vr4181a_writew(value, NVREG2);
		break;
	case 3:
		vr4181a_writew(value, NVREG3);
		break;
	}

	vr4181a_write_fixed;
}

void
vr4181a_power_active_usb_function(void)
{
	u16 val;

	val = vr4181a_readw(USBSIGCTRL);
	val |= IENF;
	vr4181a_writew(val, USBSIGCTRL);
	vr4181a_write_fixed;

	val = vr4181a_readw(USBSIGCTRL);
	val |= PSF;
	vr4181a_writew(val, USBSIGCTRL);
	vr4181a_write_fixed;
}

void
vr4181a_power_save_usb_function(void)
{
	u16 val;

	val = vr4181a_readw(USBSIGCTRL);
	val &= ~IENF;
	vr4181a_writew(val, USBSIGCTRL);
	vr4181a_write_fixed;

	val = vr4181a_readw(USBSIGCTRL);
	val &= ~PSF;
	vr4181a_writew(val, USBSIGCTRL);
	vr4181a_write_fixed;
}

void
vr4181a_power_active_usb_host(void)
{
	u16 val;

	val = vr4181a_readw(USBSIGCTRL);
	val |= IENH;
	vr4181a_writew(val, USBSIGCTRL);
	vr4181a_write_fixed;

	val = vr4181a_readw(USBSIGCTRL);
	val |= PSH;
	vr4181a_writew(val, USBSIGCTRL);
	vr4181a_write_fixed;
}

void
vr4181a_power_save_usb_host(void)
{
	u16 val;

	val = vr4181a_readw(USBSIGCTRL);
	val &= ~IENH;
	vr4181a_writew(val, USBSIGCTRL);
	vr4181a_write_fixed;

	val = vr4181a_readw(USBSIGCTRL);
	val &= ~PSH;
	vr4181a_writew(val, USBSIGCTRL);
	vr4181a_write_fixed;
}
