/*
 * arch/mips/vr4181a/common/ecu.c
 *
 * CompactFlash/PC Card/IDE (ATA) Control Unit routines for the NEC VR4181A.
 *
 * Author: Yoichi Yuasa <yyuasa@mvista.com, or source@mvista.com>
 *
 * IDE routines initial work by:
 *	Copyright 2002 NEC Micro Systems, Ltd.
 *	Author: Takahiro Miwa <t-miwa@pi.jp.nec.com>
 *
 * 2003 (c) MontaVista, Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */
#include <linux/ide.h>

#include <asm/vr4181a/vr4181a_ecu.h>

#define SIZE_16MB	(16 << 20)
#define SIZE_32MB	(32 << 20)

static unsigned long io_slot0_offset;
static unsigned long io_slot1_offset;

static unsigned long memory_slot0_sys_start;
static unsigned long memory_slot0_map_size;

static unsigned long memory_slot1_sys_start;
static unsigned long memory_slot1_map_size;

void
vr4181a_ecu_set_window(struct vr4181a_address_window *win)
{
	unsigned long memory_base, io_base, size;

	memory_base = win->base_address;
	size = win->size;

	memory_slot0_sys_start = memory_base;
	if (size > SIZE_32MB)
		memory_slot0_map_size = SIZE_32MB;
	else
		memory_slot0_map_size = size;

	if (size < SIZE_16MB) {
		memory_slot1_sys_start = 0;
		memory_slot1_map_size = 0;
	} else {
		memory_slot1_sys_start = memory_base + SIZE_16MB;
		if (size > SIZE_32MB)
			memory_slot1_map_size = SIZE_16MB;
		else
			memory_slot1_map_size = size - SIZE_16MB;
	}

	size >>= 1;		/* size = size / 2 */
	io_base = memory_base + size;

	io_slot0_offset = io_base;

	if (size < SIZE_16MB)
		io_slot1_offset = 0xffffffff;
	else
		io_slot1_offset = io_base + SIZE_16MB;
}

unsigned long
vr4181a_ecu_get_io_offset(int slot)
{
	switch (slot) {
	case 0:
		return io_slot0_offset;
	case 1:
		return io_slot1_offset;
	}

	return 0;
}

unsigned long
vr4181a_ecu_get_map_size(int slot)
{
	switch (slot) {
	case 0:
		return memory_slot0_map_size;
	case 1:
		return memory_slot1_map_size;
	}

	return 0;
}

unsigned long
vr4181a_ecu_get_sys_start(int slot)
{
	switch (slot) {
	case 0:
		return memory_slot0_sys_start;
	case 1:
		return memory_slot1_sys_start;
	}

	return 0;
}

static int ecu_mode = ECU_MODE_UNKNOWN;

int
vr4181a_ecu_get_mode(void)
{
	return ecu_mode;
}

void
vr4181a_ecu_set_mode(int mode)
{
	switch (mode) {
	case ECU_MODE_CF:
		vr4181a_writeb(CF_IREQ | CF_RESET | CF_CS_EN, ECUIDE0);
		ecu_mode = ECU_MODE_CF;
		break;
	case ECU_MODE_IDE:
		vr4181a_writeb(IDE_IREQ | IDE_RESET | IDE_CS_EN, ECUIDE0);
		ecu_mode = ECU_MODE_IDE;
		break;
	default:
		ecu_mode = ECU_MODE_UNKNOWN;
		break;
	}
}

/*============================================================================*/

static int
ecu_ide_default_irq(ide_ioreg_t base)
{
	if (base == 0x100 && ecu_mode == ECU_MODE_IDE)
		return GPIO_IRQ(CONFIG_IREQ0_GPIO_NO);

	return 0;
}

static ide_ioreg_t
ecu_ide_default_io_base(int index)
{
	if (index == 0 && ecu_mode == ECU_MODE_IDE)
		return 0x100;

	return 0;
}

static void
ecu_ide_init_hwif_ports(hw_regs_t * hw, ide_ioreg_t data_port,
			ide_ioreg_t ctrl_port, int *irq)
{
	ide_ioreg_t port = data_port;
	int i;

	if (data_port == 0x100 && ecu_mode == ECU_MODE_IDE) {
		for (i = IDE_DATA_OFFSET; i <= IDE_STATUS_OFFSET; i++) {
			hw->io_ports[i] = port;
			port += 2;
		}

		ctrl_port = hw->io_ports[IDE_DATA_OFFSET] + 0x010;
		hw->io_ports[IDE_CONTROL_OFFSET] = ctrl_port;

		exca_writew(0, IOADSLBREG(0), data_port);
		exca_writew(0, IOSLBREG(0), (data_port + 0xf));
		exca_writew(0, IOADSLBREG(1), ctrl_port);
		exca_writew(0, IOSLBREG(1), (ctrl_port + 0xf));
		exca_writeb(0, IOCTRL_REG, IO_WINDOW_DATA_WIDTH_16BIT(0));
		exca_writeb(0, ADWINENREG,
			    IO_WINDOW_ENABLE(0) | IO_WINDOW_ENABLE(1));
	} else {
		for (i = IDE_DATA_OFFSET; i <= IDE_STATUS_OFFSET; i++) {
			hw->io_ports[i] = port;
			port += 1;
		}

		if (ctrl_port)
			hw->io_ports[IDE_CONTROL_OFFSET] = ctrl_port;
		else
			hw->io_ports[IDE_CONTROL_OFFSET] =
			    hw->io_ports[IDE_DATA_OFFSET] + 0x206;
	}

	if (irq != NULL)
		*irq = 0;
	hw->io_ports[IDE_IRQ_OFFSET] = 0;
}

static int
ecu_ide_request_irq(unsigned int irq,
		    void (*handler) (int, void *, struct pt_regs *),
		    unsigned long flags, const char *device, void *dev_id)
{
	return request_irq(irq, handler, flags, device, dev_id);
}

static void
ecu_ide_free_irq(unsigned int irq, void *dev_id)
{
	free_irq(irq, dev_id);
}

static int
ecu_ide_check_region(ide_ioreg_t from, unsigned int extent)
{
	return check_region(from, extent);
}

static void
ecu_ide_request_region(ide_ioreg_t from, unsigned int extent, const char *name)
{
	request_region(from, extent, name);
}

static void
ecu_ide_release_region(ide_ioreg_t from, unsigned int extent)
{
	release_region(from, extent);
}

static struct ide_ops ecu_ide_ops = {
	.ide_default_irq = &ecu_ide_default_irq,
	.ide_default_io_base = &ecu_ide_default_io_base,
	.ide_init_hwif_ports = &ecu_ide_init_hwif_ports,
	.ide_request_irq = &ecu_ide_request_irq,
	.ide_free_irq = &ecu_ide_free_irq,
	.ide_check_region = &ecu_ide_check_region,
	.ide_request_region = &ecu_ide_request_region,
	.ide_release_region = &ecu_ide_release_region,
};

/*============================================================================*/

void
vr4181a_ecu_init(int mode)
{
	vr4181a_ecu_set_mode(mode);

	if (mode == ECU_MODE_CF) {
		vr4181a_set_gpio_mode(24, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(25, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(26, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(27, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(28, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(29, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(30, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(31, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(32, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(33, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(34, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(35, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
		vr4181a_set_gpio_mode(36, GPIO_OTHER, GPIO_INPUT_PROHIBIT);
	}

	/*
	 * Known problem about NEC VR4181A ECU:
	 * The interrupt from IREQ may not be generated.
	 * We can not use ECU interrupt.
	 */
	vr4181a_writeb(0, ECUINTMSK0);
	vr4181a_writeb(0, ECUINTMSK1);

	/*
	 * set IREQ of ECU Slot0.
	 */
	vr4181a_set_irq_trigger(CONFIG_IREQ0_GPIO_NO, TRIGGER_LEVEL);
	if (mode == ECU_MODE_IDE)
		vr4181a_set_irq_level(CONFIG_IREQ0_GPIO_NO, LEVEL_HIGH);
	else
		vr4181a_set_irq_level(CONFIG_IREQ0_GPIO_NO, LEVEL_LOW);

	/*
	 * set IREQ of ECU Slot1.
	 */
	if (vr4181a_get_pinmode0() & PINMODE0_ECU1) {
		vr4181a_set_irq_trigger(CONFIG_IREQ1_GPIO_NO, TRIGGER_LEVEL);
		vr4181a_set_irq_level(CONFIG_IREQ1_GPIO_NO, LEVEL_LOW);
	}

	if (mode == ECU_MODE_IDE) {
		exca_writeb(0, PWRRSETDRV, POWER_ENABLE);
		exca_writeb(0, PWRRSETDRV, POWER_ENABLE | POWER_OUTPUT_ENABLE);
		exca_writeb(0, ITGENCTREG, CARD_RESET_INACTIVE);
	}
#ifdef CONFIG_BLK_DEV_IDE
	ide_ops = &ecu_ide_ops;
#endif
}
